const ERROR_MESSAGES = {
    'saveRecording not available': {
        title: '保存功能不可用',
        message: '录制保存功能当前不可用，请稍后重试。',
        suggestion: '如果问题持续存在，请重启应用程序。'
    },
    'saveRecording failed': {
        title: '保存失败',
        message: '无法保存录制文件。',
        suggestion: '请检查磁盘空间和文件权限。'
    },
    'recording not started': {
        title: '未开始录制',
        message: '请先开始录制。',
        suggestion: '点击录制按钮开始录制。'
    },
    'no display selected': {
        title: '未选择显示器',
        message: '请选择要录制的显示器。',
        suggestion: '在设置中选择一个显示器。'
    },
    'audio device not found': {
        title: '音频设备未找到',
        message: '无法访问音频设备。',
        suggestion: '请检查系统音频设置。'
    },
    'permission denied': {
        title: '权限被拒绝',
        message: '应用程序缺少必要的权限。',
        suggestion: '请在系统设置中授予应用程序必要的权限。'
    },
    'file not found': {
        title: '文件未找到',
        message: '无法找到指定的文件。',
        suggestion: '请检查文件路径是否正确。'
    },
    'invalid file format': {
        title: '无效的文件格式',
        message: '文件格式不受支持。',
        suggestion: '请使用支持的文件格式（MP4、GIF、PNG等）。'
    },
    'disk full': {
        title: '磁盘空间不足',
        message: '磁盘空间不足以完成操作。',
        suggestion: '请清理磁盘空间后重试。'
    },
    'network error': {
        title: '网络错误',
        message: '无法连接到服务器。',
        suggestion: '请检查网络连接后重试。'
    },
    'timeout': {
        title: '操作超时',
        message: '操作耗时过长，已超时。',
        suggestion: '请重试或联系技术支持。'
    },
    'default': {
        title: '发生错误',
        message: '操作失败。',
        suggestion: '请重试或联系技术支持。'
    }
};
/**
 * 用户友好的错误处理器
 */
export class UserFriendlyErrorHandler {
    toastManager;
    errorHistory;
    maxHistory;
    constructor(toastManager) {
        this.toastManager = toastManager || null;
        this.errorHistory = [];
        this.maxHistory = 50;
    }
    /**
     * 处理错误并显示用户友好的消息
     * @param {Error|string} error - 错误对象或错误消息
     * @param {string} context - 错误上下文
     * @param {Object} options - 选项
     */
    handle(error, context = '', options = {}) {
        const { showToUser = true, logToConsole = true, type = 'error' } = options;
        const errorMessage = this.extractErrorMessage(error);
        const friendlyError = this.getFriendlyError(errorMessage, context);
        this.addToHistory({
            originalError: errorMessage,
            friendlyError,
            context,
            timestamp: new Date().toISOString()
        });
        if (logToConsole) {
            console.error(`[${context}]`, error);
        }
        if (showToUser && this.toastManager) {
            this.showUserNotification(friendlyError, type);
        }
        return friendlyError;
    }
    /**
     * 提取错误消息
     * @param {Error|string} error - 错误对象或错误消息
     * @returns {string} 错误消息
     */
    extractErrorMessage(error) {
        if (typeof error === 'string') {
            return error;
        }
        const e = error;
        if (e && e.message) {
            return String(e.message);
        }
        if (e && typeof e.toString === 'function') {
            return String(e.toString());
        }
        return 'Unknown error';
    }
    /**
     * 获取用户友好的错误信息
     * @param {string} errorMessage - 原始错误消息
     * @param {string} context - 错误上下文
     * @returns {Object} 用户友好的错误信息
     */
    getFriendlyError(errorMessage, context) {
        const lowerError = errorMessage.toLowerCase();
        for (const [key, value] of Object.entries(ERROR_MESSAGES)) {
            if (lowerError.includes(key.toLowerCase())) {
                return {
                    ...value,
                    originalMessage: errorMessage,
                    context
                };
            }
        }
        return {
            ...ERROR_MESSAGES.default,
            originalMessage: errorMessage,
            context,
            message: errorMessage
        };
    }
    /**
     * 显示用户通知
     * @param {Object} friendlyError - 用户友好的错误信息
     * @param {string} type - 通知类型
     */
    showUserNotification(friendlyError, type) {
        const { title, message, suggestion } = friendlyError;
        const fullMessage = suggestion
            ? `${message}\n${suggestion}`
            : message;
        const fn = this.toastManager ? this.toastManager[type] : null;
        if (!fn)
            return;
        fn(fullMessage, {
            duration: 5000
        });
    }
    /**
     * 添加到错误历史
     * @param {Object} errorInfo - 错误信息
     */
    addToHistory(errorInfo) {
        this.errorHistory.push(errorInfo);
        if (this.errorHistory.length > this.maxHistory) {
            this.errorHistory.shift();
        }
    }
    /**
     * 获取错误历史
     * @returns {Array} 错误历史数组
     */
    getErrorHistory() {
        return [...this.errorHistory];
    }
    /**
     * 清除错误历史
     */
    clearHistory() {
        this.errorHistory = [];
    }
    /**
     * 添加自定义错误消息映射
     * @param {string} key - 错误键
     * @param {Object} value - 错误信息
     */
    addErrorMessage(key, value) {
        ERROR_MESSAGES[key] = value;
    }
}
/**
 * 创建用户友好的错误处理器实例
 * @param {Object} toastManager - Toast 管理器实例
 * @returns {UserFriendlyErrorHandler} 用户友好的错误处理器实例
 */
export function createUserFriendlyErrorHandler(toastManager) {
    return new UserFriendlyErrorHandler(toastManager);
}
export default createUserFriendlyErrorHandler;
//# sourceMappingURL=userFriendlyErrorHandler.js.map