export class EventManager {
    listeners;
    constructor() {
        this.listeners = [];
    }
    /**
     * 添加事件监听器
     * @param {EventTarget} target - 事件目标
     * @param {string} event - 事件名称
     * @param {Function} handler - 事件处理函数
     * @param {Object|boolean} [options] - 事件选项
     */
    add(target, event, handler, options) {
        if (!target || !event || typeof handler !== 'function')
            return;
        target.addEventListener(event, handler, options);
        this.listeners.push({ target, event, handler, options });
    }
    /**
     * 移除所有事件监听器
     */
    removeAll() {
        for (const { target, event, handler, options } of this.listeners) {
            try {
                target.removeEventListener(event, handler, options);
            }
            catch (error) {
                console.error('[EventManager] Failed to remove listener:', error);
            }
        }
        this.listeners = [];
    }
    /**
     * 获取监听器数量
     * @returns {number} 监听器数量
     */
    getListenerCount() {
        return this.listeners.length;
    }
    destroy() {
        this.removeAll();
    }
}
/**
 * 创建事件管理器实例
 * @returns {EventManager} 事件管理器实例
 */
export function createEventManager() {
    return new EventManager();
}
export default createEventManager();
//# sourceMappingURL=eventManager.js.map