export class ErrorHandler {
    logErrors;
    throwErrors;
    onError;
    /**
     * 创建错误处理器实例
     * @param {Object} options - 配置选项
     * @param {boolean} options.logErrors - 是否记录错误日志，默认为 true
     * @param {boolean} options.throwErrors - 是否抛出错误，默认为 false
     * @param {Function} options.onError - 自定义错误处理回调函数
     */
    constructor(options = {}) {
        this.logErrors = options.logErrors !== false;
        this.throwErrors = options.throwErrors || false;
        this.onError = options.onError || null;
    }
    /**
     * 处理错误
     * @param {Error} error - 错误对象
     * @param {string} context - 错误发生的上下文信息
     */
    handle(error, context = '') {
        const e = error;
        const errorInfo = {
            error,
            context,
            timestamp: new Date().toISOString(),
            stack: e && e.stack ? String(e.stack) : undefined
        };
        if (this.logErrors) {
            console.error(`[ErrorHandler]${context ? ` [${context}]` : ''}:`, error);
        }
        if (this.onError && typeof this.onError === 'function') {
            try {
                this.onError(errorInfo);
            }
            catch (handlerError) {
                console.error('[ErrorHandler] Error in error handler:', handlerError);
            }
        }
        if (this.throwErrors) {
            throw error;
        }
    }
    /**
     * 包装同步函数以自动处理错误
     * @param {Function} fn - 要包装的函数
     * @param {string} context - 错误上下文
     * @returns {Function} 包装后的函数
     */
    wrap(fn, context = '') {
        return (...args) => {
            try {
                return fn(...args);
            }
            catch (error) {
                this.handle(error, context);
            }
        };
    }
    /**
     * 包装异步函数以自动处理错误
     * @param {Function} fn - 要包装的异步函数
     * @param {string} context - 错误上下文
     * @returns {Function} 包装后的异步函数
     */
    wrapAsync(fn, context = '') {
        return async (...args) => {
            try {
                return await fn(...args);
            }
            catch (error) {
                this.handle(error, context);
                throw error;
            }
        };
    }
    /**
     * 创建错误处理器实例的静态方法
     * @param {Object} options - 配置选项
     * @returns {ErrorHandler} 错误处理器实例
     */
    static create(options = {}) {
        return new ErrorHandler(options);
    }
}
/**
 * 默认错误处理器实例
 * 配置为记录错误但不抛出错误
 */
export const errorHandler = ErrorHandler.create({
    logErrors: true,
    throwErrors: false
});
/**
 * 处理错误的便捷函数
 * @param {Error} error - 错误对象
 * @param {string} context - 错误发生的上下文
 */
export function handleError(error, context = '') {
    errorHandler.handle(error, context);
}
/**
 * 包装同步函数以自动处理错误的便捷函数
 * @param {Function} fn - 要包装的函数
 * @param {string} context - 错误上下文
 * @returns {Function} 包装后的函数
 */
export function wrapError(fn, context = '') {
    return errorHandler.wrap(fn, context);
}
/**
 * 包装异步函数以自动处理错误的便捷函数
 * @param {Function} fn - 要包装的异步函数
 * @param {string} context - 错误上下文
 * @returns {Function} 包装后的异步函数
 */
export function wrapAsyncError(fn, context = '') {
    return errorHandler.wrapAsync(fn, context);
}
//# sourceMappingURL=errorHandler.js.map