const els = {
    winClose: document.getElementById('winClose'),
    fadeTime: document.getElementById('fadeTime'),
    bubbleBg: document.getElementById('bubbleBg'),
    bubbleBgAlpha: document.getElementById('bubbleBgAlpha'),
    bubbleText: document.getElementById('bubbleText'),
    bubbleRadius: document.getElementById('bubbleRadius'),
    bubbleFontSize: document.getElementById('bubbleFontSize'),
    bubbleVertical: document.getElementById('bubbleVertical'),
    bubbleOffsetX: document.getElementById('bubbleOffsetX'),
    mouseDotBg: document.getElementById('mouseDotBg'),
    mouseDotBgAlpha: document.getElementById('mouseDotBgAlpha'),
    mouseDotGlow: document.getElementById('mouseDotGlow'),
    mouseDotGlowAlpha: document.getElementById('mouseDotGlowAlpha'),
    mouseDotSize: document.getElementById('mouseDotSize'),
    mouseDotRing: document.getElementById('mouseDotRing'),
    mouseTrailCount: document.getElementById('mouseTrailCount'),
    mouseTrailLag: document.getElementById('mouseTrailLag'),
    toolbarBg: document.getElementById('toolbarBg'),
    toolbarBgAlpha: document.getElementById('toolbarBgAlpha'),
    toolbarRadius: document.getElementById('toolbarRadius'),
    toolbarButtonFg: document.getElementById('toolbarButtonFg'),
    toolbarButtonActiveBg: document.getElementById('toolbarButtonActiveBg'),
    toolbarButtonActiveBgAlpha: document.getElementById('toolbarButtonActiveBgAlpha'),
    toolbarButtonActiveRadius: document.getElementById('toolbarButtonActiveRadius'),
    drawLineWidth: document.getElementById('drawLineWidth'),
    drawColor: document.getElementById('drawColor'),
    eraserSize: document.getElementById('eraserSize'),
    shotTextSize: document.getElementById('shotTextSize'),
    toolbarVisible: document.getElementById('toolbarVisible'),
    overlayVisible: document.getElementById('overlayVisible'),
    mouseButtonVisible: document.getElementById('mouseButtonVisible'),
    mouseDotMoveVisible: document.getElementById('mouseDotMoveVisible'),
    mouseDotClickVisible: document.getElementById('mouseDotClickVisible'),
    mouseDotStyle: document.getElementById('mouseDotStyle'),
    autoLaunch: document.getElementById('autoLaunch'),
    uiTheme: document.getElementById('uiTheme'),
    recordSystemAudio: document.getElementById('recordSystemAudio'),
    recordMicAudio: document.getElementById('recordMicAudio'),
    recordMicDeviceId: document.getElementById('recordMicDeviceId'),
    hotkeyScreenshotSave: document.getElementById('hotkeyScreenshotSave'),
    hotkeyScreenshotClipboard: document.getElementById('hotkeyScreenshotClipboard'),
    hotkeyToolRefimg: document.getElementById('hotkeyToolRefimg'),
    hotkeyRecordToggle: document.getElementById('hotkeyRecordToggle'),
    hotkeyRecordSaveGif: document.getElementById('hotkeyRecordSaveGif'),
    hotkeyRecordSaveMp4: document.getElementById('hotkeyRecordSaveMp4'),
    hotkeyToggleOverlay: document.getElementById('hotkeyToggleOverlay'),
    hotkeyToggleMouseButtons: document.getElementById('hotkeyToggleMouseButtons'),
    hotkeyToggleMouseDot: document.getElementById('hotkeyToggleMouseDot'),
    hotkeyOpenSettings: document.getElementById('hotkeyOpenSettings'),
    hotkeyToolPen: document.getElementById('hotkeyToolPen'),
    hotkeyToolEraser: document.getElementById('hotkeyToolEraser'),
    hotkeyToolArrow: document.getElementById('hotkeyToolArrow'),
    hotkeyToolRect: document.getElementById('hotkeyToolRect'),
    hotkeyToolCircle: document.getElementById('hotkeyToolCircle'),
    hotkeyToolText: document.getElementById('hotkeyToolText'),
    hotkeyToolMosaic: document.getElementById('hotkeyToolMosaic'),
    hotkeyToolRecord: document.getElementById('hotkeyToolRecord'),
    hotkeyUndo: document.getElementById('hotkeyUndo'),
    hotkeyClear: document.getElementById('hotkeyClear'),
    resetKey: document.getElementById('resetKey'),
    resetMouse: document.getElementById('resetMouse'),
    resetToolbar: document.getElementById('resetToolbar'),
    resetRecord: document.getElementById('resetRecord'),
    resetHotkeys: document.getElementById('resetHotkeys'),
    resetAll: document.getElementById('resetAll'),
};
const titlebarIcon = document.querySelector('.titlebar-icon');
if (titlebarIcon) {
    titlebarIcon.addEventListener('error', () => {
        const fallback = 'assets/icons/active.png';
        if (String(titlebarIcon.getAttribute('src') || '') !== fallback)
            titlebarIcon.setAttribute('src', fallback);
    }, { once: true });
}
const FIELD_SPECS = [
    { id: 'bubbleBg', key: '--bubble-bg', kind: 'color-alpha', alphaId: 'bubbleBgAlpha' },
    { id: 'bubbleText', key: '--bubble-text', kind: 'color' },
    { id: 'bubbleRadius', key: '--bubble-radius', kind: 'px' },
    { id: 'bubbleFontSize', key: '--bubble-font-size', kind: 'px' },
    { id: 'bubbleVertical', key: '--bubble-vertical', kind: 'px' },
    { id: 'bubbleOffsetX', key: '--bubble-offset-x', kind: 'px' },
    { id: 'mouseDotBg', key: '--mouse-dot-bg', kind: 'color-alpha', alphaId: 'mouseDotBgAlpha' },
    { id: 'mouseDotGlow', key: '--mouse-dot-glow', kind: 'color-alpha', alphaId: 'mouseDotGlowAlpha' },
    { id: 'mouseDotSize', key: '--mouse-dot-size', kind: 'px' },
    { id: 'mouseDotRing', key: '--mouse-dot-ring', kind: 'px' },
    { id: 'mouseTrailCount', key: '--mouse-trail-count', kind: 'raw' },
    { id: 'mouseTrailLag', key: '--mouse-trail-lag', kind: 'raw' },
    { id: 'toolbarBg', key: '--toolbar-bg', kind: 'color-alpha', alphaId: 'toolbarBgAlpha' },
    { id: 'toolbarRadius', key: '--toolbar-radius', kind: 'px' },
    { id: 'toolbarButtonFg', key: '--toolbar-button-fg', kind: 'color' },
    { id: 'toolbarButtonActiveBg', key: '--toolbar-button-active-bg', kind: 'color-alpha', alphaId: 'toolbarButtonActiveBgAlpha' },
    { id: 'toolbarButtonActiveRadius', key: '--toolbar-button-active-radius', kind: 'px' },
    { id: 'drawLineWidth', key: '--draw-line-width', kind: 'px' },
    { id: 'drawColor', key: '--draw-stroke-color', kind: 'color' },
    { id: 'eraserSize', key: '--draw-eraser-width', kind: 'px' },
    { id: 'shotTextSize', key: '--shot-text-font-size', kind: 'px' },
];
const KEY_GROUP = [
    'overlayVisible',
    'mouseButtonVisible',
    '--bubble-bg',
    '--bubble-text',
    '--bubble-radius',
    '--bubble-font-size',
    '--bubble-vertical',
    '--bubble-offset-x',
];
const MOUSE_GROUP = [
    'mouseButtonVisible',
    'mouseDotMoveVisible',
    'mouseDotClickVisible',
    '--mouse-dot-bg',
    '--mouse-dot-glow',
    '--mouse-dot-size',
    '--mouse-dot-ring',
    '--mouse-trail-count',
    '--mouse-trail-lag',
];
const TOOLBAR_GROUP = [
    'toolbarVisible',
    '--toolbar-bg',
    '--toolbar-radius',
    '--toolbar-button-fg',
    '--toolbar-button-active-bg',
    '--toolbar-button-active-radius',
    '--draw-line-width',
    '--draw-stroke-color',
    '--draw-eraser-width',
    '--shot-text-font-size',
];
const RECORD_GROUP = [
    'recordSystemAudio',
    'recordMicAudio',
    'recordMicDeviceId',
];
const swatches = new Map();
let defaultCssVars = {};
let suppressHydrate = false;
function normalizeUiTheme(v) {
    const s = String(v || '').trim().toLowerCase();
    return s === 'dark' ? 'dark' : 'light';
}
function applyUiTheme(theme) {
    const t = normalizeUiTheme(theme);
    try {
        document.documentElement.dataset.theme = t;
    }
    catch { }
    defaultCssVars = readDefaultCssVars();
}
function setupSidebarNav() {
    const navButtons = Array.from(document.querySelectorAll('button.nav-btn[data-target]'));
    const panels = Array.from(document.querySelectorAll('.panel[id]'));
    if (!navButtons.length || !panels.length)
        return;
    const activate = (targetId) => {
        for (const btn of navButtons)
            btn.classList.toggle('active', btn.dataset.target === targetId);
        for (const p of panels)
            p.classList.toggle('active', p.id === targetId);
    };
    for (const btn of navButtons) {
        btn.addEventListener('click', () => activate(btn.dataset.target));
    }
    const initial = navButtons.find(b => b.classList.contains('active')) || navButtons[0];
    if (initial && initial.dataset && initial.dataset.target)
        activate(initial.dataset.target);
}
function findSpecById(id) {
    for (const spec of FIELD_SPECS) {
        if (spec.id === id)
            return spec;
        if (spec.alphaId === id)
            return spec;
    }
    return null;
}
function setDefaultModePair(spec, isDefault) {
    const baseEl = spec && els[spec.id];
    if (baseEl)
        setDefaultMode(baseEl, isDefault);
    if (spec && spec.kind === 'color-alpha' && spec.alphaId) {
        const alphaEl = els[spec.alphaId];
        if (alphaEl)
            setDefaultMode(alphaEl, isDefault);
    }
}
function isDefaultModePair(spec) {
    const baseEl = spec && els[spec.id];
    if (!baseEl)
        return true;
    if (spec && spec.kind === 'color-alpha' && spec.alphaId) {
        const alphaEl = els[spec.alphaId];
        return getDefaultMode(baseEl) && (!alphaEl || getDefaultMode(alphaEl));
    }
    return getDefaultMode(baseEl);
}
function toNum(v) {
    const n = Number(v);
    return Number.isFinite(n) ? n : null;
}
function setDefaultMode(el, isDefault) {
    if (!el)
        return;
    try {
        el.dataset.defaultMode = isDefault ? '1' : '0';
        el.classList.toggle('is-default', !!isDefault);
    }
    catch { }
}
function getDefaultMode(el) {
    try {
        return el && el.dataset && el.dataset.defaultMode === '1';
    }
    catch {
        return false;
    }
}
function readDefaultCssVars() {
    const out = {};
    try {
        const cs = getComputedStyle(document.documentElement);
        for (const spec of FIELD_SPECS) {
            out[spec.key] = String(cs.getPropertyValue(spec.key) || '').trim();
        }
    }
    catch { }
    return out;
}
function ensureSwatch(inputEl) {
    if (!inputEl || !inputEl.id)
        return null;
    if (swatches.has(inputEl.id))
        return swatches.get(inputEl.id);
    const wrap = inputEl.parentElement;
    if (!wrap)
        return null;
    const btn = document.createElement('button');
    btn.type = 'button';
    btn.className = 'swatch';
    btn.title = '颜色';
    btn.style.background = String(inputEl.value || '').trim() || 'transparent';
    wrap.insertBefore(btn, inputEl);
    swatches.set(inputEl.id, btn);
    return btn;
}
function updateSwatchFromInput(inputEl) {
    if (!inputEl || !inputEl.id)
        return;
    const sw = swatches.get(inputEl.id);
    if (!sw)
        return;
    const spec = findSpecById(inputEl.id);
    if (spec && spec.kind === 'color-alpha') {
        const v = colorWithAlphaFromFields(spec.id, spec.alphaId);
        sw.style.background = v ? v : 'transparent';
    }
    else {
        const v = String(inputEl.value || '').trim();
        sw.style.background = v ? v : 'transparent';
    }
}
function clamp01(x) {
    const n = Number(x);
    if (!Number.isFinite(n))
        return null;
    if (n <= 0)
        return 0;
    if (n >= 1)
        return 1;
    return n;
}
function parseCssColorWithAlpha(s) {
    const str = String(s || '').trim();
    if (!str)
        return null;
    const hex3 = /^#([0-9a-f]{3})$/i.exec(str);
    if (hex3) {
        const h = hex3[1];
        return {
            r: parseInt(h[0] + h[0], 16),
            g: parseInt(h[1] + h[1], 16),
            b: parseInt(h[2] + h[2], 16),
            a: 1,
        };
    }
    const hex6 = /^#([0-9a-f]{6})$/i.exec(str);
    if (hex6) {
        const h = hex6[1];
        return {
            r: parseInt(h.slice(0, 2), 16),
            g: parseInt(h.slice(2, 4), 16),
            b: parseInt(h.slice(4, 6), 16),
            a: 1,
        };
    }
    const rgb = /^rgba?\(\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*(\d{1,3})(?:\s*,\s*([\d.]+))?\s*\)$/i.exec(str);
    if (rgb) {
        const r = Math.max(0, Math.min(255, Number(rgb[1]) || 0));
        const g = Math.max(0, Math.min(255, Number(rgb[2]) || 0));
        const b = Math.max(0, Math.min(255, Number(rgb[3]) || 0));
        const a = typeof rgb[4] !== 'undefined' ? clamp01(rgb[4]) : 1;
        return { r, g, b, a: a === null ? 1 : a };
    }
    return null;
}
function rgbCss(c) {
    if (!c)
        return '';
    return `rgb(${c.r}, ${c.g}, ${c.b})`;
}
function colorWithAlphaFromFields(id, alphaId) {
    const baseEl = els[id];
    const alphaEl = alphaId ? els[alphaId] : null;
    if (!baseEl)
        return null;
    const baseStr = String(baseEl.value || '').trim();
    if (!baseStr)
        return null;
    const parsed = parseCssColorWithAlpha(baseStr);
    if (!parsed)
        return baseStr;
    let a = parsed.a;
    if (alphaEl) {
        const raw = String(alphaEl.value || '').trim();
        if (raw) {
            const p = Number(raw);
            if (Number.isFinite(p)) {
                const frac = clamp01(p / 100);
                if (frac !== null)
                    a = frac;
            }
        }
    }
    if (a >= 1)
        return `rgb(${parsed.r}, ${parsed.g}, ${parsed.b})`;
    const strA = a === 0 ? '0' : String(Math.round(a * 100) / 100);
    return `rgba(${parsed.r}, ${parsed.g}, ${parsed.b}, ${strA})`;
}
function ensureCssVarsObj(settings) {
    if (!settings || typeof settings !== 'object')
        return { fadeTime: 1000, cssVars: {} };
    const cssVars = settings.cssVars && typeof settings.cssVars === 'object' ? settings.cssVars : {};
    const fadeTime = typeof settings.fadeTime === 'number' && Number.isFinite(settings.fadeTime) ? settings.fadeTime : 1000;
    return { ...settings, fadeTime, cssVars };
}
let cachedMicDevices = [];
let micDevicesSeq = 0;
function normalizeMicDeviceId(id) {
    const s = String(id || '').trim();
    if (!s)
        return null;
    if (s === 'default' || s === 'communications')
        return null;
    return s;
}
function safeMicLabel(dev, index) {
    const label = String(dev && dev.label ? dev.label : '').trim();
    if (label)
        return label;
    return `麦克风 ${index + 1}`;
}
function setMicDeviceOptions(selectedId, forceDefault = false) {
    const sel = els.recordMicDeviceId;
    if (!sel)
        return;
    const nextSelected = String(selectedId || '').trim();
    const prev = String(sel.value || '').trim();
    const keep = forceDefault ? nextSelected : (nextSelected || prev);
    try {
        sel.innerHTML = '';
    }
    catch { }
    const optDefault = document.createElement('option');
    optDefault.value = '';
    optDefault.textContent = '默认麦克风';
    sel.appendChild(optDefault);
    const devs = Array.isArray(cachedMicDevices) ? cachedMicDevices : [];
    for (let i = 0; i < devs.length; i++) {
        const d = devs[i];
        const id = String(d && d.deviceId ? d.deviceId : '').trim();
        if (!id)
            continue;
        const opt = document.createElement('option');
        opt.value = id;
        opt.textContent = safeMicLabel(d, i);
        sel.appendChild(opt);
    }
    const hasKeep = keep && Array.from((sel.options || [])).some((o) => String(o && o.value ? o.value : '') === keep);
    try {
        sel.value = hasKeep ? keep : '';
    }
    catch { }
}
function setMicDeviceEnabled(enabled) {
    if (!els.recordMicDeviceId)
        return;
    try {
        els.recordMicDeviceId.disabled = !enabled;
    }
    catch { }
}
async function refreshMicDevices() {
    const seq = ++micDevicesSeq;
    if (!navigator.mediaDevices || typeof navigator.mediaDevices.enumerateDevices !== 'function') {
        cachedMicDevices = [];
        return;
    }
    let list = [];
    try {
        list = await navigator.mediaDevices.enumerateDevices();
    }
    catch {
        list = [];
    }
    if (seq !== micDevicesSeq)
        return;
    const mics = (Array.isArray(list) ? list : [])
        .filter(d => d && d.kind === 'audioinput')
        .filter(d => {
        const id = String(d && d.deviceId ? d.deviceId : '').trim();
        return !!id && id !== 'default' && id !== 'communications';
    });
    const seen = new Set();
    cachedMicDevices = mics.filter(d => {
        const id = String(d && d.deviceId ? d.deviceId : '').trim();
        if (!id || seen.has(id))
            return false;
        seen.add(id);
        return true;
    });
}
function formatAcceleratorLabel(acc) {
    const s = String(acc || '').trim();
    if (!s)
        return '';
    return s.replace(/\s*\+\s*/g, ' + ');
}
let hotkeyCaptureEl = null;
let hotkeyCapturePrevValue = '';
function isModifierOnlyKey(key) {
    const k = String(key || '');
    return k === 'Shift' || k === 'Control' || k === 'Alt' || k === 'Meta' || k === 'AltGraph';
}
function normalizeHotkeyKey(key) {
    const k = String(key || '');
    if (!k)
        return '';
    if (k === ' ')
        return 'Space';
    if (k === 'Spacebar')
        return 'Space';
    if (k === 'ArrowUp')
        return 'Up';
    if (k === 'ArrowDown')
        return 'Down';
    if (k === 'ArrowLeft')
        return 'Left';
    if (k === 'ArrowRight')
        return 'Right';
    if (k === 'Esc')
        return 'Escape';
    if (k.length === 1)
        return k.toUpperCase();
    return k[0].toUpperCase() + k.slice(1);
}
function buildAcceleratorFromKeyboardEvent(e) {
    if (!e)
        return '';
    const mods = [];
    if (e.ctrlKey)
        mods.push('Ctrl');
    if (e.altKey)
        mods.push('Alt');
    if (e.shiftKey)
        mods.push('Shift');
    if (e.metaKey)
        mods.push('Super');
    const mainKey = normalizeHotkeyKey(e.key);
    if (!mainKey || isModifierOnlyKey(mainKey))
        return '';
    return [...mods, mainKey].join('+');
}
function normalizeAcceleratorForCompare(raw) {
    const s = String(raw || '').trim();
    if (!s)
        return '';
    const cleaned = s.replace(/\s+/g, '');
    const parts = cleaned.split('+').map(x => String(x || '').trim()).filter(Boolean);
    if (!parts.length)
        return '';
    const normalized = parts.map((p) => {
        const low = p.toLowerCase();
        if (low === 'control')
            return 'Ctrl';
        if (low === 'cmdorctrl')
            return 'CommandOrControl';
        if (low === 'commandorcontrol')
            return 'CommandOrControl';
        if (low === 'cmd')
            return 'Command';
        if (low === 'ctrl')
            return 'Ctrl';
        if (low === 'alt' || low === 'option')
            return 'Alt';
        if (low === 'shift')
            return 'Shift';
        if (low === 'win' || low === 'meta' || low === 'super')
            return 'Super';
        if (p.length === 1)
            return p.toUpperCase();
        return p[0].toUpperCase() + p.slice(1);
    });
    return normalized.join('+');
}
function getHotkeyFieldLabelText(inputEl) {
    if (!inputEl || !inputEl.id)
        return '';
    const lb = document.querySelector(`label[for="${CSS.escape(inputEl.id)}"]`);
    return String(lb && lb.textContent ? lb.textContent : '').trim();
}
function flashHotkeyConflict(el, message) {
    if (!el)
        return;
    try {
        el.classList.add('hotkey-conflict');
    }
    catch { }
    try {
        el.title = String(message || '').trim();
    }
    catch { }
    setTimeout(() => { try {
        el.classList.remove('hotkey-conflict');
    }
    catch { } }, 1200);
}
function stopHotkeyCapture(restore = false) {
    if (!hotkeyCaptureEl)
        return;
    const el = hotkeyCaptureEl;
    hotkeyCaptureEl = null;
    try {
        el.classList.remove('hotkey-capturing');
    }
    catch { }
    if (restore) {
        try {
            el.value = hotkeyCapturePrevValue;
        }
        catch { }
    }
    hotkeyCapturePrevValue = '';
}
function startHotkeyCapture(el) {
    if (!el)
        return;
    if (hotkeyCaptureEl && hotkeyCaptureEl !== el)
        stopHotkeyCapture(true);
    if (hotkeyCaptureEl === el)
        return;
    hotkeyCaptureEl = el;
    hotkeyCapturePrevValue = String(el.value || '');
    try {
        el.classList.add('hotkey-capturing');
    }
    catch { }
    try {
        el.value = '按下快捷键';
    }
    catch { }
}
function setupHotkeyCaptureUI() {
    const inputs = Array.from(document.querySelectorAll('input[data-hotkey]'));
    for (const input of inputs) {
        try {
            input.setAttribute('readonly', 'readonly');
        }
        catch { }
        input.addEventListener('mousedown', (e) => {
            try {
                e.preventDefault();
            }
            catch { }
            try {
                input.focus();
            }
            catch { }
            startHotkeyCapture(input);
        });
        input.addEventListener('focus', () => startHotkeyCapture(input));
        input.addEventListener('blur', () => { if (hotkeyCaptureEl === input)
            stopHotkeyCapture(true); });
    }
    const clearBtns = Array.from(document.querySelectorAll('button[data-clear-hotkey]'));
    for (const btn of clearBtns) {
        btn.addEventListener('click', () => {
            const targetId = String(btn.dataset.clearHotkey || '').trim();
            const input = targetId ? document.getElementById(targetId) : null;
            if (!input)
                return;
            if (hotkeyCaptureEl === input)
                stopHotkeyCapture(false);
            try {
                input.value = '';
            }
            catch { }
            try {
                input.dispatchEvent(new Event('input', { bubbles: true }));
            }
            catch { }
            try {
                input.blur();
            }
            catch { }
        });
    }
    document.addEventListener('keydown', (e) => {
        if (!hotkeyCaptureEl)
            return;
        try {
            e.preventDefault();
        }
        catch { }
        try {
            e.stopPropagation();
        }
        catch { }
        const key = String(e.key || '');
        if (key === 'Escape' || key === 'Esc') {
            const el = hotkeyCaptureEl;
            stopHotkeyCapture(true);
            try {
                el && el.blur();
            }
            catch { }
            ;
            return;
        }
        if (isModifierOnlyKey(key))
            return;
        const acc = buildAcceleratorFromKeyboardEvent(e);
        if (!acc)
            return;
        const el = hotkeyCaptureEl;
        const nextNorm = normalizeAcceleratorForCompare(acc);
        if (nextNorm) {
            const conflict = inputs.find((other) => {
                if (!other || other === el)
                    return false;
                const curNorm = normalizeAcceleratorForCompare(other.value);
                return !!curNorm && curNorm === nextNorm;
            });
            if (conflict) {
                const conflictName = getHotkeyFieldLabelText(conflict);
                stopHotkeyCapture(true);
                flashHotkeyConflict(el, conflictName ? `快捷键重复：已用于「${conflictName}」` : '快捷键重复');
                flashHotkeyConflict(conflict, '快捷键重复');
                try {
                    el.blur();
                }
                catch { }
                return;
            }
        }
        stopHotkeyCapture(false);
        try {
            el.value = formatAcceleratorLabel(acc);
        }
        catch { }
        try {
            el.dispatchEvent(new Event('input', { bubbles: true }));
        }
        catch { }
        try {
            el.blur();
        }
        catch { }
    }, true);
}
function setCssVar(settings, key, value) {
    const next = ensureCssVarsObj(settings);
    next.cssVars = { ...next.cssVars, [key]: value };
    return next;
}
const MOUSE_STYLE_KEYS = [
    '--mouse-dot-bg',
    '--mouse-dot-glow',
    '--mouse-dot-size',
    '--mouse-dot-ring',
    '--mouse-trail-count',
    '--mouse-trail-lag',
];
function normalizeMouseDotStyle(v) {
    const raw = String(v || '').trim().toLowerCase();
    if (raw === 'fixed' || raw === 'trail' || raw === 'nebula' || raw === 'firecracker')
        return raw;
    return 'trail';
}
function pickDefaultVar(key, fallback) {
    const v = String((defaultCssVars && defaultCssVars[key]) || '').trim();
    return v ? v : fallback;
}
function buildDefaultMouseProfile(style) {
    const base = {
        '--mouse-dot-bg': pickDefaultVar('--mouse-dot-bg', 'rgba(255,255,0,0.85)'),
        '--mouse-dot-glow': pickDefaultVar('--mouse-dot-glow', 'rgba(255,255,0,0.25)'),
        '--mouse-dot-size': pickDefaultVar('--mouse-dot-size', '14px'),
        '--mouse-dot-ring': pickDefaultVar('--mouse-dot-ring', '4px'),
        '--mouse-trail-count': '6',
        '--mouse-trail-lag': pickDefaultVar('--mouse-trail-lag', '0.35'),
    };
    if (style === 'fixed')
        return { ...base, '--mouse-trail-count': '0' };
    if (style === 'nebula') {
        return {
            ...base,
            '--mouse-dot-bg': 'rgba(138,43,226,0.85)',
            '--mouse-dot-glow': 'rgba(0,255,255,0.60)',
            '--mouse-trail-count': '6',
            '--mouse-trail-lag': '0.35',
        };
    }
    if (style === 'firecracker') {
        return {
            ...base,
            '--mouse-dot-bg': 'rgba(255,69,0,0.85)',
            '--mouse-dot-glow': 'rgba(255,165,0,0.50)',
            '--mouse-trail-count': '8',
            '--mouse-trail-lag': '0.25',
        };
    }
    return { ...base, '--mouse-trail-count': pickDefaultVar('--mouse-trail-count', '6') };
}
function normalizeMouseDotProfiles(raw) {
    const src = raw && typeof raw === 'object' ? raw : {};
    const out = { fixed: {}, trail: {}, nebula: {}, firecracker: {} };
    for (const style of ['fixed', 'trail', 'nebula', 'firecracker']) {
        const defaults = buildDefaultMouseProfile(style);
        const provided = src[style] && typeof src[style] === 'object' ? src[style] : {};
        const merged = {};
        for (const k of MOUSE_STYLE_KEYS) {
            if (Object.prototype.hasOwnProperty.call(provided, k))
                merged[k] = provided[k];
            else
                merged[k] = defaults[k];
        }
        out[style] = merged;
    }
    return out;
}
function ensureMouseDotProfiles(settings) {
    const next = ensureCssVarsObj(settings);
    next.mouseDotProfiles = normalizeMouseDotProfiles(next.mouseDotProfiles);
    return next;
}
function extractMouseProfileFromSettings(settings) {
    const s = ensureCssVarsObj(settings);
    const cssVars = s.cssVars && typeof s.cssVars === 'object' ? s.cssVars : {};
    const out = {};
    for (const k of MOUSE_STYLE_KEYS) {
        out[k] = Object.prototype.hasOwnProperty.call(cssVars, k) ? cssVars[k] : null;
    }
    return out;
}
function applyMouseProfileToSettings(settings, style) {
    let next = ensureMouseDotProfiles(settings);
    const mode = normalizeMouseDotStyle(style);
    const profile = next.mouseDotProfiles && next.mouseDotProfiles[mode] ? next.mouseDotProfiles[mode] : buildDefaultMouseProfile(mode);
    const cssVars = { ...(next.cssVars || {}) };
    for (const k of MOUSE_STYLE_KEYS)
        cssVars[k] = Object.prototype.hasOwnProperty.call(profile, k) ? profile[k] : cssVars[k];
    next.cssVars = cssVars;
    return next;
}
function readFormToSettings(current) {
    let next = ensureCssVarsObj(current);
    const fade = toNum(els.fadeTime && els.fadeTime.value);
    if (fade !== null)
        next.fadeTime = Math.max(0, Math.round(fade));
    if (els.overlayVisible)
        next.overlayVisible = !!els.overlayVisible.checked;
    if (els.mouseButtonVisible)
        next.mouseButtonVisible = !!els.mouseButtonVisible.checked;
    if (els.mouseDotMoveVisible)
        next.mouseDotMoveVisible = !!els.mouseDotMoveVisible.checked;
    if (els.mouseDotClickVisible)
        next.mouseDotClickVisible = !!els.mouseDotClickVisible.checked;
    if (els.mouseDotStyle) {
        const raw = String(els.mouseDotStyle.value || '').trim().toLowerCase();
        next.mouseDotStyle = raw === 'fixed' ? 'fixed' : (raw === 'nebula' ? 'nebula' : (raw === 'firecracker' ? 'firecracker' : 'trail'));
    }
    if (els.toolbarVisible)
        next.toolbarVisible = !!els.toolbarVisible.checked;
    if (els.autoLaunch)
        next.autoLaunch = !!els.autoLaunch.checked;
    if (els.uiTheme)
        next.uiTheme = els.uiTheme.checked ? 'light' : 'dark';
    if (els.recordSystemAudio)
        next.recordSystemAudio = !!els.recordSystemAudio.checked;
    if (els.recordMicAudio)
        next.recordMicAudio = !!els.recordMicAudio.checked;
    if (els.recordMicDeviceId) {
        next.recordMicDeviceId = normalizeMicDeviceId(els.recordMicDeviceId.value);
    }
    const readHotkey = (el) => String(el && el.value ? el.value : '').trim();
    const saveKeyRaw = readHotkey(els.hotkeyScreenshotSave);
    const clipKeyRaw = readHotkey(els.hotkeyScreenshotClipboard);
    const toolRefimgRaw = readHotkey(els.hotkeyToolRefimg);
    const recordToggleRaw = readHotkey(els.hotkeyRecordToggle);
    const recordSaveGifRaw = readHotkey(els.hotkeyRecordSaveGif);
    const recordSaveMp4Raw = readHotkey(els.hotkeyRecordSaveMp4);
    const toggleOverlayRaw = readHotkey(els.hotkeyToggleOverlay);
    const toggleMouseButtonsRaw = readHotkey(els.hotkeyToggleMouseButtons);
    const toggleMouseDotRaw = readHotkey(els.hotkeyToggleMouseDot);
    const openSettingsRaw = readHotkey(els.hotkeyOpenSettings);
    const toolPenRaw = readHotkey(els.hotkeyToolPen);
    const toolEraserRaw = readHotkey(els.hotkeyToolEraser);
    const toolArrowRaw = readHotkey(els.hotkeyToolArrow);
    const toolRectRaw = readHotkey(els.hotkeyToolRect);
    const toolCircleRaw = readHotkey(els.hotkeyToolCircle);
    const toolTextRaw = readHotkey(els.hotkeyToolText);
    const toolMosaicRaw = readHotkey(els.hotkeyToolMosaic);
    const toolRecordRaw = readHotkey(els.hotkeyToolRecord);
    const undoRaw = readHotkey(els.hotkeyUndo);
    const clearRaw = readHotkey(els.hotkeyClear);
    const curHotkeys = next.hotkeys && typeof next.hotkeys === 'object' ? next.hotkeys : {};
    next.hotkeys = {
        ...curHotkeys,
        screenshotSave: saveKeyRaw ? saveKeyRaw : null,
        screenshotClipboard: clipKeyRaw ? clipKeyRaw : null,
        toolRefimg: toolRefimgRaw ? toolRefimgRaw : null,
        recordToggle: recordToggleRaw ? recordToggleRaw : null,
        recordSaveGif: recordSaveGifRaw ? recordSaveGifRaw : null,
        recordSaveMp4: recordSaveMp4Raw ? recordSaveMp4Raw : null,
        toggleOverlay: toggleOverlayRaw ? toggleOverlayRaw : null,
        toggleMouseButtons: toggleMouseButtonsRaw ? toggleMouseButtonsRaw : null,
        toggleMouseDot: toggleMouseDotRaw ? toggleMouseDotRaw : null,
        openSettings: openSettingsRaw ? openSettingsRaw : null,
        toolPen: toolPenRaw ? toolPenRaw : null,
        toolEraser: toolEraserRaw ? toolEraserRaw : null,
        toolArrow: toolArrowRaw ? toolArrowRaw : null,
        toolRect: toolRectRaw ? toolRectRaw : null,
        toolCircle: toolCircleRaw ? toolCircleRaw : null,
        toolText: toolTextRaw ? toolTextRaw : null,
        toolMosaic: toolMosaicRaw ? toolMosaicRaw : null,
        toolRecord: toolRecordRaw ? toolRecordRaw : null,
        undo: undoRaw ? undoRaw : null,
        clear: clearRaw ? clearRaw : null,
    };
    for (const spec of FIELD_SPECS) {
        const el = els[spec.id];
        if (!el)
            continue;
        if (isDefaultModePair(spec)) {
            next = setCssVar(next, spec.key, null);
            continue;
        }
        if (spec.kind === 'px') {
            const raw = String(el.value || '').trim();
            if (!raw) {
                next = setCssVar(next, spec.key, null);
                continue;
            }
            const n = toNum(raw);
            if (n === null) {
                next = setCssVar(next, spec.key, null);
                continue;
            }
            next = setCssVar(next, spec.key, `${Math.round(n)}px`);
        }
        else if (spec.kind === 'color-alpha') {
            const v = colorWithAlphaFromFields(spec.id, spec.alphaId);
            next = setCssVar(next, spec.key, v ? v : null);
        }
        else {
            const v = String(el.value || '').trim();
            next = setCssVar(next, spec.key, v ? v : null);
        }
    }
    return next;
}
function hydrateForm(settings) {
    const s = ensureCssVarsObj(settings);
    const theme = normalizeUiTheme(s.uiTheme);
    if (els.uiTheme)
        els.uiTheme.checked = theme === 'light';
    applyUiTheme(theme);
    if (els.fadeTime)
        els.fadeTime.value = String(s.fadeTime ?? 1000);
    if (els.overlayVisible)
        els.overlayVisible.checked = s.overlayVisible !== false;
    if (els.mouseButtonVisible)
        els.mouseButtonVisible.checked = s.mouseButtonVisible !== false;
    if (els.mouseDotMoveVisible)
        els.mouseDotMoveVisible.checked = s.mouseDotMoveVisible !== false;
    if (els.mouseDotClickVisible)
        els.mouseDotClickVisible.checked = s.mouseDotClickVisible !== false;
    if (els.mouseDotStyle) {
        const raw = String(s.mouseDotStyle || '').trim().toLowerCase();
        els.mouseDotStyle.value = raw === 'fixed' ? 'fixed' : (raw === 'nebula' ? 'nebula' : (raw === 'firecracker' ? 'firecracker' : 'trail'));
        lastMouseDotStyleUi = normalizeMouseDotStyle(els.mouseDotStyle.value);
    }
    if (els.toolbarVisible)
        els.toolbarVisible.checked = s.toolbarVisible !== false;
    if (els.autoLaunch)
        els.autoLaunch.checked = !!s.autoLaunch;
    if (els.recordSystemAudio)
        els.recordSystemAudio.checked = !!s.recordSystemAudio;
    if (els.recordMicAudio)
        els.recordMicAudio.checked = !!s.recordMicAudio;
    setMicDeviceEnabled(!!s.recordMicAudio);
    setMicDeviceOptions(normalizeMicDeviceId(s.recordMicDeviceId), s.recordMicDeviceId == null);
    try {
        const hotkeys = s && s.hotkeys && typeof s.hotkeys === 'object' ? s.hotkeys : {};
        if (els.hotkeyScreenshotSave)
            els.hotkeyScreenshotSave.value = formatAcceleratorLabel(hotkeys.screenshotSave);
        if (els.hotkeyScreenshotClipboard)
            els.hotkeyScreenshotClipboard.value = formatAcceleratorLabel(hotkeys.screenshotClipboard);
        if (els.hotkeyToolRefimg)
            els.hotkeyToolRefimg.value = formatAcceleratorLabel(hotkeys.toolRefimg);
        if (els.hotkeyRecordToggle)
            els.hotkeyRecordToggle.value = formatAcceleratorLabel(hotkeys.recordToggle);
        if (els.hotkeyRecordSaveGif)
            els.hotkeyRecordSaveGif.value = formatAcceleratorLabel(hotkeys.recordSaveGif);
        if (els.hotkeyRecordSaveMp4)
            els.hotkeyRecordSaveMp4.value = formatAcceleratorLabel(hotkeys.recordSaveMp4);
        if (els.hotkeyToggleOverlay)
            els.hotkeyToggleOverlay.value = formatAcceleratorLabel(hotkeys.toggleOverlay);
        if (els.hotkeyToggleMouseButtons)
            els.hotkeyToggleMouseButtons.value = formatAcceleratorLabel(hotkeys.toggleMouseButtons);
        if (els.hotkeyToggleMouseDot)
            els.hotkeyToggleMouseDot.value = formatAcceleratorLabel(hotkeys.toggleMouseDot);
        if (els.hotkeyOpenSettings)
            els.hotkeyOpenSettings.value = formatAcceleratorLabel(hotkeys.openSettings);
        if (els.hotkeyToolPen)
            els.hotkeyToolPen.value = formatAcceleratorLabel(hotkeys.toolPen);
        if (els.hotkeyToolEraser)
            els.hotkeyToolEraser.value = formatAcceleratorLabel(hotkeys.toolEraser);
        if (els.hotkeyToolArrow)
            els.hotkeyToolArrow.value = formatAcceleratorLabel(hotkeys.toolArrow);
        if (els.hotkeyToolRect)
            els.hotkeyToolRect.value = formatAcceleratorLabel(hotkeys.toolRect);
        if (els.hotkeyToolCircle)
            els.hotkeyToolCircle.value = formatAcceleratorLabel(hotkeys.toolCircle);
        if (els.hotkeyToolText)
            els.hotkeyToolText.value = formatAcceleratorLabel(hotkeys.toolText);
        if (els.hotkeyToolMosaic)
            els.hotkeyToolMosaic.value = formatAcceleratorLabel(hotkeys.toolMosaic);
        if (els.hotkeyToolRecord)
            els.hotkeyToolRecord.value = formatAcceleratorLabel(hotkeys.toolRecord);
        if (els.hotkeyUndo)
            els.hotkeyUndo.value = formatAcceleratorLabel(hotkeys.undo);
        if (els.hotkeyClear)
            els.hotkeyClear.value = formatAcceleratorLabel(hotkeys.clear);
    }
    catch { }
    const hasOwn = (obj, k) => obj && Object.prototype.hasOwnProperty.call(obj, k);
    for (const spec of FIELD_SPECS) {
        const el = els[spec.id];
        if (!el)
            continue;
        const has = hasOwn(s.cssVars, spec.key);
        const raw = has ? s.cssVars[spec.key] : undefined;
        const isCustom = has && raw !== null && typeof raw !== 'undefined';
        if (!isCustom) {
            const dv = String(defaultCssVars[spec.key] || '').trim();
            if (spec.kind === 'px') {
                const n = dv ? dv.replace(/px$/i, '').trim() : '';
                el.value = n ? String(Number(n)) : '';
            }
            else if (spec.kind === 'color-alpha') {
                const parsed = parseCssColorWithAlpha(dv);
                el.value = parsed ? rgbCss(parsed) : dv;
                if (spec.alphaId) {
                    const alphaEl = els[spec.alphaId];
                    if (alphaEl) {
                        if (parsed) {
                            const pct = Math.round((parsed.a ?? 1) * 100);
                            alphaEl.value = String(pct);
                        }
                        else {
                            alphaEl.value = '';
                        }
                    }
                }
            }
            else {
                el.value = dv;
            }
            setDefaultModePair(spec, true);
        }
        else {
            if (spec.kind === 'px') {
                const n = String(raw || '').trim().replace(/px$/i, '');
                el.value = n ? String(Number(n)) : '';
            }
            else if (spec.kind === 'color-alpha') {
                const str = String(raw || '').trim();
                const parsed = parseCssColorWithAlpha(str);
                el.value = parsed ? rgbCss(parsed) : str;
                if (spec.alphaId) {
                    const alphaEl = els[spec.alphaId];
                    if (alphaEl) {
                        if (parsed) {
                            const pct = Math.round((parsed.a ?? 1) * 100);
                            alphaEl.value = String(pct);
                        }
                        else {
                            alphaEl.value = '';
                        }
                    }
                }
            }
            else {
                el.value = String(raw || '');
            }
            setDefaultModePair(spec, false);
        }
        if (spec.kind === 'color' || spec.kind === 'color-alpha')
            updateSwatchFromInput(el);
    }
    syncMouseTrailControls();
}
function syncMouseTrailControls() {
    const mode = els.mouseDotStyle ? String(els.mouseDotStyle.value || '') : 'fixed';
    const show = mode === 'trail' || mode === 'nebula' || mode === 'firecracker';
    try {
        const row1 = els.mouseTrailCount && typeof els.mouseTrailCount.closest === 'function' ? els.mouseTrailCount.closest('.row') : null;
        const row2 = els.mouseTrailLag && typeof els.mouseTrailLag.closest === 'function' ? els.mouseTrailLag.closest('.row') : null;
        if (row1)
            row1.style.display = show ? '' : 'none';
        if (row2)
            row2.style.display = show ? '' : 'none';
    }
    catch { }
}
let currentSettings = { fadeTime: 1000, cssVars: {}, uiTheme: 'light' };
let lastMouseDotStyleUi = 'trail';
let pendingTimer = null;
async function commit(next) {
    if (!window.tt || typeof window.tt.setSettings !== 'function')
        return;
    try {
        const saved = await window.tt.setSettings(next);
        currentSettings = ensureCssVarsObj(saved);
        if (!suppressHydrate)
            hydrateForm(currentSettings);
    }
    catch { }
}
function scheduleCommit(next) {
    if (pendingTimer)
        clearTimeout(pendingTimer);
    pendingTimer = setTimeout(() => { pendingTimer = null; commit(next); }, 150);
}
function wireAutoSave() {
    const controls = Object.values(els).filter(el => el && (el.tagName === 'INPUT' || el.tagName === 'SELECT'));
    const onCommit = (el) => {
        const spec = findSpecById(el.id);
        if (spec) {
            setDefaultModePair(spec, false);
            if (spec.kind === 'color-alpha') {
                const baseEl = els[spec.id];
                if (baseEl)
                    updateSwatchFromInput(baseEl);
            }
            else {
                if (spec.kind === 'color')
                    updateSwatchFromInput(el);
            }
        }
        else {
            if (el.type !== 'checkbox')
                setDefaultMode(el, false);
        }
        let next = readFormToSettings(currentSettings);
        if (el.id === 'uiTheme')
            applyUiTheme(next.uiTheme);
        if (el.id === 'mouseDotStyle') {
            const nextStyle = normalizeMouseDotStyle(next && next.mouseDotStyle);
            const prevStyle = normalizeMouseDotStyle(lastMouseDotStyleUi || (currentSettings && currentSettings.mouseDotStyle));
            next = ensureMouseDotProfiles(next);
            next.mouseDotProfiles = { ...(next.mouseDotProfiles || {}), [prevStyle]: extractMouseProfileFromSettings(next) };
            next.mouseDotStyle = nextStyle;
            next = applyMouseProfileToSettings(next, nextStyle);
            hydrateForm(next);
            lastMouseDotStyleUi = nextStyle;
            scheduleCommit(next);
            return;
        }
        const affectsMouseVars = (spec && MOUSE_STYLE_KEYS.includes(spec.key)) || el.id === 'mouseDotBgAlpha' || el.id === 'mouseDotGlowAlpha';
        if (affectsMouseVars) {
            const activeStyle = normalizeMouseDotStyle(next && next.mouseDotStyle);
            next = ensureMouseDotProfiles(next);
            next.mouseDotProfiles = { ...(next.mouseDotProfiles || {}), [activeStyle]: extractMouseProfileFromSettings(next) };
        }
        if (el.id === 'recordMicAudio')
            setMicDeviceEnabled(!!(els.recordMicAudio && els.recordMicAudio.checked));
        scheduleCommit(next);
    };
    for (const el of controls) {
        el.addEventListener('input', () => onCommit(el));
        el.addEventListener('change', () => onCommit(el));
    }
}
function clampNumberInput(el, n) {
    if (!el)
        return n;
    let out = n;
    const minRaw = el.getAttribute('min');
    const maxRaw = el.getAttribute('max');
    if (minRaw !== null && minRaw !== '') {
        const mn = Number(minRaw);
        if (Number.isFinite(mn))
            out = Math.max(mn, out);
    }
    if (maxRaw !== null && maxRaw !== '') {
        const mx = Number(maxRaw);
        if (Number.isFinite(mx))
            out = Math.min(mx, out);
    }
    return out;
}
function getStep(el) {
    const raw = el && el.getAttribute('step');
    const n = Number(raw);
    if (!Number.isFinite(n) || n <= 0)
        return 1;
    return n;
}
function setNumberValue(el, n) {
    if (!el)
        return;
    const next = clampNumberInput(el, n);
    el.value = String(Number.isFinite(next) ? next : '');
    try {
        el.dispatchEvent(new Event('input', { bubbles: true }));
    }
    catch { }
}
function stepNumber(el, dir) {
    if (!el)
        return;
    const step = getStep(el);
    const cur = toNum(String(el.value || '').trim());
    const base = cur !== null ? cur : (toNum(el.getAttribute('min')) ?? 0);
    setNumberValue(el, base + (dir * step));
}
function upgradeNumberInput(el) {
    if (!el || el.tagName !== 'INPUT' || el.type !== 'number')
        return;
    const id = el.id;
    if (!id)
        return;
    const parent = el.parentElement;
    if (!parent)
        return;
    if (parent.classList && parent.classList.contains('num'))
        return;
    const field = typeof el.closest === 'function' ? el.closest('.field') : null;
    const scope = field || parent;
    if (scope.querySelector(`[data-num-decr="${CSS.escape(id)}"]`) || scope.querySelector(`[data-num-incr="${CSS.escape(id)}"]`))
        return;
    const decr = document.createElement('button');
    decr.type = 'button';
    decr.className = 'num-btn';
    decr.dataset.numDecr = id;
    decr.textContent = '◀';
    const incr = document.createElement('button');
    incr.type = 'button';
    incr.className = 'num-btn';
    incr.dataset.numIncr = id;
    incr.textContent = '▶';
    const wrap = document.createElement('div');
    wrap.className = 'num';
    if (id.toLowerCase().includes('alpha'))
        wrap.classList.add('alpha');
    parent.insertBefore(wrap, el);
    wrap.appendChild(decr);
    wrap.appendChild(el);
    wrap.appendChild(incr);
}
function setupNumberControls() {
    const all = Array.from(document.querySelectorAll('input[type="number"]'));
    for (const el of all)
        upgradeNumberInput(el);
    document.addEventListener('click', (e) => {
        const t = e.target;
        if (!t || t.tagName !== 'BUTTON')
            return;
        const decrId = t.dataset && t.dataset.numDecr;
        const incrId = t.dataset && t.dataset.numIncr;
        if (!decrId && !incrId)
            return;
        const id = decrId || incrId;
        const inputEl = document.getElementById(id);
        if (!inputEl)
            return;
        stepNumber(inputEl, decrId ? -1 : 1);
    });
    const inputs = Array.from(document.querySelectorAll('input[type="number"]'));
    for (const el of inputs) {
        let dragging = false;
        let startX = 0;
        let startVal = 0;
        let step = 1;
        let moved = false;
        let raf = 0;
        let lastX = 0;
        const onMove = (ev) => {
            if (!dragging)
                return;
            lastX = ev.clientX;
            if (raf)
                return;
            raf = requestAnimationFrame(() => {
                raf = 0;
                const dx = lastX - startX;
                if (!moved && Math.abs(dx) >= 3)
                    moved = true;
                if (!moved)
                    return;
                const ticks = Math.round(dx / 8);
                const next = startVal + ticks * step;
                setNumberValue(el, next);
            });
        };
        const onUp = () => {
            if (!dragging)
                return;
            dragging = false;
            if (raf) {
                cancelAnimationFrame(raf);
                raf = 0;
            }
            try {
                document.body.style.cursor = '';
            }
            catch { }
            window.removeEventListener('mousemove', onMove);
            window.removeEventListener('mouseup', onUp);
        };
        el.addEventListener('mousedown', (ev) => {
            if (ev.button !== 0)
                return;
            const v = toNum(String(el.value || '').trim());
            startVal = v !== null ? v : (toNum(el.getAttribute('min')) ?? 0);
            startX = ev.clientX;
            step = getStep(el);
            moved = false;
            dragging = true;
            lastX = ev.clientX;
            try {
                document.body.style.cursor = 'ew-resize';
            }
            catch { }
            window.addEventListener('mousemove', onMove);
            window.addEventListener('mouseup', onUp);
        });
    }
}
async function resetByKeys(keys, opts = null) {
    const o = (opts && typeof opts === 'object') ? opts : { resetFadeTime: false, patch: null };
    const resetFadeTime = !!o.resetFadeTime;
    const patch = o.patch;
    if (!window.tt || typeof window.tt.setSettings !== 'function')
        return;
    const next = ensureCssVarsObj(currentSettings);
    const cssVars = { ...(next.cssVars || {}) };
    const out = { ...next, ...(patch && typeof patch === 'object' ? patch : {}) };
    for (const k of keys || []) {
        if (k.startsWith('--')) {
            cssVars[k] = null;
        }
        else if (k === 'recordMicDeviceId') {
            out[k] = null;
        }
        else {
            out[k] = true;
        }
    }
    if (resetFadeTime)
        out.fadeTime = 1000;
    out.cssVars = cssVars;
    try {
        const saved = await window.tt.setSettings(out);
        currentSettings = ensureCssVarsObj(saved);
        hydrateForm(currentSettings);
    }
    catch { }
}
function setupColorPicker() {
    const panel = document.createElement('div');
    panel.style.position = 'fixed';
    panel.style.zIndex = '1006';
    panel.style.background = '#ffffff';
    panel.style.borderRadius = '8px';
    panel.style.boxShadow = '0 6px 20px rgba(0,0,0,0.35)';
    panel.style.padding = '8px';
    panel.style.display = 'none';
    panel.style.userSelect = 'none';
    const canvas = document.createElement('canvas');
    canvas.width = 240;
    canvas.height = 140;
    canvas.style.display = 'block';
    canvas.style.cursor = 'crosshair';
    canvas.style.marginBottom = '8px';
    const hueRange = document.createElement('input');
    hueRange.type = 'range';
    hueRange.min = '0';
    hueRange.max = '360';
    hueRange.value = '0';
    hueRange.style.width = '240px';
    const alphaWrap = document.createElement('div');
    alphaWrap.style.width = '240px';
    alphaWrap.style.marginTop = '8px';
    alphaWrap.style.display = 'none';
    const alphaRange = document.createElement('input');
    alphaRange.type = 'range';
    alphaRange.min = '0';
    alphaRange.max = '100';
    alphaRange.step = '1';
    alphaRange.value = '100';
    alphaRange.style.width = '240px';
    alphaWrap.appendChild(alphaRange);
    panel.appendChild(canvas);
    panel.appendChild(hueRange);
    panel.appendChild(alphaWrap);
    document.body.appendChild(panel);
    const ctx = canvas.getContext('2d');
    let hue = 0;
    let sat = 1;
    let val = 1;
    let activeInput = null;
    let activeSwatch = null;
    const hsvToRgb = (h, s, v) => {
        const f = (n) => {
            const k = (n + h * 6) % 6;
            return v - v * s * Math.max(Math.min(k, 4 - k, 1), 0);
        };
        const r = Math.round(f(5) * 255);
        const g = Math.round(f(3) * 255);
        const b = Math.round(f(1) * 255);
        return { r, g, b };
    };
    const rgbToCss = ({ r, g, b }) => `rgb(${r}, ${g}, ${b})`;
    const rgbToHsv = (r, g, b) => {
        const rn = r / 255, gn = g / 255, bn = b / 255;
        const max = Math.max(rn, gn, bn);
        const min = Math.min(rn, gn, bn);
        const d = max - min;
        let h = 0;
        if (d !== 0) {
            if (max === rn)
                h = ((gn - bn) / d) % 6;
            else if (max === gn)
                h = (bn - rn) / d + 2;
            else
                h = (rn - gn) / d + 4;
            h /= 6;
            if (h < 0)
                h += 1;
        }
        const s = max === 0 ? 0 : d / max;
        const v = max;
        return { h, s, v };
    };
    const parseCssColor = (s) => {
        const str = String(s || '').trim();
        if (!str)
            return null;
        const hex3 = /^#([0-9a-f]{3})$/i.exec(str);
        if (hex3) {
            const h = hex3[1];
            return {
                r: parseInt(h[0] + h[0], 16),
                g: parseInt(h[1] + h[1], 16),
                b: parseInt(h[2] + h[2], 16),
            };
        }
        const hex6 = /^#([0-9a-f]{6})$/i.exec(str);
        if (hex6) {
            const h = hex6[1];
            return {
                r: parseInt(h.slice(0, 2), 16),
                g: parseInt(h.slice(2, 4), 16),
                b: parseInt(h.slice(4, 6), 16),
            };
        }
        const rgb = /^rgba?\(\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*(\d{1,3})(?:\s*,\s*([\d.]+))?\s*\)$/i.exec(str);
        if (rgb) {
            const r = Math.max(0, Math.min(255, Number(rgb[1]) || 0));
            const g = Math.max(0, Math.min(255, Number(rgb[2]) || 0));
            const b = Math.max(0, Math.min(255, Number(rgb[3]) || 0));
            return { r, g, b };
        }
        return null;
    };
    function drawSV() {
        const base = hsvToRgb(hue / 360, 1, 1);
        const w = canvas.width, h = canvas.height;
        const gradX = ctx.createLinearGradient(0, 0, w, 0);
        gradX.addColorStop(0, '#ffffff');
        gradX.addColorStop(1, rgbToCss(base));
        ctx.fillStyle = gradX;
        ctx.fillRect(0, 0, w, h);
        const gradY = ctx.createLinearGradient(0, 0, 0, h);
        gradY.addColorStop(0, 'rgba(0,0,0,0)');
        gradY.addColorStop(1, 'rgba(0,0,0,1)');
        ctx.fillStyle = gradY;
        ctx.fillRect(0, 0, w, h);
        const x = sat * w;
        const y = (1 - val) * h;
        ctx.beginPath();
        ctx.arc(x, y, 6, 0, Math.PI * 2);
        ctx.lineWidth = 2;
        ctx.strokeStyle = '#fff';
        ctx.stroke();
    }
    function setActiveColor() {
        const css = rgbToCss(hsvToRgb(hue / 360, sat, val));
        const spec = activeInput && activeInput.id ? findSpecById(activeInput.id) : null;
        if (activeInput) {
            activeInput.value = css;
            if (spec)
                setDefaultModePair(spec, false);
            else
                setDefaultMode(activeInput, false);
        }
        if (activeSwatch) {
            if (spec && spec.kind === 'color-alpha') {
                const v = colorWithAlphaFromFields(spec.id, spec.alphaId);
                activeSwatch.style.background = v ? v : 'transparent';
            }
            else {
                activeSwatch.style.background = css;
            }
        }
    }
    function posFromEvent(e) {
        const rect = canvas.getBoundingClientRect();
        const x = Math.max(0, Math.min(rect.width, e.clientX - rect.left));
        const y = Math.max(0, Math.min(rect.height, e.clientY - rect.top));
        return { x, y };
    }
    let dragging = false;
    canvas.addEventListener('mousedown', (e) => {
        dragging = true;
        const p = posFromEvent(e);
        sat = p.x / canvas.width;
        val = 1 - p.y / canvas.height;
        setActiveColor();
        drawSV();
    });
    window.addEventListener('mousemove', (e) => {
        if (!dragging)
            return;
        const p = posFromEvent(e);
        sat = p.x / canvas.width;
        val = 1 - p.y / canvas.height;
        setActiveColor();
        drawSV();
    });
    window.addEventListener('mouseup', () => {
        if (!dragging)
            return;
        dragging = false;
        if (!activeInput)
            return;
        const next = readFormToSettings(currentSettings);
        scheduleCommit(next);
    });
    hueRange.addEventListener('input', (e) => {
        const t = e && e.target ? e.target : null;
        hue = Number(t && t.value) || 0;
        setActiveColor();
        drawSV();
        if (!activeInput)
            return;
        const next = readFormToSettings(currentSettings);
        scheduleCommit(next);
    });
    alphaRange.addEventListener('input', () => {
        if (!activeInput || !activeInput.id)
            return;
        const spec = findSpecById(activeInput.id);
        if (!spec || spec.kind !== 'color-alpha' || !spec.alphaId)
            return;
        const alphaEl = els[spec.alphaId];
        if (!alphaEl)
            return;
        const pct = Math.max(0, Math.min(100, Math.round(Number(alphaRange.value) || 0)));
        alphaEl.value = String(pct);
        try {
            alphaEl.dispatchEvent(new Event('input', { bubbles: true }));
        }
        catch { }
        setActiveColor();
    });
    function showPanel(inputEl, swatchEl) {
        activeInput = inputEl;
        activeSwatch = swatchEl;
        const spec = activeInput && activeInput.id ? findSpecById(activeInput.id) : null;
        if (spec && spec.kind === 'color-alpha' && spec.alphaId) {
            const alphaEl = els[spec.alphaId];
            const pct = toNum(alphaEl && alphaEl.value) ?? 100;
            alphaRange.value = String(Math.max(0, Math.min(100, Math.round(pct))));
            alphaWrap.style.display = 'block';
        }
        else {
            alphaWrap.style.display = 'none';
        }
        try {
            const parsed = parseCssColor(String(inputEl && inputEl.value ? inputEl.value : ''));
            if (parsed) {
                const hsv = rgbToHsv(parsed.r, parsed.g, parsed.b);
                hue = Math.round(hsv.h * 360);
                sat = hsv.s;
                val = hsv.v;
                hueRange.value = String(hue);
            }
        }
        catch { }
        const r = swatchEl.getBoundingClientRect();
        panel.style.display = 'block';
        const w = panel.offsetWidth || 256;
        const left = Math.max(8, Math.min(Math.round(r.left), window.innerWidth - w - 8));
        panel.style.left = `${left}px`;
        panel.style.top = `${Math.round(r.bottom + 6)}px`;
        suppressHydrate = true;
        drawSV();
        setActiveColor();
    }
    function hidePanel() {
        panel.style.display = 'none';
        suppressHydrate = false;
        if (activeInput) {
            const next = readFormToSettings(currentSettings);
            scheduleCommit(next);
        }
        activeInput = null;
        activeSwatch = null;
    }
    panel.addEventListener('mouseleave', hidePanel);
    document.addEventListener('keydown', (e) => { if (e.key === 'Escape')
        hidePanel(); });
    return { showPanel };
}
async function init() {
    defaultCssVars = readDefaultCssVars();
    applyUiTheme(currentSettings.uiTheme);
    setupSidebarNav();
    setupHotkeyCaptureUI();
    if (els.winClose) {
        els.winClose.addEventListener('click', () => {
            try {
                if (window.tt && typeof window.tt.closeSettings === 'function')
                    window.tt.closeSettings();
                else
                    window.close();
            }
            catch {
                try {
                    window.close();
                }
                catch { }
            }
        });
    }
    setupNumberControls();
    const picker = setupColorPicker();
    for (const spec of FIELD_SPECS) {
        if (spec.kind !== 'color' && spec.kind !== 'color-alpha')
            continue;
        const inputEl = els[spec.id];
        if (!inputEl)
            continue;
        const sw = ensureSwatch(inputEl);
        if (!sw)
            continue;
        sw.addEventListener('click', () => picker.showPanel(inputEl, sw));
    }
    if (!window.tt || typeof window.tt.getSettings !== 'function') {
        hydrateForm(currentSettings);
        wireAutoSave();
        try {
            await refreshMicDevices();
            setMicDeviceOptions(normalizeMicDeviceId(currentSettings.recordMicDeviceId), currentSettings.recordMicDeviceId == null);
        }
        catch { }
        try {
            if (navigator.mediaDevices && typeof navigator.mediaDevices.addEventListener === 'function') {
                navigator.mediaDevices.addEventListener('devicechange', async () => {
                    await refreshMicDevices();
                    setMicDeviceOptions(normalizeMicDeviceId(currentSettings.recordMicDeviceId), currentSettings.recordMicDeviceId == null);
                });
            }
        }
        catch { }
        return;
    }
    try {
        const s = await window.tt.getSettings();
        currentSettings = ensureCssVarsObj(s);
    }
    catch { }
    hydrateForm(currentSettings);
    wireAutoSave();
    try {
        await refreshMicDevices();
        setMicDeviceOptions(normalizeMicDeviceId(currentSettings.recordMicDeviceId), currentSettings.recordMicDeviceId == null);
    }
    catch { }
    try {
        if (navigator.mediaDevices && typeof navigator.mediaDevices.addEventListener === 'function') {
            navigator.mediaDevices.addEventListener('devicechange', async () => {
                await refreshMicDevices();
                setMicDeviceOptions(normalizeMicDeviceId(currentSettings.recordMicDeviceId), currentSettings.recordMicDeviceId == null);
            });
        }
    }
    catch { }
    if (els.recordMicAudio) {
        els.recordMicAudio.addEventListener('input', async () => {
            if (!els.recordMicAudio.checked)
                return;
            try {
                await refreshMicDevices();
                setMicDeviceOptions(normalizeMicDeviceId(currentSettings.recordMicDeviceId), currentSettings.recordMicDeviceId == null);
            }
            catch { }
        });
    }
    if (els.resetKey) {
        els.resetKey.addEventListener('click', () => resetByKeys(KEY_GROUP, { resetFadeTime: true }));
    }
    if (els.resetMouse) {
        els.resetMouse.addEventListener('click', () => resetByKeys(MOUSE_GROUP, { resetFadeTime: false, patch: { mouseDotStyle: 'trail', mouseDotProfiles: null } }));
    }
    if (els.resetToolbar) {
        els.resetToolbar.addEventListener('click', () => resetByKeys(TOOLBAR_GROUP, { resetFadeTime: false }));
    }
    if (els.resetRecord) {
        els.resetRecord.addEventListener('click', () => resetByKeys(RECORD_GROUP, { resetFadeTime: false }));
    }
    if (els.resetHotkeys) {
        els.resetHotkeys.addEventListener('click', () => {
            try {
                const applyVal = (el, value) => {
                    if (!el)
                        return;
                    if (hotkeyCaptureEl === el)
                        stopHotkeyCapture(false);
                    try {
                        el.value = String(value ?? '');
                    }
                    catch { }
                    try {
                        el.dispatchEvent(new Event('input', { bubbles: true }));
                    }
                    catch { }
                };
                applyVal(els.hotkeyScreenshotSave, 'Ctrl + Alt + S');
                applyVal(els.hotkeyScreenshotClipboard, 'Alt + S');
                applyVal(els.hotkeyToolRefimg, '');
                applyVal(els.hotkeyRecordToggle, 'Ctrl + Alt + R');
                applyVal(els.hotkeyRecordSaveGif, 'Ctrl + Alt + G');
                applyVal(els.hotkeyRecordSaveMp4, 'Ctrl + Alt + M');
                applyVal(els.hotkeyUndo, 'Ctrl + Z');
                applyVal(els.hotkeyToggleOverlay, '');
                applyVal(els.hotkeyToggleMouseButtons, '');
                applyVal(els.hotkeyToggleMouseDot, '');
                applyVal(els.hotkeyOpenSettings, '');
                applyVal(els.hotkeyToolPen, '');
                applyVal(els.hotkeyToolEraser, '');
                applyVal(els.hotkeyToolArrow, '');
                applyVal(els.hotkeyToolRect, '');
                applyVal(els.hotkeyToolCircle, '');
                applyVal(els.hotkeyToolText, '');
                applyVal(els.hotkeyToolMosaic, '');
                applyVal(els.hotkeyToolRecord, '');
                applyVal(els.hotkeyClear, '');
            }
            catch { }
        });
    }
    if (els.resetAll) {
        els.resetAll.addEventListener('click', async () => {
            if (!window.tt || typeof window.tt.resetSettings !== 'function')
                return;
            try {
                const s = await window.tt.resetSettings();
                currentSettings = ensureCssVarsObj(s);
                hydrateForm(currentSettings);
            }
            catch { }
        });
    }
    try {
        if (typeof window.tt.onSettingsUpdated === 'function') {
            window.tt.onSettingsUpdated((s) => {
                currentSettings = ensureCssVarsObj(s);
                hydrateForm(currentSettings);
            });
        }
    }
    catch { }
}
const titlebar = document.querySelector('.titlebar');
if (titlebar && window.tt && typeof window.tt.startSettingsDrag === 'function') {
    titlebar.addEventListener('mousedown', (e) => {
        const me = e;
        const target = me.target;
        if (target.closest('.win-actions'))
            return;
        const rect = titlebar.getBoundingClientRect();
        const offsetX = me.clientX - rect.left;
        const offsetY = me.clientY - rect.top;
        window.tt.startSettingsDrag({ offsetX, offsetY });
    });
}
init();
//# sourceMappingURL=settings.js.map