import { createKeyBubble } from './ui/bubble.js';
import { createMouseDot } from './ui/mouse.js';
import { getKeyIdFromEvent, nameFromEvent, buildLabelFromPressed, mouseButtonLabel, mouseButtonIconSVG } from './keymap.js';
import { createTooltipEnhancer } from './utils/tooltipEnhancer.js';
const container = document.getElementById('container');
let bubbleFadeTime = 1000;
const bubble = createKeyBubble({ container, getFadeTime: () => bubbleFadeTime });
const mouse = createMouseDot();
let appliedCssKeys = new Set();
const root = document.documentElement;
let refImageViews = new Map();
function ensureSettingsObj(v) {
    if (!v || typeof v !== 'object')
        return { fadeTime: 1000, cssVars: {}, uiTheme: 'light' };
    const cssVars = v.cssVars && typeof v.cssVars === 'object' ? v.cssVars : {};
    const fadeTime = typeof v.fadeTime === 'number' && Number.isFinite(v.fadeTime) ? Math.max(0, Math.round(v.fadeTime)) : 1000;
    const uiTheme = String(v.uiTheme || '').trim().toLowerCase() === 'dark' ? 'dark' : 'light';
    const styleRaw = String(v.mouseDotStyle || '').trim().toLowerCase();
    const mouseDotStyle = (styleRaw === 'fixed' || styleRaw === 'nebula') ? styleRaw : 'trail';
    const recordSystemAudio = typeof v.recordSystemAudio === 'boolean' ? v.recordSystemAudio : false;
    const recordMicAudio = typeof v.recordMicAudio === 'boolean' ? v.recordMicAudio : false;
    const recordMicDeviceId = String(v.recordMicDeviceId || '').trim() || null;
    return { ...v, fadeTime, cssVars, uiTheme, mouseDotStyle, recordSystemAudio, recordMicAudio, recordMicDeviceId };
}
const displayId = (() => {
    try {
        return String(new URLSearchParams(String(location && location.search ? location.search : '')).get('displayId') || '');
    }
    catch {
        return '';
    }
})();
function formatAcceleratorLabel(acc) {
    const s = String(acc || '').trim();
    if (!s)
        return '';
    return s.replace(/\s*\+\s*/g, ' + ');
}
const HOTKEY_MOD_ORDER = ['Ctrl', 'Shift', 'Alt', 'Super'];
let currentHotkeysNorm = { undo: '', clear: '', toolPen: '', toolEraser: '', toolArrow: '', toolRect: '', toolCircle: '', toolText: '', toolMosaic: '', toolRefimg: '', toolRecord: '' };
function normalizeHotkeyString(raw) {
    const s = String(raw || '').trim();
    if (!s)
        return '';
    const cleaned = s.replace(/\s+/g, '');
    const parts = cleaned.split('+').map(x => String(x || '').trim()).filter(Boolean);
    if (!parts.length)
        return '';
    const mods = new Set();
    const others = [];
    for (const p0 of parts) {
        const p = String(p0 || '').trim();
        if (!p)
            continue;
        const low = p.toLowerCase();
        if (low === 'control' || low === 'ctrl' || low === 'commandorcontrol' || low === 'cmdorctrl') {
            mods.add('Ctrl');
            continue;
        }
        if (low === 'shift') {
            mods.add('Shift');
            continue;
        }
        if (low === 'alt' || low === 'option') {
            mods.add('Alt');
            continue;
        }
        if (low === 'win' || low === 'meta' || low === 'super') {
            mods.add('Super');
            continue;
        }
        let key = p;
        if (key === 'Esc')
            key = 'Escape';
        if (key.length === 1)
            key = key.toUpperCase();
        others.push(key);
    }
    const orderedMods = HOTKEY_MOD_ORDER.filter(m => mods.has(m));
    return orderedMods.concat(others).join('+');
}
function chordFromPressed(map) {
    const nameSet = new Set();
    map.forEach(v => { const n = v && v.name ? v.name : ''; if (n)
        nameSet.add(n); });
    const mods = new Set();
    if (nameSet.has('Ctrl'))
        mods.add('Ctrl');
    if (nameSet.has('Shift'))
        mods.add('Shift');
    if (nameSet.has('Alt'))
        mods.add('Alt');
    if (nameSet.has('Win'))
        mods.add('Super');
    const others = Array.from(nameSet).filter(n => n !== 'Ctrl' && n !== 'Shift' && n !== 'Alt' && n !== 'Win' && n !== 'Caps Lock');
    if (others.length !== 1)
        return '';
    let key = String(others[0] || '');
    if (key === 'Esc')
        key = 'Escape';
    if (key.length === 1)
        key = key.toUpperCase();
    const orderedMods = HOTKEY_MOD_ORDER.filter(m => mods.has(m));
    return orderedMods.concat([key]).join('+');
}
let screenshotSelectActive = false;
let screenshotSelectCancel = null;
let screenshotSelectSession = null;
let screenshotSelectLayer = null;
let screenshotSelectBox = null;
let screenshotSelectLabel = null;
let screenshotSelectBg = null;
function ensureScreenshotSelectLayer() {
    if (screenshotSelectLayer && screenshotSelectLayer.isConnected)
        return screenshotSelectLayer;
    const layer = document.createElement('div');
    layer.id = 'screenshot-select-layer';
    const bg = document.createElement('img');
    bg.id = 'screenshot-select-bg';
    bg.alt = '';
    bg.draggable = false;
    bg.decoding = 'async';
    bg.style.position = 'absolute';
    bg.style.inset = '0';
    bg.style.width = '100%';
    bg.style.height = '100%';
    bg.style.objectFit = 'fill';
    bg.style.pointerEvents = 'none';
    bg.style.display = 'none';
    const box = document.createElement('div');
    box.id = 'screenshot-select-box';
    const label = document.createElement('div');
    label.id = 'screenshot-select-label';
    box.appendChild(label);
    for (const dir of ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w']) {
        const h = document.createElement('div');
        h.className = 'screenshot-select-handle';
        h.dataset.dir = dir;
        box.appendChild(h);
    }
    layer.appendChild(bg);
    layer.appendChild(box);
    document.body.appendChild(layer);
    screenshotSelectLayer = layer;
    screenshotSelectBg = bg;
    screenshotSelectBox = box;
    screenshotSelectLabel = label;
    return layer;
}
function setSelectBoxRect(rect) {
    const r = rect && typeof rect === 'object' ? rect : { x: 0, y: 0, w: 0, h: 0 };
    const x = Math.max(0, Math.round(Number(r.x) || 0));
    const y = Math.max(0, Math.round(Number(r.y) || 0));
    const w = Math.max(0, Math.round(Number(r.w) || 0));
    const h = Math.max(0, Math.round(Number(r.h) || 0));
    if (screenshotSelectBox) {
        screenshotSelectBox.style.left = `${x}px`;
        screenshotSelectBox.style.top = `${y}px`;
        screenshotSelectBox.style.width = `${w}px`;
        screenshotSelectBox.style.height = `${h}px`;
        const ready = !!(screenshotSelectSession && screenshotSelectSession.created && w > 0 && h > 0);
        try {
            screenshotSelectBox.classList.toggle('ready', ready);
        }
        catch { }
    }
    if (screenshotSelectLabel) {
        screenshotSelectLabel.textContent = `${w} × ${h}`;
        screenshotSelectLabel.style.display = w > 0 && h > 0 ? 'block' : 'none';
        screenshotSelectLabel.style.left = '0px';
        screenshotSelectLabel.style.top = '0px';
    }
}
async function pickScreenshotRegion(opts) {
    const layer = ensureScreenshotSelectLayer();
    const vw = Math.max(1, Math.round(window.innerWidth || 1));
    const vh = Math.max(1, Math.round(window.innerHeight || 1));
    const frozenRaw = opts && typeof opts === 'object' ? String(opts.frozenPngBase64 || '').trim() : '';
    if (screenshotSelectBg) {
        if (frozenRaw) {
            screenshotSelectBg.src = `data:image/png;base64,${frozenRaw}`;
            screenshotSelectBg.style.display = 'block';
        }
        else {
            screenshotSelectBg.src = '';
            screenshotSelectBg.style.display = 'none';
        }
    }
    try {
        if (window.tt && typeof window.tt.setScreenshotPickMode === 'function') {
            window.tt.setScreenshotPickMode(true, { displayId });
        }
    }
    catch { }
    screenshotSelectActive = true;
    layer.classList.add('active');
    setSelectBoxRect({ x: 0, y: 0, w: 0, h: 0 });
    const prevUserSelect = document.body.style.userSelect;
    document.body.style.userSelect = 'none';
    return await new Promise((resolve) => {
        const cleanup = () => {
            screenshotSelectActive = false;
            screenshotSelectCancel = null;
            screenshotSelectSession = null;
            layer.classList.remove('active');
            setSelectBoxRect({ x: 0, y: 0, w: 0, h: 0 });
            document.body.style.userSelect = prevUserSelect;
            if (screenshotSelectBg) {
                screenshotSelectBg.src = '';
                screenshotSelectBg.style.display = 'none';
            }
            try {
                if (window.tt && typeof window.tt.setScreenshotPickMode === 'function') {
                    window.tt.setScreenshotPickMode(false, { displayId });
                }
            }
            catch { }
        };
        const done = (res) => { cleanup(); resolve(res); };
        const cancel = () => done(null);
        screenshotSelectCancel = cancel;
        const normalize = (p1, p2) => {
            const x1 = Math.max(0, Math.min(vw, Math.round(Number(p1 && p1.x) || 0)));
            const y1 = Math.max(0, Math.min(vh, Math.round(Number(p1 && p1.y) || 0)));
            const x2 = Math.max(0, Math.min(vw, Math.round(Number(p2 && p2.x) || 0)));
            const y2 = Math.max(0, Math.min(vh, Math.round(Number(p2 && p2.y) || 0)));
            const x = Math.min(x1, x2);
            const y = Math.min(y1, y2);
            const w = Math.max(0, Math.abs(x2 - x1));
            const h = Math.max(0, Math.abs(y2 - y1));
            return { x, y, w, h };
        };
        const cursorForDir = (dir) => {
            switch (String(dir || '')) {
                case 'n':
                case 's': return 'ns-resize';
                case 'e':
                case 'w': return 'ew-resize';
                case 'ne':
                case 'sw': return 'nesw-resize';
                case 'nw':
                case 'se': return 'nwse-resize';
                default: return 'move';
            }
        };
        const edgeDirAt = (pos, rect) => {
            const p = (pos && typeof pos === 'object') ? pos : null;
            const r = (rect && typeof rect === 'object') ? rect : null;
            if (!p || !r)
                return '';
            const w = Math.max(0, Number(r.w) || 0);
            const h = Math.max(0, Number(r.h) || 0);
            if (w < 2 || h < 2)
                return '';
            const m = 10;
            const x = Number(p.x) - Number(r.x || 0);
            const y = Number(p.y) - Number(r.y || 0);
            const onW = x >= -m && x <= m;
            const onE = x >= w - m && x <= w + m;
            const onN = y >= -m && y <= m;
            const onS = y >= h - m && y <= h + m;
            let d = '';
            if (onN)
                d += 'n';
            else if (onS)
                d += 's';
            if (onW)
                d += 'w';
            else if (onE)
                d += 'e';
            return d;
        };
        const clampRect = (r0) => {
            const r = (r0 && typeof r0 === 'object') ? r0 : { x: 0, y: 0, w: 0, h: 0 };
            const x = Math.max(0, Math.min(vw, Math.round(Number(r.x) || 0)));
            const y = Math.max(0, Math.min(vh, Math.round(Number(r.y) || 0)));
            const w = Math.max(0, Math.min(vw - x, Math.round(Number(r.w) || 0)));
            const h = Math.max(0, Math.min(vh - y, Math.round(Number(r.h) || 0)));
            return { x, y, w, h };
        };
        screenshotSelectSession = {
            vw,
            vh,
            rect: { x: 0, y: 0, w: 0, h: 0 },
            start: null,
            dragging: false,
            mode: '',
            created: false,
            anchorRect: null,
            resizeDir: '',
            moveOffset: null,
            lastClickTs: 0,
            lastClickPos: null,
            normalize,
            edgeDirAt,
            cursorForDir,
            clampRect,
            done,
        };
    });
}
let recordUiActive = false;
let recordSelectActive = false;
let recordLayer = null;
let recordBox = null;
let recordTitle = null;
let recordToolbar = null;
let recordToolbarHotzone = null;
let recordSaveMenu = null;
let recordSelectSession = null;
let recordState = 'idle';
let recordStream = null;
let recordSourceStreams = [];
let recordAudioContext = null;
let recordRecorder = null;
let recordChunks = [];
let recordLastWebm = null;
let recordLastVideoInfo = null;
let recordFfmpeg = null;
let recordFfmpegLoading = null;
let recordSelectCancel = null;
let recordToolbarWired = false;
let recordUiRects = [];
let recordSysAudioBtn = null;
let recordMicAudioBtn = null;
let recordFullscreenBtn = null;
let recordMicDeviceSelect = null;
let recordMicSelectBtn = null;
let recordMicSelectLabel = null;
let recordMicMenu = null;
let cachedRecordMicDevices = [];
let recordMicDevicesSeq = 0;
let recordMicDevicesLastRefreshTs = 0;
let recordToolbarAutoHideTimer = null;
let recordToolbarLastMousePos = null;
let recordPrevRectBeforeFullscreen = null;
let recordDesktopSourceCache = { displayId: '', sourceId: '', ts: 0 };
let recordDesktopSourceLoading = null;
const recordIconPlay = `
  <svg class="tb-icon" viewBox="0 0 24 24" role="img" aria-label="开始">
    <path d="M9 7l10 5-10 5V7z" fill="currentColor"/>
  </svg>
`;
const recordIconPause = `
  <svg class="tb-icon" viewBox="0 0 24 24" role="img" aria-label="暂停">
    <path d="M8 7h3v10H8V7zm5 0h3v10h-3V7z" fill="currentColor"/>
  </svg>
`;
function ensureRecordElements() {
    if (!recordLayer || !recordLayer.isConnected)
        recordLayer = document.getElementById('record-layer');
    if (!recordBox || !recordBox.isConnected)
        recordBox = document.getElementById('record-box');
    if (!recordTitle || !recordTitle.isConnected)
        recordTitle = document.getElementById('record-title');
    if (!recordToolbar || !recordToolbar.isConnected)
        recordToolbar = document.getElementById('record-toolbar');
    if (!recordToolbarHotzone || !recordToolbarHotzone.isConnected)
        recordToolbarHotzone = document.getElementById('record-toolbar-hotzone');
    if (!recordSaveMenu || !recordSaveMenu.isConnected)
        recordSaveMenu = document.getElementById('record-save-menu');
    if (!recordMicMenu || !recordMicMenu.isConnected)
        recordMicMenu = document.getElementById('record-mic-menu');
    return { recordLayer, recordBox, recordTitle, recordToolbar, recordToolbarHotzone, recordSaveMenu, recordMicMenu };
}
function pad2(n) { return String(n).padStart(2, '0'); }
function pad3(n) { return String(n).padStart(3, '0'); }
function formatTimestamp(tsMs) {
    const d = new Date(Math.max(0, Number(tsMs) || 0));
    const y = d.getFullYear();
    const m = pad2(d.getMonth() + 1);
    const day = pad2(d.getDate());
    const hh = pad2(d.getHours());
    const mm = pad2(d.getMinutes());
    const ss = pad2(d.getSeconds());
    const ms = pad3(d.getMilliseconds());
    return `${y}${m}${day}_${hh}${mm}${ss}_${ms}`;
}
function setRecordHotzoneVisible(visible) {
    ensureRecordElements();
    if (!recordToolbarHotzone)
        return;
    const next = !!visible;
    const prev = recordToolbarHotzone.classList.contains('visible');
    try {
        recordToolbarHotzone.classList.toggle('visible', next);
    }
    catch { }
    if (prev !== next) {
        positionRecordToolbar();
        syncRecordInteractiveRects();
    }
}
function pointInDomRect(pos, domRect) {
    const p = (pos && typeof pos === 'object') ? pos : null;
    const r = domRect && typeof domRect === 'object' ? domRect : null;
    if (!p || !r)
        return false;
    const x = Math.round(Number(p.x) || 0);
    const y = Math.round(Number(p.y) || 0);
    const left = Math.round(Number(r.left) || 0);
    const top = Math.round(Number(r.top) || 0);
    const right = Math.round(Number(r.right) || (left + (Number(r.width) || 0)));
    const bottom = Math.round(Number(r.bottom) || (top + (Number(r.height) || 0)));
    return x >= left && x <= right && y >= top && y <= bottom;
}
function positionRecordToolbar() {
    ensureRecordElements();
    if (!recordToolbar)
        return;
    const margin = 12;
    const tb = recordToolbar.getBoundingClientRect();
    const w = Math.max(1, tb.width || recordToolbar.offsetWidth || 1);
    const h = Math.max(1, tb.height || recordToolbar.offsetHeight || 1);
    let cx = window.innerWidth / 2;
    let top = margin;
    if (recordBox && recordSelectSession && recordSelectSession.created) {
        const s = recordBox.getBoundingClientRect();
        const sw = Math.max(0, s.width || 0);
        const sh = Math.max(0, s.height || 0);
        if (sw >= 2 && sh >= 2) {
            cx = s.left + sw / 2;
            const outsideTop = s.bottom + margin;
            const outsideFits = outsideTop + h <= window.innerHeight - margin;
            top = outsideFits ? outsideTop : (s.bottom - margin - h);
            top = clamp(top, margin, window.innerHeight - margin - h);
            cx = clamp(cx, margin + w / 2, window.innerWidth - margin - w / 2);
        }
        else {
            top = clamp(top, margin, window.innerHeight - margin - h);
            cx = clamp(cx, margin + w / 2, window.innerWidth - margin - w / 2);
        }
    }
    else {
        top = clamp(top, margin, window.innerHeight - margin - h);
        cx = clamp(cx, margin + w / 2, window.innerWidth - margin - w / 2);
    }
    recordToolbar.style.left = `${Math.round(cx)}px`;
    recordToolbar.style.top = `${Math.round(top)}px`;
    if (recordToolbarHotzone) {
        recordToolbarHotzone.style.left = `${Math.round(cx)}px`;
        recordToolbarHotzone.style.top = `${Math.round(top)}px`;
    }
}
function updateRecordToolbarAutoVisibility(pos) {
    if (!recordUiActive)
        return;
    ensureRecordElements();
    if (!recordSelectSession) {
        setRecordHotzoneVisible(false);
        setRecordToolbarVisible(false);
        return;
    }
    if (!pos || typeof pos !== 'object')
        return;
    if (!recordSelectSession.created)
        return;
    recordToolbarLastMousePos = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
    positionRecordToolbar();
    const tbVisible = !!(recordToolbar && recordToolbar.classList.contains('visible'));
    const hzVisible = !!(recordToolbarHotzone && recordToolbarHotzone.classList.contains('visible'));
    const inHotzone = hzVisible && recordToolbarHotzone ? pointInDomRect(recordToolbarLastMousePos, recordToolbarHotzone.getBoundingClientRect()) : false;
    const inTb = tbVisible && recordToolbar ? pointInDomRect(recordToolbarLastMousePos, recordToolbar.getBoundingClientRect()) : false;
    let inMenu = false;
    if (tbVisible) {
        if (recordSaveMenu && recordSaveMenu.classList.contains('visible') && pointInDomRect(recordToolbarLastMousePos, recordSaveMenu.getBoundingClientRect()))
            inMenu = true;
        if (recordMicMenu && recordMicMenu.classList.contains('visible') && pointInDomRect(recordToolbarLastMousePos, recordMicMenu.getBoundingClientRect()))
            inMenu = true;
    }
    if (!tbVisible) {
        if (!hzVisible)
            setRecordHotzoneVisible(true);
        if (inHotzone) {
            if (recordToolbarAutoHideTimer) {
                clearTimeout(recordToolbarAutoHideTimer);
                recordToolbarAutoHideTimer = null;
            }
            setRecordToolbarVisible(true);
            setRecordHotzoneVisible(false);
        }
        return;
    }
    if (inTb || inMenu) {
        if (recordToolbarAutoHideTimer) {
            clearTimeout(recordToolbarAutoHideTimer);
            recordToolbarAutoHideTimer = null;
        }
        return;
    }
    if (recordToolbarAutoHideTimer)
        return;
    recordToolbarAutoHideTimer = setTimeout(() => {
        recordToolbarAutoHideTimer = null;
        if (!recordUiActive)
            return;
        ensureRecordElements();
        if (!recordSelectSession || !recordSelectSession.created)
            return;
        positionRecordToolbar();
        const p = recordToolbarLastMousePos;
        const stillInTb = recordToolbar && recordToolbar.classList.contains('visible') ? pointInDomRect(p, recordToolbar.getBoundingClientRect()) : false;
        let stillInMenu = false;
        if (recordSaveMenu && recordSaveMenu.classList.contains('visible') && pointInDomRect(p, recordSaveMenu.getBoundingClientRect()))
            stillInMenu = true;
        if (recordMicMenu && recordMicMenu.classList.contains('visible') && pointInDomRect(p, recordMicMenu.getBoundingClientRect()))
            stillInMenu = true;
        if (stillInTb || stillInMenu)
            return;
        setRecordToolbarVisible(false);
        setRecordHotzoneVisible(true);
    }, 260);
}
function setRecordToggleButtonVisual(btn, state) {
    if (!btn)
        return;
    if (state === 'recording') {
        btn.title = '暂停';
        btn.setAttribute('aria-label', '暂停');
        btn.innerHTML = recordIconPause;
        return;
    }
    if (state === 'paused') {
        btn.title = '继续';
        btn.setAttribute('aria-label', '继续');
        btn.innerHTML = recordIconPlay;
        return;
    }
    btn.title = '开始';
    btn.setAttribute('aria-label', '开始');
    btn.innerHTML = recordIconPlay;
}
function ensureRecordAudioControls() {
    ensureRecordElements();
    if (!recordToolbar)
        return { recordSysAudioBtn: null, recordMicAudioBtn: null, recordMicSelectBtn: null };
    if (!recordSysAudioBtn || !recordSysAudioBtn.isConnected)
        recordSysAudioBtn = recordToolbar.querySelector('button[data-rec-audio="system"]');
    if (!recordMicAudioBtn || !recordMicAudioBtn.isConnected)
        recordMicAudioBtn = recordToolbar.querySelector('button[data-rec-audio="mic"]');
    if (!recordFullscreenBtn || !recordFullscreenBtn.isConnected)
        recordFullscreenBtn = recordToolbar.querySelector('button[data-rec-action="fullscreen"]');
    if (!recordMicSelectBtn || !recordMicSelectBtn.isConnected)
        recordMicSelectBtn = document.getElementById('record-mic-select-btn');
    if (!recordMicSelectLabel || !recordMicSelectLabel.isConnected)
        recordMicSelectLabel = document.getElementById('record-mic-select-label');
    return { recordSysAudioBtn, recordMicAudioBtn, recordFullscreenBtn, recordMicSelectBtn, recordMicSelectLabel };
}
function normalizeRecordMicDeviceId(id) {
    const s = String(id || '').trim();
    if (!s)
        return null;
    if (s === 'default' || s === 'communications')
        return null;
    return s;
}
function safeRecordMicLabel(dev, index) {
    const label = String(dev && dev.label ? dev.label : '').trim();
    if (label)
        return label;
    return `麦克风 ${index + 1}`;
}
function maybeRefreshRecordMicDevices(force) {
    const now = Date.now();
    const f = !!force;
    if (!f && now - recordMicDevicesLastRefreshTs < 3000)
        return;
    recordMicDevicesLastRefreshTs = now;
    try {
        refreshRecordMicDevices();
    }
    catch { }
}
function setRecordMicDeviceOptions(selectedId) {
    ensureRecordElements();
    const { recordMicSelectLabel, recordMicSelectBtn } = ensureRecordAudioControls();
    if (!recordMicMenu)
        return;
    const nextSelected = String(selectedId || '').trim();
    try {
        recordMicMenu.innerHTML = '';
    }
    catch { }
    let foundLabel = '默认麦克风';
    const createBtn = (id, text) => {
        const b = document.createElement('button');
        b.dataset.micId = id;
        b.textContent = text;
        b.title = text;
        if (id === nextSelected) {
            b.classList.add('active');
            foundLabel = text;
        }
        return b;
    };
    recordMicMenu.appendChild(createBtn('', '默认麦克风'));
    const devs = Array.isArray(cachedRecordMicDevices) ? cachedRecordMicDevices : [];
    for (let i = 0; i < devs.length; i++) {
        const d = devs[i];
        const id = String(d && d.deviceId ? d.deviceId : '').trim();
        if (!id)
            continue;
        recordMicMenu.appendChild(createBtn(id, safeRecordMicLabel(d, i)));
    }
    if (recordMicSelectLabel)
        recordMicSelectLabel.textContent = foundLabel;
    if (recordMicSelectBtn)
        recordMicSelectBtn.title = foundLabel;
}
async function refreshRecordMicDevices() {
    const seq = ++recordMicDevicesSeq;
    if (!navigator.mediaDevices || typeof navigator.mediaDevices.enumerateDevices !== 'function') {
        cachedRecordMicDevices = [];
        setRecordMicDeviceOptions(currentSettings && currentSettings.recordMicDeviceId ? currentSettings.recordMicDeviceId : '');
        return;
    }
    let list = [];
    try {
        list = await navigator.mediaDevices.enumerateDevices();
    }
    catch {
        list = [];
    }
    if (seq !== recordMicDevicesSeq)
        return;
    const mics = (Array.isArray(list) ? list : [])
        .filter(d => d && d.kind === 'audioinput')
        .filter(d => {
        const id = String(d && d.deviceId ? d.deviceId : '').trim();
        return !!id && id !== 'default' && id !== 'communications';
    });
    const seen = new Set();
    cachedRecordMicDevices = mics.filter(d => {
        const id = String(d && d.deviceId ? d.deviceId : '').trim();
        if (!id || seen.has(id))
            return false;
        seen.add(id);
        return true;
    });
    setRecordMicDeviceOptions(currentSettings && currentSettings.recordMicDeviceId ? currentSettings.recordMicDeviceId : '');
}
function warmRecordDesktopSourceId(force) {
    if (!window.tt || typeof window.tt.getDesktopSourceId !== 'function')
        return null;
    const f = !!force;
    const now = Date.now();
    const cache = recordDesktopSourceCache && typeof recordDesktopSourceCache === 'object' ? recordDesktopSourceCache : { displayId: '', sourceId: '', ts: 0 };
    if (!f && cache.sourceId && cache.displayId === displayId && (now - (Number(cache.ts) || 0)) < 15000) {
        return Promise.resolve({ sourceId: cache.sourceId, error: '' });
    }
    if (recordDesktopSourceLoading)
        return recordDesktopSourceLoading;
    recordDesktopSourceLoading = (async () => {
        const res = await window.tt.getDesktopSourceId({ displayId });
        const sid = res && res.sourceId ? String(res.sourceId) : '';
        if (sid)
            recordDesktopSourceCache = { displayId, sourceId: sid, ts: Date.now() };
        recordDesktopSourceLoading = null;
        return res;
    })().catch((e) => {
        recordDesktopSourceLoading = null;
        return { sourceId: '', error: e && e.message ? String(e.message) : 'desktop_source_failed' };
    });
    return recordDesktopSourceLoading;
}
function updateRecordAudioControls() {
    const { recordSysAudioBtn: sysBtn, recordMicAudioBtn: micBtn, recordFullscreenBtn: fsBtn, recordMicSelectBtn: selBtn } = ensureRecordAudioControls();
    const sysOn = !!(currentSettings && currentSettings.recordSystemAudio);
    const micOn = !!(currentSettings && currentSettings.recordMicAudio);
    const editable = recordState === 'idle';
    const fullscreen = isRecordSelectFullscreen();
    if (sysBtn) {
        try {
            sysBtn.classList.toggle('active', sysOn);
        }
        catch { }
        try {
            sysBtn.setAttribute('aria-pressed', sysOn ? 'true' : 'false');
        }
        catch { }
        try {
            sysBtn.disabled = !editable;
        }
        catch { }
    }
    if (micBtn) {
        try {
            micBtn.classList.toggle('active', micOn);
        }
        catch { }
        try {
            micBtn.setAttribute('aria-pressed', micOn ? 'true' : 'false');
        }
        catch { }
        try {
            micBtn.disabled = !editable;
        }
        catch { }
    }
    if (fsBtn) {
        try {
            fsBtn.classList.toggle('active', fullscreen);
        }
        catch { }
        try {
            fsBtn.setAttribute('aria-pressed', fullscreen ? 'true' : 'false');
        }
        catch { }
        try {
            fsBtn.disabled = !editable;
        }
        catch { }
        try {
            fsBtn.title = fullscreen ? '退出全屏' : '全屏';
        }
        catch { }
        try {
            fsBtn.setAttribute('aria-label', fullscreen ? '退出全屏' : '全屏');
        }
        catch { }
    }
    if (selBtn) {
        try {
            selBtn.disabled = !editable || !micOn;
        }
        catch { }
        try {
            selBtn.style.opacity = (!editable || !micOn) ? '0.5' : '1';
        }
        catch { }
        try {
            selBtn.style.pointerEvents = (!editable || !micOn) ? 'none' : 'auto';
        }
        catch { }
        const wantId = String(currentSettings && currentSettings.recordMicDeviceId ? currentSettings.recordMicDeviceId : '').trim();
        setRecordMicDeviceOptions(wantId);
    }
}
function wireRecordToolbar() {
    if (recordToolbarWired)
        return;
    ensureRecordElements();
    if (!recordToolbar)
        return;
    recordToolbarWired = true;
    recordToolbar.addEventListener('click', (e) => {
        const micSelBtn = e.target.closest('#record-mic-select-btn');
        if (micSelBtn) {
            if (recordState !== 'idle') {
                bubble.showText('录制中无法修改音频');
                return;
            }
            if (micSelBtn.disabled)
                return;
            const willOpen = !(recordMicMenu && recordMicMenu.classList.contains('visible'));
            try {
                if (recordMicMenu)
                    recordMicMenu.classList.toggle('visible');
            }
            catch { }
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.remove('visible');
            }
            catch { }
            if (willOpen)
                maybeRefreshRecordMicDevices(false);
            syncRecordInteractiveRects();
            return;
        }
        const btn = e && e.target && e.target.closest ? e.target.closest('button[data-rec-action],button[data-rec-save],button[data-rec-audio]') : null;
        if (!btn)
            return;
        const action = btn.dataset && btn.dataset.recAction ? String(btn.dataset.recAction) : '';
        const saveFmt = btn.dataset && btn.dataset.recSave ? String(btn.dataset.recSave) : '';
        const audioKind = btn.dataset && btn.dataset.recAudio ? String(btn.dataset.recAudio) : '';
        if (audioKind) {
            if (recordState !== 'idle') {
                bubble.showText('录制中无法修改音频');
                return;
            }
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.remove('visible');
            }
            catch { }
            try {
                if (recordMicMenu)
                    recordMicMenu.classList.remove('visible');
            }
            catch { }
            syncRecordInteractiveRects();
            if (audioKind === 'system') {
                const next = !(currentSettings && currentSettings.recordSystemAudio);
                scheduleSettingsPatch({ recordSystemAudio: next });
                updateRecordAudioControls();
                return;
            }
            if (audioKind === 'mic') {
                const next = !(currentSettings && currentSettings.recordMicAudio);
                scheduleSettingsPatch({ recordMicAudio: next });
                if (next)
                    maybeRefreshRecordMicDevices(true);
                updateRecordAudioControls();
                return;
            }
            return;
        }
        if (saveFmt) {
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.remove('visible');
            }
            catch { }
            syncRecordInteractiveRects();
            saveRecordAs(saveFmt);
            return;
        }
        if (action === 'fullscreen') {
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.remove('visible');
            }
            catch { }
            try {
                if (recordMicMenu)
                    recordMicMenu.classList.remove('visible');
            }
            catch { }
            syncRecordInteractiveRects();
            setRecordSelectFullscreen();
            return;
        }
        if (action === 'toggle') {
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.remove('visible');
            }
            catch { }
            try {
                if (recordMicMenu)
                    recordMicMenu.classList.remove('visible');
            }
            catch { }
            syncRecordInteractiveRects();
            toggleRecordStartPause();
            return;
        }
        if (action === 'save') {
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.toggle('visible');
            }
            catch { }
            try {
                if (recordMicMenu)
                    recordMicMenu.classList.remove('visible');
            }
            catch { }
            syncRecordInteractiveRects();
            return;
        }
        if (action === 'cancel') {
            try {
                if (recordSaveMenu)
                    recordSaveMenu.classList.remove('visible');
            }
            catch { }
            try {
                if (recordMicMenu)
                    recordMicMenu.classList.remove('visible');
            }
            catch { }
            syncRecordInteractiveRects();
            cancelRecordUi();
            return;
        }
    });
    if (recordMicMenu) {
        recordMicMenu.addEventListener('click', (e) => {
            const t = e && e.target ? e.target : null;
            const b = t && t.closest ? t.closest('button[data-mic-id]') : null;
            if (!b)
                return;
            const id = b.dataset.micId;
            recordMicMenu.classList.remove('visible');
            syncRecordInteractiveRects();
            if (recordState !== 'idle') {
                bubble.showText('录制中无法修改音频');
                return;
            }
            const nextId = normalizeRecordMicDeviceId(id);
            scheduleSettingsPatch({ recordMicDeviceId: nextId });
            updateRecordAudioControls();
        });
    }
    document.addEventListener('mousedown', (e) => {
        const t = e && e.target ? e.target : null;
        const inTb = recordToolbar && t && t.closest ? t.closest('#record-toolbar') : null;
        if (inTb)
            return;
        let changed = false;
        if (recordSaveMenu && recordSaveMenu.classList.contains('visible')) {
            recordSaveMenu.classList.remove('visible');
            changed = true;
        }
        if (recordMicMenu && recordMicMenu.classList.contains('visible')) {
            recordMicMenu.classList.remove('visible');
            changed = true;
        }
        if (changed)
            syncRecordInteractiveRects();
    });
    updateRecordAudioControls();
    if (currentSettings && currentSettings.recordMicAudio)
        maybeRefreshRecordMicDevices(false);
}
function syncRecordInteractiveRects() {
    ensureRecordElements();
    const recordRects = [];
    const pushEl = (el) => {
        if (!el || !el.isConnected)
            return;
        const b = el.getBoundingClientRect();
        const w = Math.max(0, Math.round(Number(b.width) || 0));
        const h = Math.max(0, Math.round(Number(b.height) || 0));
        if (w < 2 || h < 2)
            return;
        const x = Math.max(0, Math.round(Number(b.left) || 0));
        const y = Math.max(0, Math.round(Number(b.top) || 0));
        recordRects.push({ x, y, w, h });
    };
    if (recordUiActive) {
        if (recordToolbar && recordToolbar.classList.contains('visible')) {
            pushEl(recordToolbar);
            if (recordSaveMenu && recordSaveMenu.classList.contains('visible'))
                pushEl(recordSaveMenu);
            if (recordMicMenu && recordMicMenu.classList.contains('visible'))
                pushEl(recordMicMenu);
        }
        else if (recordToolbarHotzone && recordToolbarHotzone.classList.contains('visible')) {
            pushEl(recordToolbarHotzone);
        }
    }
    recordUiRects = recordRects;
    if (!window.tt || typeof window.tt.setInteractiveRects !== 'function')
        return;
    const rects = recordRects.slice();
    for (const v of refImageViews.values()) {
        const wrap = v && v.wrap ? v.wrap : null;
        if (!wrap || !wrap.isConnected)
            continue;
        const b = wrap.getBoundingClientRect();
        const w = Math.max(0, Math.round(Number(b.width) || 0));
        const h = Math.max(0, Math.round(Number(b.height) || 0));
        if (w < 2 || h < 2)
            continue;
        const x = Math.max(0, Math.round(Number(b.left) || 0));
        const y = Math.max(0, Math.round(Number(b.top) || 0));
        rects.push({ x, y, w, h });
    }
    try {
        window.tt.setInteractiveRects(rects);
    }
    catch { }
}
function syncRecordPickMode() {
    if (!window.tt || typeof window.tt.setRecordPickMode !== 'function')
        return;
    if (!recordSelectActive)
        return;
    const capture = recordState === 'idle' || recordState === 'stopped';
    try {
        window.tt.setRecordPickMode(true, { displayId, capture });
    }
    catch { }
}
function recordUiHit(pos) {
    const p = (pos && typeof pos === 'object') ? pos : null;
    if (!p || !recordUiRects || !recordUiRects.length)
        return false;
    const x = Math.round(Number(p.x) || 0);
    const y = Math.round(Number(p.y) || 0);
    return recordUiRects.some((r) => x >= r.x && x <= (r.x + r.w) && y >= r.y && y <= (r.y + r.h));
}
function setRecordBoxRect(rect) {
    ensureRecordElements();
    const r = rect && typeof rect === 'object' ? rect : { x: 0, y: 0, w: 0, h: 0 };
    const x = Math.max(0, Math.round(Number(r.x) || 0));
    const y = Math.max(0, Math.round(Number(r.y) || 0));
    const w = Math.max(0, Math.round(Number(r.w) || 0));
    const h = Math.max(0, Math.round(Number(r.h) || 0));
    if (recordBox) {
        recordBox.style.left = `${x}px`;
        recordBox.style.top = `${y}px`;
        recordBox.style.width = `${w}px`;
        recordBox.style.height = `${h}px`;
    }
    if (recordTitle)
        recordTitle.textContent = w > 0 && h > 0 ? `录屏区域 ${w} × ${h}` : '录屏区域';
    positionRecordToolbar();
    updateRecordAudioControls();
}
function isRecordSelectFullscreen() {
    if (!recordSelectSession || !recordSelectSession.created)
        return false;
    const s = recordSelectSession;
    const r = s && s.rect ? s.rect : null;
    if (!r)
        return false;
    const vw = Math.max(1, Math.round(Number(s.vw) || window.innerWidth || 1));
    const vh = Math.max(1, Math.round(Number(s.vh) || window.innerHeight || 1));
    return (Math.round(Number(r.x) || 0) === 0)
        && (Math.round(Number(r.y) || 0) === 0)
        && (Math.round(Number(r.w) || 0) >= vw)
        && (Math.round(Number(r.h) || 0) >= vh);
}
function setRecordSelectFullscreen() {
    if (recordState !== 'idle') {
        bubble.showText('录制中无法修改区域');
        return;
    }
    if (!recordSelectActive || !recordSelectSession)
        startRecordRegionPick();
    if (!recordSelectSession)
        return;
    const s = recordSelectSession;
    const vw = Math.max(1, Math.round(Number(s.vw) || window.innerWidth || 1));
    const vh = Math.max(1, Math.round(Number(s.vh) || window.innerHeight || 1));
    if (isRecordSelectFullscreen()) {
        const prev = recordPrevRectBeforeFullscreen && typeof recordPrevRectBeforeFullscreen === 'object' ? recordPrevRectBeforeFullscreen : null;
        const next = prev ? normalizeRecordRect(prev) : normalizeRecordRect({ x: 0, y: 0, w: Math.round(vw * 0.6), h: Math.round(vh * 0.6) });
        recordPrevRectBeforeFullscreen = null;
        s.rect = next;
    }
    else {
        const cur = s.rect && typeof s.rect === 'object' ? s.rect : null;
        if (cur)
            recordPrevRectBeforeFullscreen = { x: cur.x, y: cur.y, w: cur.w, h: cur.h };
        const next = normalizeRecordRect({ x: 0, y: 0, w: vw, h: vh });
        s.rect = next;
    }
    s.created = true;
    s.start = null;
    s.dragging = false;
    s.mode = '';
    s.anchorRect = null;
    s.resizeDir = '';
    s.moveOffset = null;
    setRecordBoxRect(s.rect);
    setRecordHotzoneVisible(false);
    setRecordToolbarVisible(true);
    syncRecordInteractiveRects();
}
function setRecordToolbarVisible(visible) {
    ensureRecordElements();
    if (recordToolbar) {
        try {
            recordToolbar.classList.toggle('visible', !!visible);
        }
        catch { }
    }
    if (!visible && recordSaveMenu) {
        try {
            recordSaveMenu.classList.remove('visible');
        }
        catch { }
    }
    if (!visible && recordToolbarAutoHideTimer) {
        try {
            clearTimeout(recordToolbarAutoHideTimer);
        }
        catch { }
        recordToolbarAutoHideTimer = null;
    }
    positionRecordToolbar();
    syncRecordInteractiveRects();
    syncRecordPickMode();
}
function updateRecordToggleButton() {
    ensureRecordElements();
    if (!recordToolbar)
        return;
    const btn = recordToolbar.querySelector('button[data-rec-action="toggle"]');
    if (!btn)
        return;
    setRecordToggleButtonVisual(btn, recordState);
    try {
        btn.classList.toggle('active', recordState === 'recording' || recordState === 'paused');
    }
    catch { }
    try {
        if (recordBox)
            recordBox.classList.toggle('locked', recordState === 'recording' || recordState === 'paused');
    }
    catch { }
    updateRecordAudioControls();
    syncRecordPickMode();
}
function normalizeRecordRect(rect0) {
    if (!recordSelectSession)
        return { x: 0, y: 0, w: 0, h: 0 };
    const r = rect0 && typeof rect0 === 'object' ? rect0 : { x: 0, y: 0, w: 0, h: 0 };
    const vw = Math.max(1, Number(recordSelectSession.vw) || 1);
    const vh = Math.max(1, Number(recordSelectSession.vh) || 1);
    const x = Math.max(0, Math.min(vw, Math.round(Number(r.x) || 0)));
    const y = Math.max(0, Math.min(vh, Math.round(Number(r.y) || 0)));
    const w = Math.max(0, Math.min(vw - x, Math.round(Number(r.w) || 0)));
    const h = Math.max(0, Math.min(vh - y, Math.round(Number(r.h) || 0)));
    return { x, y, w, h };
}
function closeRecordUi() {
    recordUiActive = false;
    recordSelectActive = false;
    if (recordToolbarAutoHideTimer) {
        try {
            clearTimeout(recordToolbarAutoHideTimer);
        }
        catch { }
        recordToolbarAutoHideTimer = null;
    }
    recordToolbarLastMousePos = null;
    const prevUserSelect = recordSelectSession && typeof recordSelectSession.prevUserSelect === 'string' ? recordSelectSession.prevUserSelect : '';
    recordSelectSession = null;
    ensureRecordElements();
    if (recordLayer)
        recordLayer.classList.remove('active');
    setRecordBoxRect({ x: 0, y: 0, w: 0, h: 0 });
    setRecordHotzoneVisible(false);
    setRecordToolbarVisible(false);
    recordState = 'idle';
    updateRecordToggleButton();
    try {
        if (window.tt && typeof window.tt.setRecordPickMode === 'function') {
            window.tt.setRecordPickMode(false, { displayId });
        }
    }
    catch { }
    try {
        document.body.style.userSelect = prevUserSelect;
    }
    catch { }
}
function cancelRecordUi() {
    recordSelectCancel = null;
    const p = stopRecordStream();
    if (p && typeof p.finally === 'function')
        p.finally(() => closeRecordUi());
    else
        closeRecordUi();
}
async function stopRecordStream() {
    const r = recordRecorder;
    if (r && r.state !== 'inactive') {
        await new Promise((resolve) => {
            const done = () => resolve(true);
            try {
                r.addEventListener('stop', done, { once: true });
                r.stop();
            }
            catch {
                resolve(true);
            }
        });
    }
    if (recordStream) {
        try {
            recordStream.getTracks().forEach(t => { try {
                t.stop();
            }
            catch { } });
        }
        catch { }
    }
    if (recordSourceStreams && recordSourceStreams.length) {
        try {
            for (const s of recordSourceStreams) {
                if (!s || s === recordStream)
                    continue;
                try {
                    s.getTracks().forEach(t => { try {
                        t.stop();
                    }
                    catch { } });
                }
                catch { }
            }
        }
        catch { }
    }
    recordSourceStreams = [];
    if (recordAudioContext) {
        try {
            await recordAudioContext.close();
        }
        catch { }
    }
    recordAudioContext = null;
    recordStream = null;
    recordRecorder = null;
}
function bytesFromBlob(blob) {
    return new Promise((resolve) => {
        if (!blob) {
            resolve(new Uint8Array());
            return;
        }
        try {
            const fr = new FileReader();
            fr.onload = () => {
                const ab = fr.result instanceof ArrayBuffer ? fr.result : null;
                resolve(ab ? new Uint8Array(ab) : new Uint8Array());
            };
            fr.onerror = () => resolve(new Uint8Array());
            fr.readAsArrayBuffer(blob);
        }
        catch {
            resolve(new Uint8Array());
        }
    });
}
async function ensureRecordWebm() {
    if (recordLastWebm && recordLastWebm.length)
        return { data: recordLastWebm, info: recordLastVideoInfo };
    if (!recordChunks || !recordChunks.length)
        return { data: new Uint8Array(), info: null };
    const blob = new Blob(recordChunks, { type: (recordRecorder && recordRecorder.mimeType) ? recordRecorder.mimeType : 'video/webm' });
    const data = await bytesFromBlob(blob);
    recordLastWebm = data;
    recordChunks = [];
    const v = document.createElement('video');
    v.muted = true;
    v.playsInline = true;
    v.src = URL.createObjectURL(blob);
    try {
        await new Promise((resolve, reject) => {
            v.onloadedmetadata = resolve;
            v.onerror = reject;
        });
    }
    catch { }
    const info = { width: Math.max(1, Math.round(Number(v.videoWidth) || 1)), height: Math.max(1, Math.round(Number(v.videoHeight) || 1)) };
    recordLastVideoInfo = info;
    try {
        URL.revokeObjectURL(v.src);
    }
    catch { }
    return { data, info };
}
async function ensureFfmpeg() {
    if (recordFfmpeg && recordFfmpeg.loaded)
        return recordFfmpeg;
    if (recordFfmpegLoading)
        return recordFfmpegLoading;
    recordFfmpegLoading = (async () => {
        let FFmpegCtor = null;
        try {
            const modUrl = new URL('../../node_modules/@ffmpeg/ffmpeg/dist/esm/index.js', import.meta.url).toString();
            const mod = await import(modUrl);
            FFmpegCtor = mod && (mod.FFmpeg || (mod.default && mod.default.FFmpeg));
        }
        catch { }
        if (!FFmpegCtor) {
            recordFfmpegLoading = null;
            const err = new Error('ffmpeg_missing');
            err.code = 'FFMPEG_MISSING';
            throw err;
        }
        const ff = new FFmpegCtor();
        const coreURL = new URL('../../node_modules/@ffmpeg/core/dist/esm/ffmpeg-core.js', import.meta.url).toString();
        const wasmURL = new URL('../../node_modules/@ffmpeg/core/dist/esm/ffmpeg-core.wasm', import.meta.url).toString();
        await ff.load({ coreURL, wasmURL });
        recordFfmpeg = ff;
        recordFfmpegLoading = null;
        return ff;
    })();
    return recordFfmpegLoading;
}
function computeCropForVideo(videoInfo, region) {
    const info = videoInfo && typeof videoInfo === 'object' ? videoInfo : { width: 1, height: 1 };
    const vw = Math.max(1, Number(recordSelectSession && recordSelectSession.vw) || 1);
    const vh = Math.max(1, Number(recordSelectSession && recordSelectSession.vh) || 1);
    const iw = Math.max(1, Number(info.width) || 1);
    const ih = Math.max(1, Number(info.height) || 1);
    const r = normalizeRecordRect(region);
    const scaleX = iw / vw;
    const scaleY = ih / vh;
    let x = Math.max(0, Math.round((Number(r.x) || 0) * scaleX));
    let y = Math.max(0, Math.round((Number(r.y) || 0) * scaleY));
    let w = Math.max(2, Math.round((Number(r.w) || 0) * scaleX));
    let h = Math.max(2, Math.round((Number(r.h) || 0) * scaleY));
    if (x + w > iw)
        w = Math.max(2, iw - x);
    if (y + h > ih)
        h = Math.max(2, ih - y);
    w = Math.max(2, w - (w % 2));
    h = Math.max(2, h - (h % 2));
    return { x, y, w, h };
}
async function transcodeRecordToMp4(webmBytes, videoInfo, region) {
    const ff = await ensureFfmpeg();
    const crop = computeCropForVideo(videoInfo, region);
    await ff.writeFile('in.webm', webmBytes);
    try {
        await ff.exec(['-i', 'in.webm', '-vf', `crop=${crop.w}:${crop.h}:${crop.x}:${crop.y}`, '-c:v', 'libx264', '-pix_fmt', 'yuv420p', 'out.mp4']);
    }
    catch {
        await ff.exec(['-i', 'in.webm', '-vf', `crop=${crop.w}:${crop.h}:${crop.x}:${crop.y}`, '-c:v', 'mpeg4', 'out.mp4']);
    }
    const out = await ff.readFile('out.mp4');
    try {
        await ff.deleteFile('in.webm');
    }
    catch { }
    try {
        await ff.deleteFile('out.mp4');
    }
    catch { }
    const u8 = out instanceof Uint8Array ? out : new Uint8Array(out);
    return u8;
}
async function transcodeRecordToGif(webmBytes, videoInfo, region) {
    const ff = await ensureFfmpeg();
    const crop = computeCropForVideo(videoInfo, region);
    const maxW = 800;
    const scaleW = crop.w > maxW ? maxW : crop.w;
    const scaleH = crop.w > maxW ? Math.max(2, Math.round((crop.h * scaleW) / crop.w)) : crop.h;
    await ff.writeFile('in.webm', webmBytes);
    await ff.exec(['-i', 'in.webm', '-vf', `crop=${crop.w}:${crop.h}:${crop.x}:${crop.y},fps=12,scale=${scaleW}:${scaleH}:flags=lanczos,palettegen`, 'palette.png']);
    await ff.exec(['-i', 'in.webm', '-i', 'palette.png', '-lavfi', `crop=${crop.w}:${crop.h}:${crop.x}:${crop.y},fps=12,scale=${scaleW}:${scaleH}:flags=lanczos[x];[x][1:v]paletteuse`, 'out.gif']);
    const out = await ff.readFile('out.gif');
    try {
        await ff.deleteFile('in.webm');
    }
    catch { }
    try {
        await ff.deleteFile('palette.png');
    }
    catch { }
    try {
        await ff.deleteFile('out.gif');
    }
    catch { }
    const u8 = out instanceof Uint8Array ? out : new Uint8Array(out);
    return u8;
}
async function startRecordCapture() {
    if (!recordSelectSession || !recordSelectSession.created) {
        bubble.showText('请先拖拽选择录屏区域');
        return false;
    }
    if (!window.tt || typeof window.tt.getDesktopSourceId !== 'function') {
        bubble.showText('桌面版才支持录屏');
        return false;
    }
    bubble.showText('启动录屏...');
    let res0 = null;
    try {
        res0 = await warmRecordDesktopSourceId(false);
    }
    catch {
        res0 = null;
    }
    if (!res0)
        res0 = { sourceId: '', error: 'desktopCapturer_unavailable' };
    let sourceId = res0 && res0.sourceId ? String(res0.sourceId) : '';
    let error = res0 && res0.error ? String(res0.error) : '';
    if (!sourceId) {
        let res1 = null;
        try {
            res1 = await warmRecordDesktopSourceId(true);
        }
        catch {
            res1 = null;
        }
        if (res1 && res1.sourceId)
            sourceId = String(res1.sourceId);
        if (res1 && res1.error)
            error = String(res1.error);
    }
    if (!sourceId) {
        bubble.showToast({ type: 'error', message: '录屏失败', details: normalizeScreenshotError(error) });
        return false;
    }
    const wantSystemAudio = !!(currentSettings && currentSettings.recordSystemAudio);
    const wantMicAudio = !!(currentSettings && currentSettings.recordMicAudio);
    const micDeviceId = String(currentSettings && currentSettings.recordMicDeviceId ? currentSettings.recordMicDeviceId : '').trim();
    const videoMandatory = { chromeMediaSource: 'desktop', chromeMediaSourceId: sourceId };
    const getScreenStream = async (withAudio) => {
        const constraints = { video: { mandatory: videoMandatory }, audio: false };
        if (withAudio)
            constraints.audio = { mandatory: videoMandatory };
        let s = null;
        let err = null;
        try {
            s = await navigator.mediaDevices.getUserMedia(constraints);
        }
        catch (e) {
            s = null;
            err = e;
        }
        return { s, err };
    };
    let screenStream = null;
    let screenErr = null;
    const r1 = await getScreenStream(wantSystemAudio);
    screenStream = r1.s;
    screenErr = r1.err;
    if (!screenStream && wantSystemAudio) {
        const r2 = await getScreenStream(false);
        screenStream = r2.s;
        screenErr = r2.err || screenErr;
    }
    if (!screenStream) {
        const details = screenErr
            ? `${String(screenErr.name || 'getUserMedia')}${screenErr && screenErr.message ? `: ${String(screenErr.message)}` : ''}`
            : '';
        bubble.showToast({ type: 'error', message: '录屏失败', details });
        return false;
    }
    let micStream = null;
    if (wantMicAudio) {
        const audio = { echoCancellation: true, noiseSuppression: true, autoGainControl: true };
        if (micDeviceId)
            audio.deviceId = { exact: micDeviceId };
        try {
            micStream = await navigator.mediaDevices.getUserMedia({ audio, video: false });
        }
        catch (e) {
            micStream = null;
            const details = e ? `${String(e.name || 'getUserMedia')}${e && e.message ? `: ${String(e.message)}` : ''}` : '';
            bubble.showToast({ type: 'warn', message: '麦克风不可用，将仅录制屏幕', details });
        }
    }
    let stream = screenStream;
    recordSourceStreams = [screenStream].concat(micStream ? [micStream] : []);
    recordAudioContext = null;
    const sysTracks = screenStream.getAudioTracks ? screenStream.getAudioTracks() : [];
    const micTracks = micStream && micStream.getAudioTracks ? micStream.getAudioTracks() : [];
    const sysHas = !!(sysTracks && sysTracks.length);
    const micHas = !!(micTracks && micTracks.length);
    if (micHas && sysHas) {
        try {
            const Ctor = window.AudioContext || window.webkitAudioContext;
            if (Ctor) {
                const ac = new Ctor();
                recordAudioContext = ac;
                const dest = ac.createMediaStreamDestination();
                const connect = (s) => {
                    const src = ac.createMediaStreamSource(s);
                    src.connect(dest);
                };
                connect(new MediaStream(sysTracks));
                connect(micStream);
                const mixed = dest.stream.getAudioTracks()[0];
                const vTracks = screenStream.getVideoTracks ? screenStream.getVideoTracks() : [];
                stream = mixed && vTracks && vTracks.length ? new MediaStream([vTracks[0], mixed]) : new MediaStream([].concat(vTracks, sysTracks, micTracks));
            }
            else {
                stream = new MediaStream([].concat(screenStream.getVideoTracks(), sysTracks, micTracks));
            }
        }
        catch {
            stream = new MediaStream([].concat(screenStream.getVideoTracks(), sysTracks, micTracks));
        }
    }
    else if (micHas && !sysHas) {
        stream = new MediaStream([].concat(screenStream.getVideoTracks(), micTracks));
    }
    const types = ['video/webm;codecs=vp9', 'video/webm;codecs=vp8', 'video/webm'];
    let mimeType = '';
    for (const t of types) {
        if (window.MediaRecorder && typeof MediaRecorder.isTypeSupported === 'function' && MediaRecorder.isTypeSupported(t)) {
            mimeType = t;
            break;
        }
    }
    recordChunks = [];
    recordLastWebm = null;
    recordLastVideoInfo = null;
    try {
        if (recordSelectSession && !recordSelectSession.recordStartTs)
            recordSelectSession.recordStartTs = Date.now();
        if (recordSelectSession)
            recordSelectSession.recordStopTs = 0;
    }
    catch { }
    recordStream = stream;
    const rec = mimeType ? new MediaRecorder(stream, { mimeType }) : new MediaRecorder(stream);
    recordRecorder = rec;
    rec.addEventListener('dataavailable', (e) => {
        try {
            if (e && e.data && e.data.size)
                recordChunks.push(e.data);
        }
        catch { }
    });
    rec.addEventListener('error', (e) => {
        const err = e && e.error ? e.error : e;
        const details = err ? `${String(err.name || 'MediaRecorder')}${err && err.message ? `: ${String(err.message)}` : ''}` : '';
        bubble.showToast({ type: 'error', message: '录屏出错', details });
    });
    try {
        rec.start(200);
    }
    catch {
        try {
            rec.start();
        }
        catch { }
    }
    if (rec.state !== 'recording') {
        bubble.showToast({ type: 'error', message: '录屏启动失败', details: rec.state ? `state=${String(rec.state)}` : '' });
        try {
            await stopRecordStream();
        }
        catch { }
        return false;
    }
    recordState = 'recording';
    updateRecordToggleButton();
    bubble.showText('录屏中...');
    return true;
}
async function toggleRecordStartPause() {
    if (!recordRecorder || recordRecorder.state === 'inactive') {
        await startRecordCapture();
        return;
    }
    if (recordRecorder.state === 'recording') {
        try {
            recordRecorder.pause();
        }
        catch { }
        recordState = 'paused';
        updateRecordToggleButton();
        bubble.showText('已暂停');
        return;
    }
    if (recordRecorder.state === 'paused') {
        try {
            recordRecorder.resume();
        }
        catch { }
        recordState = 'recording';
        updateRecordToggleButton();
        bubble.showText('继续录屏');
    }
}
function showIpcToast({ action, res, successMessage }) {
    const a = String(action || '').trim();
    const r = (res && typeof res === 'object') ? res : {};
    if (r && r.canceled) {
        bubble.showToast({ type: 'warn', message: `已取消${a || '操作'}` });
        return true;
    }
    if (r && r.error) {
        const code = String(r.error || '').trim();
        let message = a ? `${a}失败` : '操作失败';
        if (code === 'ffmpeg_missing')
            message = '未找到 ffmpeg';
        else if (code === 'invalid_or_too_large')
            message = '数据过大或格式不正确';
        else if (code === 'invalid_format')
            message = '导出格式不支持';
        else if (code === 'empty_data')
            message = '没有可导出的数据';
        else if (code === 'ffmpeg_failed')
            message = '导出失败 (ffmpeg)';
        else if (code === 'write_input_failed')
            message = '写入临时文件失败';
        else if (code === 'write_output_failed')
            message = '写入输出文件失败';
        bubble.showToast({ type: 'error', message, details: r.details ? String(r.details) : code });
        return true;
    }
    if (successMessage)
        bubble.showToast({ type: 'success', message: String(successMessage) });
    return false;
}
function normalizeScreenshotError(code) {
    const c = String(code || '').trim();
    if (!c)
        return '';
    if (c === 'desktopCapturer_unavailable')
        return 'desktopCapturer 不可用';
    if (c === 'no_sources')
        return '未找到屏幕源';
    if (c === 'no_source_id')
        return '屏幕源 ID 获取失败';
    if (c === 'getSources_failed')
        return '屏幕源获取失败';
    if (c === 'empty_png')
        return '图片数据为空';
    if (c === 'decode_failed')
        return '图片解码失败';
    if (c === 'canvas_failed')
        return 'Canvas 初始化失败';
    if (c === 'crop_failed')
        return '裁剪失败';
    if (c === 'invalid_or_too_large')
        return '图片数据过大或格式不正确';
    if (c === 'nativeImage_failed')
        return '图片数据解析失败';
    if (c === 'clipboard_write_failed')
        return '写入剪贴板失败';
    if (c === 'clipboard_invoke_failed')
        return '剪贴板调用失败';
    return c;
}
async function finishRecordAndGetWebm() {
    try {
        if (recordSelectSession && !recordSelectSession.recordStopTs)
            recordSelectSession.recordStopTs = Date.now();
    }
    catch { }
    if (recordRecorder && recordRecorder.state !== 'inactive') {
        bubble.showText('结束录屏...');
        await stopRecordStream();
    }
    recordState = 'stopped';
    updateRecordToggleButton();
    const { data, info } = await ensureRecordWebm();
    return { data, info };
}
async function saveRecordAs(format) {
    const fmt = String(format || '').trim().toLowerCase();
    if (!fmt)
        return;
    if (!recordSelectSession || !recordSelectSession.created)
        return;
    const { data: webmBytes, info } = await finishRecordAndGetWebm();
    if (!webmBytes || !webmBytes.length) {
        bubble.showText('没有可保存的录屏');
        return;
    }
    if (!window.tt) {
        bubble.showText('桌面版才支持保存');
        return;
    }
    try {
        bubble.showText('处理中...');
        const ts = Number(recordSelectSession && (recordSelectSession.recordStopTs || recordSelectSession.recordStartTs)) || Date.now();
        if (fmt === 'mp4') {
            const crop = computeCropForVideo(info, recordSelectSession.rect);
            if (typeof window.tt.exportRecording === 'function') {
                const res = await window.tt.exportRecording(webmBytes, { fmt: 'mp4', crop, suggestedName: `TT_Record_${formatTimestamp(ts)}.mp4` });
                if (showIpcToast({ action: '导出', res, successMessage: '已保存录屏' }))
                    return;
                closeRecordUi();
                return;
            }
            if (typeof window.tt.saveRecording !== 'function') {
                bubble.showText('桌面版才支持保存');
                return;
            }
            const res = await window.tt.saveRecording(webmBytes, { ext: 'webm', suggestedName: `TT_Record_${formatTimestamp(ts)}.webm` });
            showIpcToast({ action: '保存', res, successMessage: '已保存录屏' });
            closeRecordUi();
            return;
        }
        if (fmt === 'gif') {
            const crop = computeCropForVideo(info, recordSelectSession.rect);
            const maxW = 800;
            const scaleW = crop.w > maxW ? maxW : crop.w;
            const scaleH = crop.w > maxW ? Math.max(2, Math.round((crop.h * scaleW) / crop.w)) : crop.h;
            if (typeof window.tt.exportRecording === 'function') {
                const res = await window.tt.exportRecording(webmBytes, { fmt: 'gif', crop, scale: { w: scaleW, h: scaleH }, fps: 12, suggestedName: `TT_Record_${formatTimestamp(ts)}.gif` });
                if (showIpcToast({ action: '导出', res, successMessage: '已保存录屏' }))
                    return;
                closeRecordUi();
                return;
            }
            if (typeof window.tt.saveRecording !== 'function') {
                bubble.showText('桌面版才支持保存');
                return;
            }
            const res = await window.tt.saveRecording(webmBytes, { ext: 'webm', suggestedName: `TT_Record_${formatTimestamp(ts)}.webm` });
            showIpcToast({ action: '保存', res, successMessage: '已保存录屏' });
            closeRecordUi();
        }
    }
    catch (e) {
        bubble.showToast({ type: 'error', message: '保存失败', details: e && e.message ? String(e.message) : '' });
    }
}
async function startRecordRegionPick() {
    ensureRecordElements();
    wireRecordToolbar();
    const layer = recordLayer;
    const vw = Math.max(1, Math.round(window.innerWidth || 1));
    const vh = Math.max(1, Math.round(window.innerHeight || 1));
    recordUiActive = true;
    try {
        if (window.tt && typeof window.tt.setRecordPickMode === 'function') {
            window.tt.setRecordPickMode(true, { displayId, capture: true });
        }
    }
    catch { }
    try {
        warmRecordDesktopSourceId(false);
    }
    catch { }
    recordSelectActive = true;
    recordSelectCancel = () => cancelRecordUi();
    if (layer)
        layer.classList.add('active');
    setRecordBoxRect({ x: 0, y: 0, w: 0, h: 0 });
    setRecordHotzoneVisible(false);
    setRecordToolbarVisible(false);
    const prevUserSelect = document.body.style.userSelect;
    document.body.style.userSelect = 'none';
    recordState = 'idle';
    updateRecordToggleButton();
    recordSelectSession = null;
    const normalize = (p1, p2) => {
        const x1 = Math.max(0, Math.min(vw, Math.round(Number(p1 && p1.x) || 0)));
        const y1 = Math.max(0, Math.min(vh, Math.round(Number(p1 && p1.y) || 0)));
        const x2 = Math.max(0, Math.min(vw, Math.round(Number(p2 && p2.x) || 0)));
        const y2 = Math.max(0, Math.min(vh, Math.round(Number(p2 && p2.y) || 0)));
        const x = Math.min(x1, x2);
        const y = Math.min(y1, y2);
        const w = Math.max(0, Math.abs(x2 - x1));
        const h = Math.max(0, Math.abs(y2 - y1));
        return { x, y, w, h };
    };
    const cursorForDir = (dir) => {
        switch (String(dir || '')) {
            case 'n':
            case 's': return 'ns-resize';
            case 'e':
            case 'w': return 'ew-resize';
            case 'ne':
            case 'sw': return 'nesw-resize';
            case 'nw':
            case 'se': return 'nwse-resize';
            default: return 'move';
        }
    };
    const edgeDirAt = (pos, rect) => {
        const p = (pos && typeof pos === 'object') ? pos : null;
        const r = (rect && typeof rect === 'object') ? rect : null;
        if (!p || !r)
            return '';
        const w = Math.max(0, Number(r.w) || 0);
        const h = Math.max(0, Number(r.h) || 0);
        if (w < 2 || h < 2)
            return '';
        const m = 10;
        const x = Number(p.x) - Number(r.x || 0);
        const y = Number(p.y) - Number(r.y || 0);
        const onW = x >= -m && x <= m;
        const onE = x >= w - m && x <= w + m;
        const onN = y >= -m && y <= m;
        const onS = y >= h - m && y <= h + m;
        let d = '';
        if (onN)
            d += 'n';
        else if (onS)
            d += 's';
        if (onW)
            d += 'w';
        else if (onE)
            d += 'e';
        return d;
    };
    const clampRect = (r0) => {
        const r = (r0 && typeof r0 === 'object') ? r0 : { x: 0, y: 0, w: 0, h: 0 };
        const x = Math.max(0, Math.min(vw, Math.round(Number(r.x) || 0)));
        const y = Math.max(0, Math.min(vh, Math.round(Number(r.y) || 0)));
        const w = Math.max(0, Math.min(vw - x, Math.round(Number(r.w) || 0)));
        const h = Math.max(0, Math.min(vh - y, Math.round(Number(r.h) || 0)));
        return { x, y, w, h };
    };
    recordSelectSession = {
        vw,
        vh,
        rect: { x: 0, y: 0, w: 0, h: 0 },
        start: null,
        dragging: false,
        mode: '',
        created: false,
        anchorRect: null,
        resizeDir: '',
        moveOffset: null,
        lastClickTs: 0,
        lastClickPos: null,
        normalize,
        edgeDirAt,
        cursorForDir,
        clampRect,
        prevUserSelect,
    };
    bubble.showText('拖拽选择录屏区域，右键取消');
}
async function cropPngBase64ByRegion(pngBase64, region) {
    const raw = String(pngBase64 || '').trim();
    if (!raw)
        return { pngBase64: '', error: 'empty_png' };
    const r = region && region.rect ? region.rect : null;
    const vp = region && region.viewport ? region.viewport : null;
    if (!r || !vp)
        return { pngBase64: raw, error: '' };
    const vw = Math.max(1, Number(vp.width) || 1);
    const vh = Math.max(1, Number(vp.height) || 1);
    const img = new Image();
    img.decoding = 'async';
    img.src = `data:image/png;base64,${raw}`;
    try {
        if (typeof img.decode === 'function')
            await img.decode();
        else
            await new Promise((resolve, reject) => { img.onload = resolve; img.onerror = reject; });
    }
    catch {
        return { pngBase64: '', error: 'decode_failed' };
    }
    const iw = Math.max(1, Number(img.naturalWidth) || 1);
    const ih = Math.max(1, Number(img.naturalHeight) || 1);
    const scaleX = iw / vw;
    const scaleY = ih / vh;
    const sx = Math.max(0, Math.min(iw - 1, Math.round((Number(r.x) || 0) * scaleX)));
    const sy = Math.max(0, Math.min(ih - 1, Math.round((Number(r.y) || 0) * scaleY)));
    const sw = Math.max(1, Math.min(iw - sx, Math.round((Number(r.w) || 0) * scaleX)));
    const sh = Math.max(1, Math.min(ih - sy, Math.round((Number(r.h) || 0) * scaleY)));
    const canvas = document.createElement('canvas');
    canvas.width = sw;
    canvas.height = sh;
    const ctx = canvas.getContext('2d', { alpha: true });
    if (!ctx)
        return { pngBase64: '', error: 'canvas_failed' };
    try {
        ctx.drawImage(img, sx, sy, sw, sh, 0, 0, sw, sh);
    }
    catch {
        return { pngBase64: '', error: 'crop_failed' };
    }
    let dataUrl = '';
    try {
        dataUrl = canvas.toDataURL('image/png');
    }
    catch {
        dataUrl = '';
    }
    const parts = String(dataUrl || '').split(',');
    const out = parts.length >= 2 ? String(parts[1] || '').trim() : '';
    return { pngBase64: out, error: out ? '' : 'empty_png' };
}
let shotEditorActive = false;
let shotEditorReady = false;
let shotEditorRoot = null;
let shotBackdrop = null;
let shotStage = null;
let shotToolbar = null;
let shotPinBtn = null;
let shotColorInput = null;
let shotSizeInput = null;
let shotTextSizeInput = null;
let shotEraserInput = null;
let shotBaseCanvas = null;
let shotAnnotCanvas = null;
let shotTempCanvas = null;
let shotBaseCtx = null;
let shotAnnotCtx = null;
let shotTempCtx = null;
let shotRect = null;
let shotCaptureTs = 0;
let shotTool = 'pen';
let shotIsDrawing = false;
let shotSx = 0;
let shotSy = 0;
let shotPathPoints = [];
let shotHistory = [];
let shotTextArea = null;
let shotTextBox = null;
let shotSelectedTextBox = null;
let shotTextAt = null;
let shotTextCaret = 0;
let shotTextResizing = false;
let shotMosaicTouched = null;
let shotColorPickerReady = false;
let pinnedShotIdsInWindow = [];
const pinnedShotViews = new Map();
let shotTextIdSeq = 0;
let shotTextFontSizeCss = 28;
function ensurePinnedShotView(id) {
    const key = String(id || '').trim();
    if (!key)
        return null;
    if (pinnedShotViews.has(key))
        return pinnedShotViews.get(key);
    const wrap = document.createElement('div');
    wrap.className = 'pinned-shot';
    wrap.dataset.pinnedShotId = key;
    const img = new Image();
    img.decoding = 'async';
    wrap.appendChild(img);
    wrap.addEventListener('mousedown', (e) => {
        if (!e || e.button !== 0)
            return;
        try {
            e.preventDefault();
        }
        catch { }
        const rr = wrap.getBoundingClientRect();
        const offsetX = Math.round(Number(e.clientX) - rr.left);
        const offsetY = Math.round(Number(e.clientY) - rr.top);
        try {
            if (window.tt && typeof window.tt.startPinnedShotDrag === 'function') {
                window.tt.startPinnedShotDrag({ id: key, offsetX, offsetY });
            }
        }
        catch { }
    });
    wrap.addEventListener('wheel', (e) => {
        if (!e)
            return;
        const delta = Number(e.deltaY) || 0;
        if (!Number.isFinite(delta) || delta === 0)
            return;
        try {
            if (window.tt && typeof window.tt.scalePinnedShot === 'function') {
                window.tt.scalePinnedShot({ id: key, deltaY: delta });
            }
        }
        catch { }
        try {
            e.preventDefault();
        }
        catch { }
    }, { passive: false });
    wrap.addEventListener('contextmenu', (e) => {
        try {
            e.preventDefault();
        }
        catch { }
        try {
            if (window.tt && typeof window.tt.removePinnedShot === 'function') {
                window.tt.removePinnedShot(key);
            }
        }
        catch { }
        bubble.showText('已关闭钉住');
    });
    document.body.appendChild(wrap);
    const view = { wrap, img };
    pinnedShotViews.set(key, view);
    return view;
}
function applyPinnedShotsState(payload) {
    const p = (payload && typeof payload === 'object') ? payload : {};
    const shots = Array.isArray(p.shots) ? p.shots : [];
    const nextIds = new Set();
    pinnedShotIdsInWindow = shots.map(s => String(s && s.id ? s.id : '')).filter(Boolean);
    for (const s0 of shots) {
        const s = (s0 && typeof s0 === 'object') ? s0 : null;
        if (!s)
            continue;
        const id = String(s.id || '').trim();
        if (!id)
            continue;
        const r = s.rect && typeof s.rect === 'object' ? s.rect : null;
        if (!r)
            continue;
        nextIds.add(id);
        const view = ensurePinnedShotView(id);
        if (!view)
            continue;
        view.wrap.style.left = `${Math.round(Number(r.x) || 0)}px`;
        view.wrap.style.top = `${Math.round(Number(r.y) || 0)}px`;
        view.wrap.style.width = `${Math.max(1, Math.round(Number(r.w) || 1))}px`;
        view.wrap.style.height = `${Math.max(1, Math.round(Number(r.h) || 1))}px`;
        if (s.pngBase64)
            view.img.src = `data:image/png;base64,${String(s.pngBase64).trim()}`;
    }
    for (const [id, view] of pinnedShotViews.entries()) {
        if (nextIds.has(id))
            continue;
        try {
            view.wrap.remove();
        }
        catch { }
        pinnedShotViews.delete(id);
    }
}
let refImageIdSeq = 0;
let refFileInput = null;
refImageViews = refImageViews instanceof Map ? refImageViews : new Map();
function ensureRefFileInput() {
    if (refFileInput && refFileInput.isConnected)
        return refFileInput;
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = 'image/*';
    input.multiple = true;
    input.style.display = 'none';
    input.addEventListener('change', async () => {
        const files = input.files ? Array.from(input.files) : [];
        try {
            input.value = '';
        }
        catch { }
        if (!files.length)
            return;
        for (const f of files) {
            try {
                await addRefImageFromFile(f);
            }
            catch { }
        }
        try {
            bubble.showText(`已导入 ${files.length} 张参考图`);
        }
        catch { }
    });
    document.body.appendChild(input);
    refFileInput = input;
    return input;
}
async function openRefImagePicker() {
    try {
        if (window.tt && typeof window.tt.pickRefImages === 'function') {
            const picked = await window.tt.pickRefImages();
            const list = Array.isArray(picked) ? picked : [];
            let ok = 0;
            for (const item of list) {
                const dataUrl = String(item && item.dataUrl ? item.dataUrl : '').trim();
                if (!dataUrl)
                    continue;
                try {
                    await addRefImageFromDataUrl(dataUrl);
                }
                catch { }
                ok++;
            }
            if (ok) {
                try {
                    bubble.showText(`已导入 ${ok} 张参考图`);
                }
                catch { }
            }
            return;
        }
    }
    catch { }
    const input = ensureRefFileInput();
    try {
        input.click();
    }
    catch { }
}
function nextRefImageId() {
    refImageIdSeq++;
    return `ref_${Date.now()}_${refImageIdSeq}_${Math.random().toString(16).slice(2)}`;
}
function fileToDataUrl(file) {
    return new Promise((resolve) => {
        const f = file || null;
        if (!f) {
            resolve('');
            return;
        }
        const r = new FileReader();
        r.onerror = () => resolve('');
        r.onload = () => resolve(String(r.result || ''));
        try {
            r.readAsDataURL(f);
        }
        catch {
            resolve('');
        }
    });
}
async function addRefImageFromDataUrl(dataUrl0) {
    const dataUrl = String(dataUrl0 || '').trim();
    if (!dataUrl)
        return;
    const img = new Image();
    img.decoding = 'async';
    img.src = dataUrl;
    try {
        if (typeof img.decode === 'function')
            await img.decode();
        else
            await new Promise((resolve, reject) => { img.onload = resolve; img.onerror = reject; });
    }
    catch { }
    const id = nextRefImageId();
    const iw0 = Math.max(1, Number(img.naturalWidth) || 1);
    const ih0 = Math.max(1, Number(img.naturalHeight) || 1);
    const maxW = Math.min(520, Math.max(220, Math.round(window.innerWidth * 0.28)));
    const baseW = Math.min(iw0, maxW);
    const baseH = Math.max(1, Math.round((ih0 / iw0) * baseW));
    const initialX = Math.round((window.innerWidth - baseW) / 2 + ((refImageIdSeq - 1) % 4) * 18);
    const initialY = Math.round(Math.max(56, (window.innerHeight - baseH) / 2) + ((refImageIdSeq - 1) % 4) * 18);
    try {
        if (window.tt && typeof window.tt.createRefImage === 'function') {
            window.tt.createRefImage({ id, dataUrl, rect: { x: initialX, y: initialY, w: baseW, h: baseH }, displayId });
        }
    }
    catch { }
}
function ensureRefImageView(id) {
    const key = String(id || '').trim();
    if (!key)
        return null;
    if (refImageViews.has(key))
        return refImageViews.get(key);
    const wrap = document.createElement('div');
    wrap.className = 'ref-image';
    wrap.dataset.refImageId = key;
    const img = new Image();
    img.decoding = 'async';
    img.style.borderRadius = '10px';
    wrap.appendChild(img);
    const gridCanvas = document.createElement('canvas');
    gridCanvas.className = 'ref-grid';
    wrap.appendChild(gridCanvas);
    const cropLayer = document.createElement('div');
    cropLayer.className = 'ref-crop-layer';
    wrap.appendChild(cropLayer);
    const transformLayer = document.createElement('div');
    transformLayer.className = 'ref-transform-layer';
    wrap.appendChild(transformLayer);
    wrap.addEventListener('mouseenter', () => {
        if (refPickerActive || refCropActive || refTransformActive)
            return;
        ensureRefToolbar();
        if (activeRefImageId !== key) {
            activeRefImageId = key;
        }
    });
    wrap.addEventListener('mouseleave', () => {
        if (refPickerActive || refCropActive || refTransformActive)
            return;
        updateRefToolbarAutoVisibility(null);
    });
    wrap.addEventListener('mousemove', (e) => {
        if (refPickerActive || refCropActive || refTransformActive)
            return;
        if (activeRefImageId !== key) {
            activeRefImageId = key;
        }
        updateRefToolbarAutoVisibility({ x: e.clientX, y: e.clientY });
    });
    wrap.addEventListener('mousedown', (e) => {
        if (!e || e.button !== 0)
            return;
        if (shotEditorActive)
            return;
        const v = refImageViews.get(key);
        if (v && v.state && v.state.locked)
            return;
        if (wrap.classList.contains('cropping') || wrap.classList.contains('transforming'))
            return;
        try {
            e.preventDefault();
        }
        catch { }
        const rr = wrap.getBoundingClientRect();
        const offsetX = Math.round(Number(e.clientX) - rr.left);
        const offsetY = Math.round(Number(e.clientY) - rr.top);
        try {
            if (window.tt && typeof window.tt.startRefImageDrag === 'function') {
                window.tt.startRefImageDrag({ id: key, offsetX, offsetY });
            }
        }
        catch { }
    });
    wrap.addEventListener('wheel', (e) => {
        if (!e)
            return;
        if (shotEditorActive)
            return;
        if (wrap.classList.contains('cropping') || wrap.classList.contains('transforming'))
            return;
        const v = refImageViews.get(key);
        if (v && v.state && v.state.locked)
            return;
        const deltaY = Number(e.deltaY) || 0;
        if (!Number.isFinite(deltaY) || deltaY === 0)
            return;
        const rr = wrap.getBoundingClientRect();
        const w = Math.max(1, Math.round(Number(rr.width) || 1));
        const h = Math.max(1, Math.round(Number(rr.height) || 1));
        const rx = w > 0 ? (Math.round(Number(e.clientX) - rr.left) / w) : 0.5;
        const ry = h > 0 ? (Math.round(Number(e.clientY) - rr.top) / h) : 0.5;
        if (e.altKey) {
            try {
                if (window.tt && typeof window.tt.rotateRefImage === 'function') {
                    const delta = deltaY > 0 ? 5 : -5;
                    window.tt.rotateRefImage({ id: key, delta });
                }
            }
            catch { }
        }
        else {
            try {
                if (window.tt && typeof window.tt.scaleRefImage === 'function') {
                    window.tt.scaleRefImage({ id: key, deltaY, rx, ry });
                }
            }
            catch { }
        }
        try {
            e.preventDefault();
        }
        catch { }
    }, { passive: false });
    wrap.addEventListener('contextmenu', (e) => {
        try {
            e.preventDefault();
        }
        catch { }
        if (wrap.classList.contains('transforming')) {
            endRefImageTransformMode(key);
            return;
        }
        try {
            if (window.tt && typeof window.tt.removeRefImage === 'function') {
                window.tt.removeRefImage(key);
            }
        }
        catch { }
        try {
            bubble.showText('已关闭参考图');
        }
        catch { }
    });
    document.body.appendChild(wrap);
    const view = { id: key, wrap, img, gridCanvas, cropLayer, transformLayer, state: null };
    refImageViews.set(key, view);
    return view;
}
function applyRefImagesState(payload) {
    const p = (payload && typeof payload === 'object') ? payload : {};
    const images = Array.isArray(p.images) ? p.images : [];
    const nextIds = new Set();
    for (const s0 of images) {
        const s = (s0 && typeof s0 === 'object') ? s0 : null;
        if (!s)
            continue;
        const id = String(s.id || '').trim();
        if (!id)
            continue;
        const r = s.rect && typeof s.rect === 'object' ? s.rect : null;
        if (!r)
            continue;
        nextIds.add(id);
        const view = ensureRefImageView(id);
        if (!view)
            continue;
        view.wrap.style.left = `${Math.round(Number(r.x) || 0)}px`;
        view.wrap.style.top = `${Math.round(Number(r.y) || 0)}px`;
        view.wrap.style.width = `${Math.max(1, Math.round(Number(r.w) || 1))}px`;
        view.wrap.style.height = `${Math.max(1, Math.round(Number(r.h) || 1))}px`;
        const z = Math.max(0, Math.round(Number(s.z) || 0));
        view.wrap.style.zIndex = String(1600 + z);
        const rotation = Number(s.rotation) || 0;
        const flipH = !!s.flipH;
        const flipV = !!s.flipV;
        const scaleX = flipH ? -1 : 1;
        const scaleY = flipV ? -1 : 1;
        view.wrap.style.transform = `rotate(${rotation}deg) scaleX(${scaleX}) scaleY(${scaleY})`;
        const grayscale = !!s.grayscale;
        const invert = !!s.invert;
        const brightness = Number(s.brightness) || 1;
        const contrast = Number(s.contrast) || 1;
        const filters = [];
        if (grayscale)
            filters.push('grayscale(1)');
        if (invert)
            filters.push('invert(1)');
        filters.push(`brightness(${brightness})`);
        filters.push(`contrast(${contrast})`);
        view.img.style.filter = filters.join(' ');
        const opacity = Number(s.opacity) || 1;
        view.wrap.style.opacity = String(opacity);
        view.wrap.classList.toggle('locked', !!s.locked);
        view.wrap.classList.toggle('show-grid', !!s.showGrid);
        if (!!s.showGrid && view.gridCanvas) {
            drawRefImageGrid(view);
        }
        view.state = {
            flipH,
            flipV,
            grayscale,
            invert,
            locked: !!s.locked,
            showGrid: !!s.showGrid,
            brightness,
            contrast,
            opacity,
            rotation
        };
        if (s.dataUrl)
            view.img.src = String(s.dataUrl || '');
    }
    for (const [id, view] of refImageViews.entries()) {
        if (nextIds.has(id))
            continue;
        try {
            view.wrap.remove();
        }
        catch { }
        refImageViews.delete(id);
        if (id === activeRefImageId) {
            hideRefHotzone();
            activeRefImageId = null;
        }
    }
    scheduleSyncInteractiveRects();
    updateRefToolbarButtonStates();
    if (refToolbarHotzone && refToolbarHotzone.classList.contains('visible') && activeRefImageId) {
        const view = refImageViews.get(activeRefImageId);
        if (view) {
            const rect = view.wrap.getBoundingClientRect();
            const margin = 12;
            const tbHeight = 48;
            let top = rect.bottom + margin;
            if (top + tbHeight > window.innerHeight - margin) {
                top = rect.top - margin - tbHeight;
            }
            top = Math.max(margin, Math.min(window.innerHeight - tbHeight - margin, top));
            const cx = rect.left + rect.width / 2;
            refToolbarHotzone.style.left = `${Math.round(cx)}px`;
            refToolbarHotzone.style.top = `${Math.round(top)}px`;
        }
    }
}
function drawRefImageGrid(view) {
    const canvas = view.gridCanvas;
    if (!canvas)
        return;
    const rect = view.wrap.getBoundingClientRect();
    const w = Math.max(1, rect.width);
    const h = Math.max(1, rect.height);
    canvas.width = w;
    canvas.height = h;
    const ctx = canvas.getContext('2d');
    if (!ctx)
        return;
    ctx.clearRect(0, 0, w, h);
    ctx.strokeStyle = 'rgba(255, 255, 255, 0.4)';
    ctx.lineWidth = 1;
    const gridSize = Math.max(20, Math.min(w, h) / 8);
    const cols = Math.ceil(w / gridSize);
    const rows = Math.ceil(h / gridSize);
    ctx.beginPath();
    for (let i = 0; i <= cols; i++) {
        const x = Math.round(i * gridSize) + 0.5;
        ctx.moveTo(x, 0);
        ctx.lineTo(x, h);
    }
    for (let i = 0; i <= rows; i++) {
        const y = Math.round(i * gridSize) + 0.5;
        ctx.moveTo(0, y);
        ctx.lineTo(w, y);
    }
    ctx.stroke();
    ctx.strokeStyle = 'rgba(255, 255, 255, 0.2)';
    ctx.beginPath();
    const subGrid = gridSize / 2;
    for (let i = 0; i <= cols * 2; i++) {
        const x = Math.round(i * subGrid) + 0.5;
        ctx.moveTo(x, 0);
        ctx.lineTo(x, h);
    }
    for (let i = 0; i <= rows * 2; i++) {
        const y = Math.round(i * subGrid) + 0.5;
        ctx.moveTo(0, y);
        ctx.lineTo(w, y);
    }
    ctx.stroke();
}
async function addRefImageFromFile(file) {
    const dataUrl = String((await fileToDataUrl(file)) || '');
    if (!dataUrl)
        return;
    const img = new Image();
    img.decoding = 'async';
    img.src = dataUrl;
    try {
        if (typeof img.decode === 'function')
            await img.decode();
        else
            await new Promise((resolve, reject) => { img.onload = resolve; img.onerror = reject; });
    }
    catch { }
    const id = nextRefImageId();
    const iw0 = Math.max(1, Number(img.naturalWidth) || 1);
    const ih0 = Math.max(1, Number(img.naturalHeight) || 1);
    const maxW = Math.min(520, Math.max(220, Math.round(window.innerWidth * 0.28)));
    const baseW = Math.min(iw0, maxW);
    const baseH = Math.max(1, Math.round((ih0 / iw0) * baseW));
    const initialX = Math.round((window.innerWidth - baseW) / 2 + ((refImageIdSeq - 1) % 4) * 18);
    const initialY = Math.round(Math.max(56, (window.innerHeight - baseH) / 2) + ((refImageIdSeq - 1) % 4) * 18);
    try {
        if (window.tt && typeof window.tt.createRefImage === 'function') {
            window.tt.createRefImage({ id, dataUrl, rect: { x: initialX, y: initialY, w: baseW, h: baseH }, displayId });
        }
    }
    catch { }
}
let refToolbar = null;
let refToolbarHotzone = null;
let refAdjustMenu = null;
let refOpacityMenu = null;
let activeRefImageId = null;
let refToolbarAutoHideTimer = null;
let refToolbarLastMousePos = null;
let refPickerActive = false;
let refCropActive = false;
let refTransformActive = false;
let refPickerCursor = null;
let refToolbarHotzoneWired = false;
let refUiRects = [];
function ensureRefToolbar() {
    if (refToolbar && refToolbar.isConnected) {
        if (!refToolbarHotzone || !refToolbarHotzone.isConnected) {
            refToolbarHotzone = document.getElementById('ref-toolbar-hotzone');
            refToolbarHotzoneWired = false;
        }
        wireRefToolbarHotzone();
        return;
    }
    refToolbar = document.getElementById('ref-image-toolbar');
    refToolbarHotzone = document.getElementById('ref-toolbar-hotzone');
    refAdjustMenu = document.getElementById('ref-adjust-menu');
    refOpacityMenu = document.getElementById('ref-opacity-menu');
    refToolbarHotzoneWired = false;
    if (!refToolbar)
        return;
    refToolbar.addEventListener('click', (e) => {
        const target = e.target;
        const btn = target.closest('button[data-ref-action]');
        if (!btn)
            return;
        const action = btn.dataset.refAction;
        if (!action || !activeRefImageId)
            return;
        if (action === 'adjust' || action === 'opacity')
            return;
        e.stopPropagation();
        let menuChanged = false;
        if (refAdjustMenu && refAdjustMenu.classList.contains('visible')) {
            refAdjustMenu.classList.remove('visible');
            menuChanged = true;
        }
        if (refOpacityMenu && refOpacityMenu.classList.contains('visible')) {
            refOpacityMenu.classList.remove('visible');
            menuChanged = true;
        }
        if (menuChanged)
            syncRefInteractiveRects();
        handleRefToolbarAction(action);
    });
    const brightnessInput = document.getElementById('ref-brightness');
    const brightnessVal = document.getElementById('ref-brightness-val');
    const contrastInput = document.getElementById('ref-contrast');
    const contrastVal = document.getElementById('ref-contrast-val');
    const opacityInput = document.getElementById('ref-opacity');
    const opacityVal = document.getElementById('ref-opacity-val');
    if (brightnessInput) {
        brightnessInput.addEventListener('input', () => {
            const v = Number(brightnessInput.value) || 100;
            if (brightnessVal)
                brightnessVal.textContent = `${v}%`;
            if (activeRefImageId && window.tt) {
                window.tt.setRefImageProps({ id: activeRefImageId, brightness: v / 100 });
            }
        });
    }
    if (contrastInput) {
        contrastInput.addEventListener('input', () => {
            const v = Number(contrastInput.value) || 100;
            if (contrastVal)
                contrastVal.textContent = `${v}%`;
            if (activeRefImageId && window.tt) {
                window.tt.setRefImageProps({ id: activeRefImageId, contrast: v / 100 });
            }
        });
    }
    if (opacityInput) {
        opacityInput.addEventListener('input', () => {
            const v = Number(opacityInput.value) || 100;
            if (opacityVal)
                opacityVal.textContent = `${v}%`;
            if (activeRefImageId && window.tt) {
                window.tt.setRefImageProps({ id: activeRefImageId, opacity: v / 100 });
            }
        });
    }
    const adjustBtn = refToolbar.querySelector('button[data-ref-action="adjust"]');
    const opacityBtn = refToolbar.querySelector('button[data-ref-action="opacity"]');
    if (adjustBtn && refAdjustMenu) {
        adjustBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            refAdjustMenu?.classList.toggle('visible');
            refOpacityMenu?.classList.remove('visible');
            syncRefInteractiveRects();
        });
    }
    if (opacityBtn && refOpacityMenu) {
        opacityBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            refOpacityMenu?.classList.toggle('visible');
            refAdjustMenu?.classList.remove('visible');
            syncRefInteractiveRects();
        });
    }
    document.addEventListener('mousedown', (e) => {
        const target = e.target;
        const inTb = refToolbar && target && target.closest ? target.closest('#ref-image-toolbar') : null;
        if (inTb)
            return;
        let changed = false;
        if (refAdjustMenu && refAdjustMenu.classList.contains('visible')) {
            refAdjustMenu.classList.remove('visible');
            changed = true;
        }
        if (refOpacityMenu && refOpacityMenu.classList.contains('visible')) {
            refOpacityMenu.classList.remove('visible');
            changed = true;
        }
        if (changed)
            syncRefInteractiveRects();
    });
    refToolbar.addEventListener('mousedown', (e) => {
        if (refToolbarAutoHideTimer) {
            clearTimeout(refToolbarAutoHideTimer);
            refToolbarAutoHideTimer = null;
        }
    });
    wireRefToolbarHotzone();
}
function wireRefToolbarHotzone() {
    if (refToolbarHotzoneWired)
        return;
    if (!refToolbarHotzone)
        return;
    refToolbarHotzoneWired = true;
    refToolbarHotzone.addEventListener('mouseenter', () => {
        if (!activeRefImageId)
            return;
        hideRefHotzone();
        showRefToolbar(activeRefImageId);
        syncRefInteractiveRects();
    });
}
function handleRefToolbarAction(action) {
    if (!activeRefImageId || !window.tt)
        return;
    const view = refImageViews.get(activeRefImageId);
    if (!view)
        return;
    switch (action) {
        case 'flipH':
            window.tt.setRefImageProps({ id: activeRefImageId, flipH: !view.state?.flipH });
            break;
        case 'flipV':
            window.tt.setRefImageProps({ id: activeRefImageId, flipV: !view.state?.flipV });
            break;
        case 'grayscale':
            window.tt.setRefImageProps({ id: activeRefImageId, grayscale: !view.state?.grayscale });
            break;
        case 'invert':
            window.tt.setRefImageProps({ id: activeRefImageId, invert: !view.state?.invert });
            break;
        case 'grid':
            window.tt.setRefImageProps({ id: activeRefImageId, showGrid: !view.state?.showGrid });
            break;
        case 'lock':
            const willLock = !view.state?.locked;
            window.tt.setRefImageProps({ id: activeRefImageId, locked: willLock });
            try {
                bubble.showText(willLock ? '已锁定' : '已解锁');
            }
            catch { }
            break;
        case 'duplicate':
            const newId = nextRefImageId();
            window.tt.duplicateRefImage({ srcId: activeRefImageId, newId });
            try {
                bubble.showText('已复制');
            }
            catch { }
            break;
        case 'reset':
            const resetView = refImageViews.get(activeRefImageId);
            if (resetView && resetView._originalImgSrc) {
                resetView.img.src = resetView._originalImgSrc;
                resetView._cropRect = null;
                resetView._pendingCropRect = null;
                resetView._originalImgSrc = null;
                resetView._originalImgSize = null;
            }
            window.tt.resetRefImageProps(activeRefImageId);
            try {
                bubble.showText('已重置');
            }
            catch { }
            break;
        case 'close':
            window.tt.removeRefImage(activeRefImageId);
            activeRefImageId = null;
            hideRefToolbar();
            break;
        case 'picker':
            startRefColorPicker(activeRefImageId);
            break;
        case 'crop':
            startRefImageCrop(activeRefImageId);
            break;
        case 'transform':
            startRefImageTransform(activeRefImageId);
            break;
    }
}
function updateRefToolbarButtonStates() {
    if (!refToolbar || !activeRefImageId)
        return;
    const view = refImageViews.get(activeRefImageId);
    if (!view)
        return;
    const state = view.state || {};
    const btns = refToolbar.querySelectorAll('button[data-ref-action]');
    btns.forEach((btn) => {
        const action = btn.dataset.refAction;
        if (!action)
            return;
        let active = false;
        switch (action) {
            case 'flipH':
                active = !!state.flipH;
                break;
            case 'flipV':
                active = !!state.flipV;
                break;
            case 'grayscale':
                active = !!state.grayscale;
                break;
            case 'invert':
                active = !!state.invert;
                break;
            case 'grid':
                active = !!state.showGrid;
                break;
            case 'lock':
                active = !!state.locked;
                break;
            case 'transform':
                active = refTransformActive;
                break;
        }
        btn.classList.toggle('active', active);
    });
    const brightnessInput = document.getElementById('ref-brightness');
    const contrastInput = document.getElementById('ref-contrast');
    const opacityInput = document.getElementById('ref-opacity');
    const brightnessVal = document.getElementById('ref-brightness-val');
    const contrastVal = document.getElementById('ref-contrast-val');
    const opacityVal = document.getElementById('ref-opacity-val');
    if (brightnessInput) {
        const v = Math.round((state.brightness || 1) * 100);
        brightnessInput.value = String(v);
        if (brightnessVal)
            brightnessVal.textContent = `${v}%`;
    }
    if (contrastInput) {
        const v = Math.round((state.contrast || 1) * 100);
        contrastInput.value = String(v);
        if (contrastVal)
            contrastVal.textContent = `${v}%`;
    }
    if (opacityInput) {
        const v = Math.round((state.opacity || 1) * 100);
        opacityInput.value = String(v);
        if (opacityVal)
            opacityVal.textContent = `${v}%`;
    }
}
function showRefToolbar(refImageId) {
    ensureRefToolbar();
    if (!refToolbar)
        return;
    if (refToolbarAutoHideTimer) {
        clearTimeout(refToolbarAutoHideTimer);
        refToolbarAutoHideTimer = null;
    }
    activeRefImageId = refImageId;
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    const rect = view.wrap.getBoundingClientRect();
    const margin = 12;
    const tbHeight = 48;
    let top = rect.bottom + margin;
    if (top + tbHeight > window.innerHeight - margin) {
        top = rect.top - margin - tbHeight;
    }
    top = Math.max(margin, Math.min(window.innerHeight - tbHeight - margin, top));
    const cx = rect.left + rect.width / 2;
    refToolbar.style.left = `${Math.round(cx)}px`;
    refToolbar.style.top = `${Math.round(top)}px`;
    refToolbar.classList.add('visible');
    if (refToolbarHotzone) {
        refToolbarHotzone.style.left = `${Math.round(cx)}px`;
        refToolbarHotzone.style.top = `${Math.round(top)}px`;
        refToolbarHotzone.classList.remove('visible');
    }
    updateRefToolbarButtonStates();
    syncRefInteractiveRects();
}
function hideRefToolbar() {
    if (!refToolbar)
        return;
    refToolbar.classList.remove('visible');
    refAdjustMenu?.classList.remove('visible');
    refOpacityMenu?.classList.remove('visible');
    if (refToolbarAutoHideTimer) {
        clearTimeout(refToolbarAutoHideTimer);
        refToolbarAutoHideTimer = null;
    }
    refToolbarLastMousePos = null;
    if (activeRefImageId && refImageViews.has(activeRefImageId)) {
        showRefHotzone(activeRefImageId);
    }
    syncRefInteractiveRects();
}
function showRefHotzone(refImageId) {
    ensureRefToolbar();
    if (!refToolbarHotzone || !refToolbar)
        return;
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    activeRefImageId = refImageId;
    const rect = view.wrap.getBoundingClientRect();
    const margin = 12;
    const tbHeight = 48;
    let top = rect.bottom + margin;
    if (top + tbHeight > window.innerHeight - margin) {
        top = rect.top - margin - tbHeight;
    }
    top = Math.max(margin, Math.min(window.innerHeight - tbHeight - margin, top));
    const cx = rect.left + rect.width / 2;
    refToolbarHotzone.style.left = `${Math.round(cx)}px`;
    refToolbarHotzone.style.top = `${Math.round(top)}px`;
    refToolbarHotzone.classList.add('visible');
    syncRefInteractiveRects();
}
function hideRefHotzone() {
    if (refToolbarHotzone) {
        refToolbarHotzone.classList.remove('visible');
    }
    syncRefInteractiveRects();
}
function scheduleRefToolbarHide() {
    if (refToolbarAutoHideTimer) {
        clearTimeout(refToolbarAutoHideTimer);
    }
    refToolbarAutoHideTimer = setTimeout(() => {
        refToolbarAutoHideTimer = null;
        hideRefToolbar();
    }, 300);
}
function positionRefToolbar() {
    ensureRefToolbar();
    if (!refToolbar || !activeRefImageId)
        return;
    const view = refImageViews.get(activeRefImageId);
    if (!view)
        return;
    const rect = view.wrap.getBoundingClientRect();
    const margin = 12;
    const tb = refToolbar.getBoundingClientRect();
    const w = Math.max(1, tb.width || refToolbar.offsetWidth || 1);
    const h = Math.max(1, tb.height || refToolbar.offsetHeight || 1);
    let cx = rect.left + rect.width / 2;
    let top = rect.bottom + margin;
    const outsideFits = top + h <= window.innerHeight - margin;
    if (!outsideFits) {
        top = rect.top - margin - h;
    }
    top = Math.max(margin, Math.min(window.innerHeight - margin - h, top));
    cx = Math.max(margin + w / 2, Math.min(window.innerWidth - margin - w / 2, cx));
    refToolbar.style.left = `${Math.round(cx)}px`;
    refToolbar.style.top = `${Math.round(top)}px`;
    if (refToolbarHotzone) {
        refToolbarHotzone.style.left = `${Math.round(cx)}px`;
        refToolbarHotzone.style.top = `${Math.round(top)}px`;
    }
}
function updateRefToolbarAutoVisibility(pos) {
    if (!activeRefImageId) {
        hideRefHotzone();
        if (refToolbar)
            refToolbar.classList.remove('visible');
        syncRefInteractiveRects();
        return;
    }
    if (!pos || typeof pos !== 'object')
        return;
    refToolbarLastMousePos = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
    positionRefToolbar();
    const tbVisible = !!(refToolbar && refToolbar.classList.contains('visible'));
    const hzVisible = !!(refToolbarHotzone && refToolbarHotzone.classList.contains('visible'));
    const inHotzone = hzVisible && refToolbarHotzone ? pointInDomRect(refToolbarLastMousePos, refToolbarHotzone.getBoundingClientRect()) : false;
    const inTb = tbVisible && refToolbar ? pointInDomRect(refToolbarLastMousePos, refToolbar.getBoundingClientRect()) : false;
    let inMenu = false;
    if (tbVisible) {
        if (refAdjustMenu && refAdjustMenu.classList.contains('visible') && pointInDomRect(refToolbarLastMousePos, refAdjustMenu.getBoundingClientRect()))
            inMenu = true;
        if (refOpacityMenu && refOpacityMenu.classList.contains('visible') && pointInDomRect(refToolbarLastMousePos, refOpacityMenu.getBoundingClientRect()))
            inMenu = true;
    }
    if (!tbVisible) {
        if (!hzVisible)
            showRefHotzone(activeRefImageId);
        if (inHotzone) {
            if (refToolbarAutoHideTimer) {
                clearTimeout(refToolbarAutoHideTimer);
                refToolbarAutoHideTimer = null;
            }
            showRefToolbar(activeRefImageId);
            hideRefHotzone();
        }
        return;
    }
    if (inTb || inMenu) {
        if (refToolbarAutoHideTimer) {
            clearTimeout(refToolbarAutoHideTimer);
            refToolbarAutoHideTimer = null;
        }
        return;
    }
    if (refToolbarAutoHideTimer)
        return;
    refToolbarAutoHideTimer = setTimeout(() => {
        refToolbarAutoHideTimer = null;
        if (!activeRefImageId)
            return;
        positionRefToolbar();
        const p = refToolbarLastMousePos;
        const stillInTb = refToolbar && refToolbar.classList.contains('visible') ? pointInDomRect(p, refToolbar.getBoundingClientRect()) : false;
        let stillInMenu = false;
        if (refAdjustMenu && refAdjustMenu.classList.contains('visible') && pointInDomRect(p, refAdjustMenu.getBoundingClientRect()))
            stillInMenu = true;
        if (refOpacityMenu && refOpacityMenu.classList.contains('visible') && pointInDomRect(p, refOpacityMenu.getBoundingClientRect()))
            stillInMenu = true;
        if (stillInTb || stillInMenu)
            return;
        hideRefToolbar();
    }, 260);
}
function syncRefInteractiveRects() {
    ensureRefToolbar();
    const rects = [];
    const pushEl = (el) => {
        if (!el || !el.isConnected)
            return;
        const b = el.getBoundingClientRect();
        const w = Math.max(0, Math.round(Number(b.width) || 0));
        const h = Math.max(0, Math.round(Number(b.height) || 0));
        if (w < 2 || h < 2)
            return;
        const x = Math.max(0, Math.round(Number(b.left) || 0));
        const y = Math.max(0, Math.round(Number(b.top) || 0));
        rects.push({ x, y, w, h });
    };
    if (activeRefImageId) {
        if (refToolbar && refToolbar.classList.contains('visible')) {
            pushEl(refToolbar);
            if (refAdjustMenu && refAdjustMenu.classList.contains('visible'))
                pushEl(refAdjustMenu);
            if (refOpacityMenu && refOpacityMenu.classList.contains('visible'))
                pushEl(refOpacityMenu);
        }
        else if (refToolbarHotzone && refToolbarHotzone.classList.contains('visible')) {
            pushEl(refToolbarHotzone);
        }
    }
    refUiRects = rects;
    if (!window.tt || typeof window.tt.setInteractiveRects !== 'function')
        return;
    const allRects = rects.slice();
    for (const v of refImageViews.values()) {
        const wrap = v && v.wrap ? v.wrap : null;
        if (!wrap || !wrap.isConnected)
            continue;
        const b = wrap.getBoundingClientRect();
        const w = Math.max(0, Math.round(Number(b.width) || 0));
        const h = Math.max(0, Math.round(Number(b.height) || 0));
        if (w < 2 || h < 2)
            continue;
        const x = Math.max(0, Math.round(Number(b.left) || 0));
        const y = Math.max(0, Math.round(Number(b.top) || 0));
        allRects.push({ x, y, w, h });
    }
    try {
        window.tt.setInteractiveRects(allRects);
    }
    catch { }
}
function refUiHit(pos) {
    if (!pos || !refUiRects || !refUiRects.length)
        return false;
    const x = Math.round(Number(pos.x) || 0);
    const y = Math.round(Number(pos.y) || 0);
    return refUiRects.some((r) => x >= r.x && x <= (r.x + r.w) && y >= r.y && y <= (r.y + r.h));
}
function startRefColorPicker(refImageId) {
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    refPickerActive = true;
    document.body.style.cursor = 'crosshair';
    hideRefToolbar();
    if (!refPickerCursor) {
        refPickerCursor = document.createElement('div');
        refPickerCursor.className = 'ref-picker-cursor';
        document.body.appendChild(refPickerCursor);
    }
    refPickerCursor.classList.add('visible');
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d', { willReadFrequently: true });
    if (!ctx)
        return;
    const img = view.img;
    canvas.width = img.naturalWidth || 1;
    canvas.height = img.naturalHeight || 1;
    const state = view.state || {};
    const filters = [];
    if (state.grayscale)
        filters.push('grayscale(1)');
    if (state.invert)
        filters.push('invert(1)');
    filters.push(`brightness(${state.brightness || 1})`);
    filters.push(`contrast(${state.contrast || 1})`);
    ctx.filter = filters.join(' ');
    const rotation = state.rotation || 0;
    const flipH = state.flipH;
    const flipV = state.flipV;
    const rad = (rotation * Math.PI) / 180;
    const cos = Math.cos(rad);
    const sin = Math.sin(rad);
    const iw = canvas.width;
    const ih = canvas.height;
    const transformPoint = (px, py) => {
        let x = px - iw / 2;
        let y = py - ih / 2;
        if (flipH)
            x = -x;
        if (flipV)
            y = -y;
        const rx = x * cos - y * sin + iw / 2;
        const ry = x * sin + y * cos + ih / 2;
        return { x: Math.floor(rx), y: Math.floor(ry) };
    };
    try {
        ctx.drawImage(img, 0, 0);
    }
    catch { }
    ctx.filter = 'none';
    const pickHandler = (e) => {
        if (!refPickerActive)
            return;
        const rect = view.wrap.getBoundingClientRect();
        const rx = (e.clientX - rect.left) / rect.width;
        const ry = (e.clientY - rect.top) / rect.height;
        let px = Math.floor(rx * canvas.width);
        let py = Math.floor(ry * canvas.height);
        if (rotation !== 0 || flipH || flipV) {
            const transformed = transformPoint(px, py);
            px = transformed.x;
            py = transformed.y;
        }
        if (px < 0 || px >= canvas.width || py < 0 || py >= canvas.height)
            return;
        try {
            const pixel = ctx.getImageData(px, py, 1, 1).data;
            const r = pixel[0];
            const g = pixel[1];
            const b = pixel[2];
            const hex = `#${r.toString(16).padStart(2, '0')}${g.toString(16).padStart(2, '0')}${b.toString(16).padStart(2, '0')}`;
            if (refPickerCursor) {
                refPickerCursor.style.left = `${e.clientX}px`;
                refPickerCursor.style.top = `${e.clientY}px`;
                refPickerCursor.style.setProperty('--picker-color', hex);
            }
        }
        catch { }
    };
    const clickHandler = (e) => {
        if (!refPickerActive)
            return;
        e.preventDefault();
        e.stopPropagation();
        const rect = view.wrap.getBoundingClientRect();
        const rx = (e.clientX - rect.left) / rect.width;
        const ry = (e.clientY - rect.top) / rect.height;
        let px = Math.floor(rx * canvas.width);
        let py = Math.floor(ry * canvas.height);
        if (rotation !== 0 || flipH || flipV) {
            const transformed = transformPoint(px, py);
            px = transformed.x;
            py = transformed.y;
        }
        if (px < 0 || px >= canvas.width || py < 0 || py >= canvas.height) {
            endRefColorPicker();
            return;
        }
        try {
            const pixel = ctx.getImageData(px, py, 1, 1).data;
            const r = pixel[0];
            const g = pixel[1];
            const b = pixel[2];
            const hex = `#${r.toString(16).padStart(2, '0')}${g.toString(16).padStart(2, '0')}${b.toString(16).padStart(2, '0')}`;
            if (window.tt) {
                window.tt.copyText(hex);
            }
            try {
                bubble.showText(`已复制颜色: ${hex}`);
            }
            catch { }
        }
        catch { }
        endRefColorPicker();
    };
    const keyHandler = (e) => {
        if (e.key === 'Escape') {
            endRefColorPicker();
        }
    };
    document.addEventListener('mousemove', pickHandler);
    document.addEventListener('click', clickHandler, true);
    document.addEventListener('keydown', keyHandler);
    const endRefColorPicker = () => {
        refPickerActive = false;
        document.body.style.cursor = '';
        if (refPickerCursor)
            refPickerCursor.classList.remove('visible');
        document.removeEventListener('mousemove', pickHandler);
        document.removeEventListener('click', clickHandler, true);
        document.removeEventListener('keydown', keyHandler);
    };
}
function startRefImageCrop(refImageId) {
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    const { wrap, img, cropLayer } = view;
    if (wrap.classList.contains('cropping')) {
        const cropRect = view._pendingCropRect;
        const originalSrc = view._originalImgSrc;
        if (cropRect && originalSrc) {
            applyRefImageCropAsMask(refImageId, cropRect, originalSrc);
        }
        endRefImageCropMode(refImageId);
        return;
    }
    const originalSrc = view._originalImgSrc || img.src;
    view._originalImgSrc = originalSrc;
    const tempImg = new Image();
    tempImg.onload = () => {
        const imgW = tempImg.naturalWidth || 100;
        const imgH = tempImg.naturalHeight || 100;
        view._originalImgSize = { w: imgW, h: imgH };
        const wrapRect = wrap.getBoundingClientRect();
        const displayW = wrapRect.width;
        const displayH = wrapRect.height;
        const existingCrop = view._cropRect;
        let cropRect = existingCrop
            ? { ...existingCrop }
            : { x: 0, y: 0, w: imgW, h: imgH };
        view._pendingCropRect = cropRect;
        refCropActive = true;
        wrap.classList.add('cropping');
        hideRefToolbar();
        cropLayer.innerHTML = '';
        const cropBox = document.createElement('div');
        cropBox.className = 'ref-crop-box';
        const cropInner = document.createElement('div');
        cropInner.className = 'ref-crop-box-inner';
        const cropImg = new Image();
        cropImg.src = originalSrc;
        cropInner.appendChild(cropImg);
        cropBox.appendChild(cropInner);
        const handleDirs = ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w'];
        handleDirs.forEach(dir => {
            const h = document.createElement('div');
            h.className = 'ref-crop-handle';
            h.dataset.dir = dir;
            cropBox.appendChild(h);
        });
        cropLayer.appendChild(cropBox);
        const updateCropBox = () => {
            const scaleX = displayW / imgW;
            const scaleY = displayH / imgH;
            const boxX = cropRect.x * scaleX;
            const boxY = cropRect.y * scaleY;
            const boxW = cropRect.w * scaleX;
            const boxH = cropRect.h * scaleY;
            cropBox.style.left = `${boxX}px`;
            cropBox.style.top = `${boxY}px`;
            cropBox.style.width = `${boxW}px`;
            cropBox.style.height = `${boxH}px`;
            cropImg.style.left = `${-cropRect.x * scaleX}px`;
            cropImg.style.top = `${-cropRect.y * scaleY}px`;
            cropImg.style.width = `${imgW * scaleX}px`;
            cropImg.style.height = `${imgH * scaleY}px`;
        };
        updateCropBox();
        let dragState = null;
        const onMouseDown = (e) => {
            if (e.button !== 0)
                return;
            e.preventDefault();
            e.stopPropagation();
            const target = e.target;
            const handle = target.closest('.ref-crop-handle');
            if (handle) {
                const dir = handle.dataset.dir || '';
                dragState = {
                    type: 'resize',
                    dir,
                    startX: e.clientX,
                    startY: e.clientY,
                    startRect: { ...cropRect }
                };
            }
            else if (target === cropBox || target === cropInner || target === cropImg) {
                dragState = {
                    type: 'move',
                    startX: e.clientX,
                    startY: e.clientY,
                    startRect: { ...cropRect }
                };
            }
        };
        const onMouseMove = (e) => {
            if (!dragState)
                return;
            const scaleX = imgW / displayW;
            const scaleY = imgH / displayH;
            const dx = (e.clientX - dragState.startX) * scaleX;
            const dy = (e.clientY - dragState.startY) * scaleY;
            if (dragState.type === 'move') {
                cropRect.x = Math.max(0, Math.min(imgW - cropRect.w, dragState.startRect.x + dx));
                cropRect.y = Math.max(0, Math.min(imgH - cropRect.h, dragState.startRect.y + dy));
            }
            else if (dragState.type === 'resize' && dragState.dir) {
                const sr = dragState.startRect;
                let nx = sr.x, ny = sr.y, nw = sr.w, nh = sr.h;
                const minSize = 20;
                switch (dragState.dir) {
                    case 'nw':
                        nx = Math.max(0, Math.min(sr.x + sr.w - minSize, sr.x + dx));
                        ny = Math.max(0, Math.min(sr.y + sr.h - minSize, sr.y + dy));
                        nw = sr.w - (nx - sr.x);
                        nh = sr.h - (ny - sr.y);
                        break;
                    case 'n':
                        ny = Math.max(0, Math.min(sr.y + sr.h - minSize, sr.y + dy));
                        nh = sr.h - (ny - sr.y);
                        break;
                    case 'ne':
                        ny = Math.max(0, Math.min(sr.y + sr.h - minSize, sr.y + dy));
                        nw = Math.max(minSize, Math.min(imgW - sr.x, sr.w + dx));
                        nh = sr.h - (ny - sr.y);
                        break;
                    case 'e':
                        nw = Math.max(minSize, Math.min(imgW - sr.x, sr.w + dx));
                        break;
                    case 'se':
                        nw = Math.max(minSize, Math.min(imgW - sr.x, sr.w + dx));
                        nh = Math.max(minSize, Math.min(imgH - sr.y, sr.h + dy));
                        break;
                    case 's':
                        nh = Math.max(minSize, Math.min(imgH - sr.y, sr.h + dy));
                        break;
                    case 'sw':
                        nx = Math.max(0, Math.min(sr.x + sr.w - minSize, sr.x + dx));
                        nw = sr.w - (nx - sr.x);
                        nh = Math.max(minSize, Math.min(imgH - sr.y, sr.h + dy));
                        break;
                    case 'w':
                        nx = Math.max(0, Math.min(sr.x + sr.w - minSize, sr.x + dx));
                        nw = sr.w - (nx - sr.x);
                        break;
                }
                cropRect.x = nx;
                cropRect.y = ny;
                cropRect.w = nw;
                cropRect.h = nh;
            }
            view._pendingCropRect = cropRect;
            updateCropBox();
        };
        const onMouseUp = () => {
            dragState = null;
        };
        const onDoubleClick = () => {
            cropRect.x = 0;
            cropRect.y = 0;
            cropRect.w = imgW;
            cropRect.h = imgH;
            view._pendingCropRect = cropRect;
            updateCropBox();
        };
        view._cropHandlers = { onMouseMove, onMouseUp };
        cropBox.addEventListener('mousedown', onMouseDown);
        cropBox.addEventListener('dblclick', onDoubleClick);
        document.addEventListener('mousemove', onMouseMove);
        document.addEventListener('mouseup', onMouseUp);
    };
    tempImg.src = originalSrc;
}
function endRefImageCropMode(refImageId) {
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    refCropActive = false;
    view.wrap.classList.remove('cropping');
    view.cropLayer.innerHTML = '';
    if (view._cropHandlers) {
        document.removeEventListener('mousemove', view._cropHandlers.onMouseMove);
        document.removeEventListener('mouseup', view._cropHandlers.onMouseUp);
        view._cropHandlers = null;
    }
}
function applyRefImageCropAsMask(refImageId, cropRect, originalSrc) {
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    if (!ctx)
        return;
    const img = new Image();
    img.onload = () => {
        canvas.width = Math.max(1, cropRect.w);
        canvas.height = Math.max(1, cropRect.h);
        ctx.drawImage(img, cropRect.x, cropRect.y, cropRect.w, cropRect.h, 0, 0, cropRect.w, cropRect.h);
        const croppedDataUrl = canvas.toDataURL('image/png');
        view._cropRect = { ...cropRect };
        view.img.src = croppedDataUrl;
        if (window.tt && typeof window.tt.setRefImageDataUrl === 'function') {
            window.tt.setRefImageDataUrl({ id: refImageId, dataUrl: croppedDataUrl });
        }
        try {
            bubble.showText('裁剪已应用');
        }
        catch { }
    };
    img.src = originalSrc;
}
function applyRefImageCrop(refImageId, cropRect, dataUrl) {
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    if (!ctx)
        return;
    const img = new Image();
    img.onload = () => {
        canvas.width = Math.max(1, cropRect.w);
        canvas.height = Math.max(1, cropRect.h);
        ctx.drawImage(img, cropRect.x, cropRect.y, cropRect.w, cropRect.h, 0, 0, cropRect.w, cropRect.h);
        const croppedDataUrl = canvas.toDataURL('image/png');
        if (window.tt) {
            const view = refImageViews.get(refImageId);
            if (view) {
                window.tt.removeRefImage(refImageId);
                const newId = refImageId;
                const rect = view.wrap.getBoundingClientRect();
                const aspectRatio = cropRect.w / cropRect.h;
                const newW = rect.width;
                const newH = newW / aspectRatio;
                window.tt.createRefImage({
                    id: newId,
                    dataUrl: croppedDataUrl,
                    rect: { x: rect.left, y: rect.top, w: newW, h: newH },
                    displayId
                });
            }
        }
        try {
            bubble.showText('裁剪完成');
        }
        catch { }
    };
    img.src = dataUrl;
}
function startRefImageTransform(refImageId) {
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    const { wrap, transformLayer } = view;
    if (wrap.classList.contains('transforming')) {
        endRefImageTransformMode(refImageId);
        return;
    }
    refTransformActive = true;
    wrap.classList.add('transforming');
    hideRefToolbar();
    transformLayer.innerHTML = '';
    const transformBox = document.createElement('div');
    transformBox.className = 'ref-transform-box';
    const handleDirs = ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w'];
    const handles = new Map();
    handleDirs.forEach(dir => {
        const h = document.createElement('div');
        h.className = 'ref-transform-handle';
        h.dataset.dir = dir;
        transformBox.appendChild(h);
        handles.set(dir, h);
    });
    transformLayer.appendChild(transformBox);
    const updateTransformBox = () => {
        const rect = wrap.getBoundingClientRect();
        transformBox.style.left = '0px';
        transformBox.style.top = '0px';
        transformBox.style.width = `${rect.width}px`;
        transformBox.style.height = `${rect.height}px`;
    };
    updateTransformBox();
    let dragState = null;
    let isRotateMode = false;
    const getRotation = () => {
        const state = view.state || {};
        return typeof state.rotation === 'number' ? state.rotation : 0;
    };
    const getCornerPosition = (dir) => {
        const rect = wrap.getBoundingClientRect();
        switch (dir) {
            case 'nw': return { x: rect.left, y: rect.top };
            case 'ne': return { x: rect.right, y: rect.top };
            case 'se': return { x: rect.right, y: rect.bottom };
            case 'sw': return { x: rect.left, y: rect.bottom };
            default: return null;
        }
    };
    const isOutsideCorner = (dir, clientX, clientY) => {
        const rect = wrap.getBoundingClientRect();
        const cx = rect.left + rect.width / 2;
        const cy = rect.top + rect.height / 2;
        const corner = getCornerPosition(dir);
        if (!corner)
            return false;
        const dx = clientX - cx;
        const dy = clientY - cy;
        const cornerDx = corner.x - cx;
        const cornerDy = corner.y - cy;
        const distToCenter = Math.sqrt(dx * dx + dy * dy);
        const distCornerToCenter = Math.sqrt(cornerDx * cornerDx + cornerDy * cornerDy);
        return distToCenter >= distCornerToCenter * 0.7;
    };
    const updateHandleCursor = (handle, dir, clientX, clientY) => {
        const isCorner = dir === 'nw' || dir === 'ne' || dir === 'se' || dir === 'sw';
        if (isCorner && isOutsideCorner(dir, clientX, clientY)) {
            handle.classList.add('rotate-nearby');
        }
        else {
            handle.classList.remove('rotate-nearby');
        }
    };
    const onHandleMouseMove = (e) => {
        if (dragState)
            return;
        const target = e.target;
        const handle = target.closest('.ref-transform-handle');
        if (handle) {
            const dir = handle.dataset.dir || '';
            updateHandleCursor(handle, dir, e.clientX, e.clientY);
        }
    };
    const onMouseDown = (e) => {
        if (e.button !== 0)
            return;
        e.preventDefault();
        e.stopPropagation();
        const target = e.target;
        const rect = wrap.getBoundingClientRect();
        const handle = target.closest('.ref-transform-handle');
        if (handle) {
            const dir = handle.dataset.dir || '';
            const isCorner = dir === 'nw' || dir === 'ne' || dir === 'se' || dir === 'sw';
            if (isCorner && isOutsideCorner(dir, e.clientX, e.clientY)) {
                dragState = {
                    type: 'rotate',
                    dir,
                    startX: e.clientX,
                    startY: e.clientY,
                    startW: rect.width,
                    startH: rect.height,
                    startRotation: getRotation(),
                    centerX: rect.left + rect.width / 2,
                    centerY: rect.top + rect.height / 2,
                    startLeft: rect.left,
                    startTop: rect.top
                };
            }
            else {
                dragState = {
                    type: 'scale',
                    dir,
                    startX: e.clientX,
                    startY: e.clientY,
                    startW: rect.width,
                    startH: rect.height,
                    startRotation: getRotation(),
                    centerX: rect.left + rect.width / 2,
                    centerY: rect.top + rect.height / 2,
                    startLeft: rect.left,
                    startTop: rect.top
                };
            }
        }
    };
    const onMouseMove = (e) => {
        if (!dragState) {
            const target = e.target;
            const handle = target.closest('.ref-transform-handle');
            handles.forEach((h, dir) => {
                if (handle === h) {
                    updateHandleCursor(h, dir, e.clientX, e.clientY);
                }
                else {
                    h.classList.remove('rotate-nearby');
                }
            });
            return;
        }
        if (dragState.type === 'rotate') {
            const dx = e.clientX - dragState.centerX;
            const dy = e.clientY - dragState.centerY;
            const angle = Math.atan2(dy, dx) * (180 / Math.PI) + 90;
            const snappedAngle = Math.round(angle / 5) * 5;
            if (window.tt && typeof window.tt.setRefImageProps === 'function') {
                window.tt.setRefImageProps({ id: refImageId, rotation: snappedAngle });
            }
        }
        else if (dragState.type === 'scale' && dragState.dir) {
            const dx = e.clientX - dragState.startX;
            const dy = e.clientY - dragState.startY;
            const dir = dragState.dir;
            const minSize = 30;
            let newW = dragState.startW;
            let newH = dragState.startH;
            let newLeft = dragState.startLeft;
            let newTop = dragState.startTop;
            switch (dir) {
                case 'nw':
                    newW = Math.max(minSize, dragState.startW - dx);
                    newH = Math.max(minSize, dragState.startH - dy);
                    newLeft = dragState.startLeft + (dragState.startW - newW);
                    newTop = dragState.startTop + (dragState.startH - newH);
                    break;
                case 'n':
                    newH = Math.max(minSize, dragState.startH - dy);
                    newTop = dragState.startTop + (dragState.startH - newH);
                    break;
                case 'ne':
                    newW = Math.max(minSize, dragState.startW + dx);
                    newH = Math.max(minSize, dragState.startH - dy);
                    newTop = dragState.startTop + (dragState.startH - newH);
                    break;
                case 'e':
                    newW = Math.max(minSize, dragState.startW + dx);
                    break;
                case 'se':
                    newW = Math.max(minSize, dragState.startW + dx);
                    newH = Math.max(minSize, dragState.startH + dy);
                    break;
                case 's':
                    newH = Math.max(minSize, dragState.startH + dy);
                    break;
                case 'sw':
                    newW = Math.max(minSize, dragState.startW - dx);
                    newH = Math.max(minSize, dragState.startH + dy);
                    newLeft = dragState.startLeft + (dragState.startW - newW);
                    break;
                case 'w':
                    newW = Math.max(minSize, dragState.startW - dx);
                    newLeft = dragState.startLeft + (dragState.startW - newW);
                    break;
            }
            if (window.tt && typeof window.tt.setRefImageRect === 'function') {
                window.tt.setRefImageRect({ id: refImageId, rect: { x: newLeft, y: newTop, w: newW, h: newH } });
            }
            setTimeout(updateTransformBox, 10);
        }
    };
    const onMouseUp = () => {
        dragState = null;
    };
    const onKeyDown = (e) => {
        if (e.key === 'Escape' || e.key === 'Enter') {
            endRefImageTransformMode(refImageId);
        }
    };
    view._transformHandlers = { onMouseMove, onMouseUp, onKeyDown };
    transformBox.addEventListener('mousedown', onMouseDown);
    document.addEventListener('mousemove', onMouseMove);
    document.addEventListener('mouseup', onMouseUp);
    document.addEventListener('keydown', onKeyDown);
    try {
        bubble.showText('拖动控制点缩放，角点外侧旋转，按 Enter 或 Esc 完成');
    }
    catch { }
}
function endRefImageTransformMode(refImageId) {
    const view = refImageViews.get(refImageId);
    if (!view)
        return;
    refTransformActive = false;
    view.wrap.classList.remove('transforming');
    view.transformLayer.innerHTML = '';
    if (view._transformHandlers) {
        document.removeEventListener('mousemove', view._transformHandlers.onMouseMove);
        document.removeEventListener('mouseup', view._transformHandlers.onMouseUp);
        document.removeEventListener('keydown', view._transformHandlers.onKeyDown);
        view._transformHandlers = null;
    }
    try {
        bubble.showText('变换完成');
    }
    catch { }
}
let syncInteractiveRectsQueued = false;
function scheduleSyncInteractiveRects() {
    if (syncInteractiveRectsQueued)
        return;
    syncInteractiveRectsQueued = true;
    try {
        requestAnimationFrame(() => {
            syncInteractiveRectsQueued = false;
            try {
                syncRecordInteractiveRects();
            }
            catch { }
            try {
                syncRefInteractiveRects();
            }
            catch { }
        });
    }
    catch {
        syncInteractiveRectsQueued = false;
        try {
            syncRecordInteractiveRects();
        }
        catch { }
        try {
            syncRefInteractiveRects();
        }
        catch { }
    }
}
function ensureShotEditor() {
    if (shotEditorReady)
        return;
    shotEditorRoot = document.getElementById('screenshot-editor');
    shotBackdrop = document.getElementById('screenshot-editor-backdrop');
    shotStage = document.getElementById('screenshot-editor-stage');
    shotToolbar = document.getElementById('screenshot-toolbar');
    shotPinBtn = document.getElementById('screenshot-toolbar-pin');
    shotColorInput = document.getElementById('shot-color');
    shotSizeInput = document.getElementById('shot-size');
    shotTextSizeInput = document.getElementById('shot-text-size');
    shotEraserInput = document.getElementById('shot-eraser');
    shotBaseCanvas = document.getElementById('screenshot-base-canvas');
    shotAnnotCanvas = document.getElementById('screenshot-annot-canvas');
    shotTempCanvas = document.getElementById('screenshot-temp-canvas');
    shotBaseCtx = shotBaseCanvas && shotBaseCanvas.getContext('2d', { alpha: true });
    shotAnnotCtx = shotAnnotCanvas && shotAnnotCanvas.getContext('2d', { alpha: true });
    shotTempCtx = shotTempCanvas && shotTempCanvas.getContext('2d', { alpha: true });
    try {
        if (shotColorInput)
            shotColorInput.value = strokeColor;
        if (shotSizeInput)
            shotSizeInput.value = String(lineWidth);
        if (shotEraserInput)
            shotEraserInput.value = String(eraserWidth);
        if (shotToolbarColorSwatch)
            shotToolbarColorSwatch.style.background = strokeColor;
        if (shotTextSizeInput)
            shotTextSizeInput.value = String(shotTextFontSizeCss);
    }
    catch { }
    try {
        ensureShotColorPicker();
    }
    catch { }
    if (shotToolbar) {
        shotToolbar.addEventListener('click', async (e) => {
            const btn = e.target && e.target.closest ? e.target.closest('button') : null;
            if (!btn)
                return;
            const nextTool = btn.dataset && btn.dataset.shotTool ? String(btn.dataset.shotTool) : '';
            const action = btn.dataset && btn.dataset.shotAction ? String(btn.dataset.shotAction) : '';
            if (nextTool) {
                if (nextTool === shotTool) {
                    setShotTool('');
                    shotIsDrawing = false;
                    shotClearTemp();
                    setEraserCursorVisible(false);
                    return;
                }
                setShotTool(nextTool);
                return;
            }
            if (action === 'undo') {
                shotUndo();
                return;
            }
            if (action === 'clear') {
                shotClear();
                return;
            }
            if (action === 'close') {
                closeShotEditor();
                return;
            }
            if (action === 'save') {
                await shotSave();
                return;
            }
            if (action === 'done') {
                await shotDone();
                return;
            }
            if (action === 'pin') {
                await shotPin();
                return;
            }
        });
    }
    if (shotColorInput)
        shotColorInput.addEventListener('input', (e) => {
            strokeColor = e && e.target ? String(e.target.value || strokeColor) : strokeColor;
            try {
                if (colorInput)
                    colorInput.value = strokeColor;
            }
            catch { }
            try {
                if (toolbarColorSwatch)
                    toolbarColorSwatch.style.background = strokeColor;
            }
            catch { }
            try {
                if (shotToolbarColorSwatch)
                    shotToolbarColorSwatch.style.background = strokeColor;
            }
            catch { }
            syncDrawVarsToSettings();
            try {
                const box = getShotActiveOrSelectedTextBox();
                if (box)
                    applyShotTextStyleToBox(box, { color: strokeColor });
            }
            catch { }
        });
    if (shotSizeInput)
        shotSizeInput.addEventListener('input', (e) => {
            lineWidth = Math.max(1, Math.round(Number(e && e.target ? e.target.value : lineWidth) || lineWidth));
            try {
                if (sizeInput)
                    sizeInput.value = String(lineWidth);
            }
            catch { }
            syncDrawVarsToSettings();
        });
    if (shotTextSizeInput)
        shotTextSizeInput.addEventListener('input', (e) => {
            shotTextFontSizeCss = clamp(Math.round(Number(e && e.target ? e.target.value : shotTextFontSizeCss) || shotTextFontSizeCss), 10, 80);
            try {
                const box = getShotActiveOrSelectedTextBox();
                if (box)
                    applyShotTextStyleToBox(box, { fontSizeCss: shotTextFontSizeCss });
            }
            catch { }
            scheduleSettingsCssVars({ '--shot-text-font-size': `${shotTextFontSizeCss}px` });
        });
    if (shotEraserInput)
        shotEraserInput.addEventListener('input', (e) => {
            eraserWidth = Math.max(2, Math.round(Number(e && e.target ? e.target.value : eraserWidth) || eraserWidth));
            try {
                if (smoothInput)
                    smoothInput.value = String(eraserWidth);
            }
            catch { }
            syncEraserCursorSize();
            syncDrawVarsToSettings();
        });
    const onDown = (e) => shotOnPointerDown(e);
    const onMove = (e) => shotOnPointerMove(e);
    const onUp = (e) => shotOnPointerUp(e);
    if (shotAnnotCanvas) {
        shotAnnotCanvas.addEventListener('pointerdown', onDown);
        shotAnnotCanvas.addEventListener('pointermove', onMove);
        shotAnnotCanvas.addEventListener('pointerup', onUp);
        shotAnnotCanvas.addEventListener('pointercancel', onUp);
        shotAnnotCanvas.addEventListener('pointerleave', onUp);
        shotAnnotCanvas.addEventListener('contextmenu', (e) => { try {
            e.preventDefault();
        }
        catch { } });
        shotAnnotCanvas.addEventListener('mouseenter', (e) => {
            try {
                lastPointerPos = { x: e.clientX, y: e.clientY };
            }
            catch { }
            refreshEraserCursor(lastPointerPos);
        });
        shotAnnotCanvas.addEventListener('mouseleave', () => setEraserCursorVisible(false));
    }
    if (shotTempCanvas) {
        shotTempCanvas.addEventListener('pointerdown', onDown);
        shotTempCanvas.addEventListener('pointermove', onMove);
        shotTempCanvas.addEventListener('pointerup', onUp);
        shotTempCanvas.addEventListener('pointercancel', onUp);
        shotTempCanvas.addEventListener('pointerleave', onUp);
        shotTempCanvas.addEventListener('contextmenu', (e) => { try {
            e.preventDefault();
        }
        catch { } });
        shotTempCanvas.addEventListener('mouseenter', (e) => {
            try {
                lastPointerPos = { x: e.clientX, y: e.clientY };
            }
            catch { }
            refreshEraserCursor(lastPointerPos);
        });
        shotTempCanvas.addEventListener('mouseleave', () => setEraserCursorVisible(false));
    }
    if (shotBackdrop)
        shotBackdrop.addEventListener('mousedown', () => { });
    window.addEventListener('resize', () => { if (shotEditorActive)
        positionShotToolbar(); });
    shotEditorReady = true;
}
function clamp(n, min, max) { return Math.max(min, Math.min(max, n)); }
function pressureFromEvent(e) {
    const pt = e && typeof e.pointerType === 'string' ? e.pointerType : '';
    const p = e && typeof e.pressure === 'number' ? e.pressure : null;
    if (pt === 'pen' && p !== null && Number.isFinite(p))
        return clamp(p, 0, 1);
    return 1;
}
function strokeWidthFromPressure(baseWidth, pressure) {
    const base = Math.max(0.5, Number(baseWidth) || 0);
    const p = clamp(Number(pressure) || 1, 0, 1);
    const minFactor = 0.2;
    return Math.max(0.5, base * (minFactor + p * (1 - minFactor)));
}
function pressureAtRatio(points, ratio) {
    const pts = Array.isArray(points) ? points : [];
    const n = pts.length;
    if (n <= 1)
        return 1;
    const r = clamp(Number(ratio) || 0, 0, 1);
    const pos = r * (n - 1);
    const i = Math.floor(pos);
    const t = pos - i;
    const p0 = clamp(Number((pts[i] || {}).p) || 1, 0, 1);
    const p1 = clamp(Number((pts[Math.min(n - 1, i + 1)] || {}).p) || p0, 0, 1);
    return p0 + (p1 - p0) * t;
}
function drawPressurePolyline(ctx, pts, baseWidth, pressureFn) {
    if (!ctx || !pts || pts.length < 2)
        return;
    const denom = Math.max(1, pts.length - 1);
    for (let i = 1; i < pts.length; i++) {
        const a = pts[i - 1];
        const b = pts[i];
        const r = i / denom;
        const p = typeof pressureFn === 'function' ? pressureFn(r) : 1;
        ctx.lineWidth = strokeWidthFromPressure(baseWidth, p);
        ctx.beginPath();
        ctx.moveTo(a.x, a.y);
        ctx.lineTo(b.x, b.y);
        ctx.stroke();
    }
}
function setShotTool(next) {
    const nextTool = (next === null || typeof next === 'undefined') ? 'pen' : String(next);
    if (shotTextArea && shotTextArea.isConnected && nextTool !== 'text')
        cleanupTextArea(true);
    shotTool = nextTool;
    if (shotToolbar) {
        shotToolbar.querySelectorAll('button[data-shot-tool]').forEach((b) => {
            b.classList.toggle('active', String(b.dataset && b.dataset.shotTool) === shotTool);
        });
    }
    refreshEraserCursor();
}
function shotClearTemp() {
    if (!shotTempCtx || !shotTempCanvas)
        return;
    shotTempCtx.clearRect(0, 0, shotTempCanvas.width, shotTempCanvas.height);
}
function listShotTextBoxes() {
    if (!shotStage)
        return [];
    return Array.from(shotStage.querySelectorAll('.shot-textbox[data-shot-text="1"]'));
}
function setShotSelectedTextBox(box) {
    const prev = shotSelectedTextBox;
    if (prev && prev.isConnected) {
        try {
            prev.classList.remove('selected');
        }
        catch { }
    }
    shotSelectedTextBox = (box && box.isConnected) ? box : null;
    if (shotSelectedTextBox) {
        try {
            shotSelectedTextBox.classList.add('selected');
        }
        catch { }
    }
}
function clearShotSelectedTextBox() { setShotSelectedTextBox(null); }
function getShotActiveOrSelectedTextBox() {
    if (shotTextBox && shotTextBox.isConnected)
        return shotTextBox;
    if (shotSelectedTextBox && shotSelectedTextBox.isConnected)
        return shotSelectedTextBox;
    return null;
}
function syncShotTextControlsFromBox(box) {
    const b = (box && box.isConnected) ? box : null;
    if (!b)
        return;
    const ta = b.querySelector('textarea.shot-textarea');
    const color = (ta && ta.isConnected) ? String(ta.style.color || strokeColor) : String(b.dataset.color || strokeColor);
    const fontSizeCss = (ta && ta.isConnected)
        ? (Number(String(ta.style.fontSize || '').replace(/px$/i, '')) || shotTextFontSizeCss)
        : (Number(b.dataset.fontSizeCss) || shotTextFontSizeCss);
    try {
        if (color) {
            strokeColor = color;
            if (shotColorInput)
                shotColorInput.value = color;
            if (colorInput)
                colorInput.value = color;
            if (toolbarColorSwatch)
                toolbarColorSwatch.style.background = color;
            if (shotToolbarColorSwatch)
                shotToolbarColorSwatch.style.background = color;
        }
    }
    catch { }
    try {
        if (fontSizeCss) {
            shotTextFontSizeCss = clamp(Math.round(Number(fontSizeCss) || shotTextFontSizeCss), 10, 80);
            if (shotTextSizeInput)
                shotTextSizeInput.value = String(shotTextFontSizeCss);
        }
    }
    catch { }
    try {
        syncDrawVarsToSettings();
    }
    catch { }
}
function applyShotTextStyleToBox(box, next) {
    const b = (box && box.isConnected) ? box : null;
    if (!b)
        return;
    const n = (next && typeof next === 'object') ? next : {};
    const ta = b.querySelector('textarea.shot-textarea');
    const staticDiv = b.querySelector('.shot-text-static');
    if (typeof n.color === 'string' && n.color) {
        try {
            b.dataset.color = String(n.color);
        }
        catch { }
        try {
            if (ta && ta.isConnected)
                ta.style.color = String(n.color);
        }
        catch { }
        try {
            if (staticDiv && staticDiv.isConnected)
                staticDiv.style.color = String(n.color);
        }
        catch { }
    }
    if (typeof n.fontSizeCss === 'number' && Number.isFinite(n.fontSizeCss)) {
        const fs = Math.max(10, Math.round(Number(n.fontSizeCss) || shotTextFontSizeCss));
        try {
            b.dataset.fontSizeCss = String(fs);
        }
        catch { }
        try {
            if (ta && ta.isConnected)
                ta.style.fontSize = `${fs}px`;
        }
        catch { }
        try {
            if (staticDiv && staticDiv.isConnected)
                staticDiv.style.fontSize = `${fs}px`;
        }
        catch { }
        try {
            shotTextFontSizeCss = clamp(fs, 10, 80);
        }
        catch { }
    }
    if (ta && ta.isConnected) {
        try {
            requestAnimationFrame(() => { try {
                ta.dispatchEvent(new Event('input'));
            }
            catch { } });
        }
        catch { }
    }
}
function snapshotShotTexts() {
    const boxes = listShotTextBoxes();
    return boxes.map((box) => {
        const r = box.getBoundingClientRect();
        const sr = shotStage ? shotStage.getBoundingClientRect() : { left: 0, top: 0 };
        const left = Math.round(r.left - sr.left);
        const top = Math.round(r.top - sr.top);
        const width = Math.round(r.width);
        const height = Math.round(r.height);
        const ta = box.querySelector('textarea.shot-textarea');
        const hasCommitted = !!String(box.dataset.text || '').trim();
        if (ta && !hasCommitted)
            return null;
        const text = (ta && hasCommitted) ? String(box.dataset.text || '') : (ta ? String(ta.value || '') : String(box.dataset.text || ''));
        const color = (ta && hasCommitted) ? String(box.dataset.color || '') : (ta ? String(ta.style.color || '') : String(box.dataset.color || ''));
        const fontSizeCss = (ta && hasCommitted) ? (Number(box.dataset.fontSizeCss) || 0) : (ta ? (Number(String(ta.style.fontSize || '').replace(/px$/i, '')) || 0) : (Number(box.dataset.fontSizeCss) || 0));
        const id = String(box.dataset.textId || '');
        return { id, left, top, width, height, text, color, fontSizeCss };
    }).filter(Boolean);
}
function removeAllShotTexts() {
    try {
        cleanupTextArea(false);
    }
    catch { }
    try {
        clearShotSelectedTextBox();
    }
    catch { }
    const boxes = listShotTextBoxes();
    for (const b of boxes) {
        try {
            b.remove();
        }
        catch { }
    }
}
function ensureShotTextBoxBound(box) {
    if (!box || box.dataset.shotTextBound === '1')
        return;
    box.dataset.shotTextBound = '1';
    if (!box.querySelector('button.shot-text-del')) {
        const delBtn = document.createElement('button');
        delBtn.type = 'button';
        delBtn.className = 'shot-text-del';
        delBtn.title = '删除';
        delBtn.setAttribute('aria-label', '删除');
        delBtn.textContent = '×';
        delBtn.addEventListener('pointerdown', (e) => {
            try {
                if (typeof e.stopImmediatePropagation === 'function')
                    e.stopImmediatePropagation();
                else if (typeof e.stopPropagation === 'function')
                    e.stopPropagation();
            }
            catch { }
            try {
                e.preventDefault();
            }
            catch { }
        });
        delBtn.addEventListener('click', (e) => {
            try {
                if (typeof e.stopImmediatePropagation === 'function')
                    e.stopImmediatePropagation();
                else if (typeof e.stopPropagation === 'function')
                    e.stopPropagation();
            }
            catch { }
            try {
                e.preventDefault();
            }
            catch { }
            try {
                shotPushHistory();
            }
            catch { }
            if (shotTextBox === box && shotTextArea && shotTextArea.isConnected) {
                try {
                    cleanupTextArea(false);
                }
                catch { }
                return;
            }
            try {
                if (shotSelectedTextBox === box)
                    clearShotSelectedTextBox();
            }
            catch { }
            try {
                box.remove();
            }
            catch { }
        });
        box.appendChild(delBtn);
    }
    const edgeDirAt = (ev) => {
        const target = ev && ev.target ? ev.target : null;
        if (target && target.closest && target.closest('.shot-text-handle'))
            return String((target.closest('.shot-text-handle').dataset || {}).dir || '');
        const r = box.getBoundingClientRect();
        const x = Number(ev.clientX) - r.left;
        const y = Number(ev.clientY) - r.top;
        const m = 8;
        const onW = x <= m;
        const onE = x >= r.width - m;
        const onN = y <= m;
        const onS = y >= r.height - m;
        let d = '';
        if (onN)
            d += 'n';
        else if (onS)
            d += 's';
        if (onW)
            d += 'w';
        else if (onE)
            d += 'e';
        return d;
    };
    const cursorForDir = (dir) => {
        switch (String(dir || '')) {
            case 'n':
            case 's': return 'ns-resize';
            case 'e':
            case 'w': return 'ew-resize';
            case 'ne':
            case 'sw': return 'nesw-resize';
            case 'nw':
            case 'se': return 'nwse-resize';
            default: return box.classList.contains('static') ? 'move' : 'text';
        }
    };
    box.addEventListener('mousemove', (e) => {
        if (!e)
            return;
        const dir = edgeDirAt(e);
        box.style.cursor = cursorForDir(dir);
    });
    box.addEventListener('pointerdown', (e) => {
        if (!e || e.button !== 0)
            return;
        if (!shotStage)
            return;
        if (edgeDirAt(e))
            return;
        const inTa = e.target && e.target.closest ? e.target.closest('textarea.shot-textarea') : null;
        if (inTa)
            return;
        const inDel = e.target && e.target.closest ? e.target.closest('button.shot-text-del') : null;
        if (inDel)
            return;
        const hasCommitted = !!String(box.dataset.text || '').trim();
        if (!hasCommitted)
            return;
        try {
            setShotSelectedTextBox(box);
        }
        catch { }
        try {
            syncShotTextControlsFromBox(box);
        }
        catch { }
        const stageB = shotStage.getBoundingClientRect();
        const boxB = box.getBoundingClientRect();
        const startX = Number(e.clientX) || 0;
        const startY = Number(e.clientY) || 0;
        const startLeft = boxB.left - stageB.left;
        const startTop = boxB.top - stageB.top;
        const maxLeft = Math.max(0, stageB.width - boxB.width);
        const maxTop = Math.max(0, stageB.height - boxB.height);
        const ta = box.querySelector('textarea.shot-textarea');
        try {
            if (ta && ta.isConnected)
                ta.focus({ preventScroll: true });
        }
        catch {
            try {
                if (ta)
                    ta.focus();
            }
            catch { }
        }
        try {
            box.setPointerCapture(e.pointerId);
            if (typeof e.stopImmediatePropagation === 'function')
                e.stopImmediatePropagation();
            else if (typeof e.stopPropagation === 'function')
                e.stopPropagation();
            e.preventDefault();
        }
        catch { }
        let pushed = false;
        const onMove = (mv) => {
            const dx = (Number(mv.clientX) || 0) - startX;
            const dy = (Number(mv.clientY) || 0) - startY;
            if (!pushed && (Math.abs(dx) > 1 || Math.abs(dy) > 1)) {
                pushed = true;
                try {
                    shotPushHistory();
                }
                catch { }
            }
            const nl = clamp(startLeft + dx, 0, maxLeft);
            const nt = clamp(startTop + dy, 0, maxTop);
            box.style.left = `${Math.round(nl)}px`;
            box.style.top = `${Math.round(nt)}px`;
        };
        const onUp = () => {
            window.removeEventListener('pointermove', onMove);
            window.removeEventListener('pointerup', onUp);
            window.removeEventListener('pointercancel', onUp);
            try {
                box.releasePointerCapture(e.pointerId);
            }
            catch { }
        };
        window.addEventListener('pointermove', onMove);
        window.addEventListener('pointerup', onUp);
        window.addEventListener('pointercancel', onUp);
    });
    box.addEventListener('dblclick', (e) => {
        try {
            if (e)
                e.preventDefault();
        }
        catch { }
        if (!box.classList.contains('static'))
            return;
        try {
            cleanupTextArea(true);
        }
        catch { }
        const text = String(box.dataset.text || '');
        const color = String(box.dataset.color || strokeColor);
        const fontSizeCss = Number(box.dataset.fontSizeCss) || 14;
        const existingStatic = box.querySelector('.shot-text-static');
        try {
            if (existingStatic)
                existingStatic.remove();
        }
        catch { }
        box.classList.remove('static');
        const ta = document.createElement('textarea');
        ta.className = 'shot-textarea';
        ta.style.fontSize = `${Math.max(12, Math.round(fontSizeCss))}px`;
        ta.style.color = color;
        ta.setAttribute('spellcheck', 'false');
        ta.value = text;
        box.appendChild(ta);
        try {
            setShotSelectedTextBox(box);
        }
        catch { }
        try {
            syncShotTextControlsFromBox(box);
        }
        catch { }
        setupShotTextEditing(box, ta);
    });
}
function restoreShotTexts(snapshot) {
    removeAllShotTexts();
    if (!shotStage)
        return;
    const list = Array.isArray(snapshot) ? snapshot : [];
    for (const s0 of list) {
        const s = (s0 && typeof s0 === 'object') ? s0 : null;
        if (!s)
            continue;
        const box = document.createElement('div');
        box.className = 'shot-textbox static';
        box.dataset.shotText = '1';
        box.dataset.textId = String(s.id || `t_${Date.now()}_${++shotTextIdSeq}`);
        box.dataset.text = String(s.text || '');
        box.dataset.color = String(s.color || strokeColor);
        box.dataset.fontSizeCss = String(Number(s.fontSizeCss) || 14);
        box.style.left = `${Math.round(Number(s.left) || 0)}px`;
        box.style.top = `${Math.round(Number(s.top) || 0)}px`;
        box.style.width = `${Math.max(120, Math.round(Number(s.width) || 120))}px`;
        box.style.height = `${Math.max(34, Math.round(Number(s.height) || 34))}px`;
        const staticDiv = document.createElement('div');
        staticDiv.className = 'shot-text-static';
        staticDiv.style.fontSize = `${Math.max(12, Math.round(Number(s.fontSizeCss) || 14))}px`;
        staticDiv.style.color = String(s.color || strokeColor);
        staticDiv.textContent = String(s.text || '');
        box.appendChild(staticDiv);
        for (const dir of ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w']) {
            const h = document.createElement('div');
            h.className = 'shot-text-handle';
            h.dataset.dir = dir;
            box.appendChild(h);
        }
        ensureShotTextBoxBound(box);
        shotStage.appendChild(box);
    }
}
function shotPushHistory() {
    if (!shotAnnotCtx || !shotAnnotCanvas)
        return;
    const texts = snapshotShotTexts();
    try {
        const img = shotAnnotCtx.getImageData(0, 0, shotAnnotCanvas.width, shotAnnotCanvas.height);
        shotHistory.push({ img, texts });
    }
    catch { }
}
function shotUndo() {
    if (!shotAnnotCtx || !shotAnnotCanvas)
        return;
    if (!shotHistory.length)
        return;
    const item = shotHistory.pop();
    const img = item && item.img ? item.img : item;
    shotClearTemp();
    shotAnnotCtx.clearRect(0, 0, shotAnnotCanvas.width, shotAnnotCanvas.height);
    try {
        shotAnnotCtx.putImageData(img, 0, 0);
    }
    catch { }
    try {
        if (item && item.texts)
            restoreShotTexts(item.texts);
    }
    catch { }
}
function shotClear() {
    if (!shotAnnotCtx || !shotAnnotCanvas)
        return;
    shotAnnotCtx.clearRect(0, 0, shotAnnotCanvas.width, shotAnnotCanvas.height);
    shotClearTemp();
    shotHistory = [];
    removeAllShotTexts();
}
function posOnShotFromEvent(e) {
    if (!shotStage || !shotAnnotCanvas)
        return { x: 0, y: 0 };
    const r = shotStage.getBoundingClientRect();
    const sx = shotAnnotCanvas.width / Math.max(1, r.width);
    const sy = shotAnnotCanvas.height / Math.max(1, r.height);
    const x = (Number(e && e.clientX) - r.left) * sx;
    const y = (Number(e && e.clientY) - r.top) * sy;
    return { x: clamp(x, 0, shotAnnotCanvas.width), y: clamp(y, 0, shotAnnotCanvas.height) };
}
let activeShotPointerId = null;
let shotStrokeUsePressure = false;
let shotUsingWindowPointerListeners = false;
function bindShotWindowPointerListeners() {
    if (shotUsingWindowPointerListeners)
        return;
    shotUsingWindowPointerListeners = true;
    window.addEventListener('pointermove', shotOnPointerMove);
    window.addEventListener('pointerup', shotOnPointerUp);
    window.addEventListener('pointercancel', shotOnPointerUp);
}
function unbindShotWindowPointerListeners() {
    if (!shotUsingWindowPointerListeners)
        return;
    shotUsingWindowPointerListeners = false;
    window.removeEventListener('pointermove', shotOnPointerMove);
    window.removeEventListener('pointerup', shotOnPointerUp);
    window.removeEventListener('pointercancel', shotOnPointerUp);
}
function beginShotStroke(x, y, pressure, pointerType) {
    shotPathPoints = [{ x, y, p: pressure }];
    shotStrokeUsePressure = String(pointerType || '') === 'pen';
}
function extendShotStroke(x, y, pressure) {
    if (!shotTempCtx)
        return;
    shotPathPoints.push({ x, y, p: pressure });
    shotClearTemp();
    setupStrokeStyle(shotTempCtx);
    const smoothed = smoothPolyline(shotPathPoints.map((p) => ({ x: p.x, y: p.y })), smoothing);
    if (shotStrokeUsePressure)
        drawPressurePolyline(shotTempCtx, smoothed, lineWidth, (r) => pressureAtRatio(shotPathPoints, r));
    else
        drawPolyline(shotTempCtx, smoothed);
}
function commitShotStroke() {
    if (!shotAnnotCtx)
        return;
    shotClearTemp();
    setupStrokeStyle(shotAnnotCtx);
    const smoothed = smoothPolyline(shotPathPoints.map((p) => ({ x: p.x, y: p.y })), smoothing);
    shotPushHistory();
    if (shotStrokeUsePressure)
        drawPressurePolyline(shotAnnotCtx, smoothed, lineWidth, (r) => pressureAtRatio(shotPathPoints, r));
    else
        drawPolyline(shotAnnotCtx, smoothed);
    shotPathPoints = [];
    shotStrokeUsePressure = false;
}
function shotMosaicSize() {
    const lw = Math.max(1, Math.round(Number(lineWidth) || 4));
    const raw = Math.max(8, Math.round(lw * 4));
    const step = 4;
    return Math.max(8, Math.round(raw / step) * step);
}
function applyMosaicAt(x, y) {
    if (!shotAnnotCtx || !shotBaseCtx || !shotBaseCanvas)
        return;
    const size = shotMosaicSize();
    const baseBx = Math.floor(x / size) * size;
    const baseBy = Math.floor(y / size) * size;
    if (!shotMosaicTouched)
        shotMosaicTouched = new Set();
    const drawCell = (bx, by) => {
        if (bx < 0 || by < 0 || bx >= shotBaseCanvas.width || by >= shotBaseCanvas.height)
            return;
        const key = `${bx},${by}`;
        if (shotMosaicTouched.has(key))
            return;
        shotMosaicTouched.add(key);
        const cx = clamp(bx + Math.floor(size / 2), 0, shotBaseCanvas.width - 1);
        const cy = clamp(by + Math.floor(size / 2), 0, shotBaseCanvas.height - 1);
        let d = null;
        try {
            d = shotBaseCtx.getImageData(cx, cy, 1, 1).data;
        }
        catch {
            d = null;
        }
        if (!d)
            return;
        shotAnnotCtx.fillStyle = `rgba(${d[0]},${d[1]},${d[2]},${(d[3] / 255).toFixed(3)})`;
        shotAnnotCtx.fillRect(bx, by, size, size);
        try {
            shotAnnotCtx.save();
            shotAnnotCtx.strokeStyle = 'rgba(0,0,0,0.18)';
            shotAnnotCtx.lineWidth = 1;
            const w = Math.max(0, size - 1);
            shotAnnotCtx.strokeRect(bx + 0.5, by + 0.5, w, w);
            shotAnnotCtx.restore();
        }
        catch { }
    };
    drawCell(baseBx, baseBy);
    drawCell(baseBx + size, baseBy);
    drawCell(baseBx, baseBy + size);
    drawCell(baseBx + size, baseBy + size);
}
function cleanupTextArea(commit) {
    const ta = shotTextArea;
    if (!ta)
        return;
    const text = String(ta.value || '');
    const box = shotTextBox;
    const fontSizeCss = Math.max(12, Math.round(Number(String(ta.style.fontSize || '').replace(/px$/i, '')) || 14));
    if (commit) {
        try {
            shotPushHistory();
        }
        catch { }
    }
    if (!commit || !text.trim() || !box || !box.isConnected) {
        try {
            if (shotSelectedTextBox === box)
                clearShotSelectedTextBox();
        }
        catch { }
        try {
            if (box)
                box.remove();
            else
                ta.remove();
        }
        catch { }
        shotTextArea = null;
        shotTextBox = null;
        shotTextAt = null;
        shotTextCaret = 0;
        shotTextResizing = false;
        return;
    }
    const color = String(ta.style.color || strokeColor);
    const id = String(box.dataset.textId || `t_${Date.now()}_${++shotTextIdSeq}`);
    box.dataset.shotText = '1';
    box.dataset.textId = id;
    box.dataset.text = text;
    box.dataset.color = color;
    box.dataset.fontSizeCss = String(fontSizeCss);
    box.classList.add('static');
    const prevStatic = box.querySelector('.shot-text-static');
    try {
        if (prevStatic)
            prevStatic.remove();
    }
    catch { }
    try {
        ta.remove();
    }
    catch { }
    const staticDiv = document.createElement('div');
    staticDiv.className = 'shot-text-static';
    staticDiv.style.fontSize = `${fontSizeCss}px`;
    staticDiv.style.color = color;
    staticDiv.textContent = text;
    box.appendChild(staticDiv);
    ensureShotTextBoxBound(box);
    try {
        setShotSelectedTextBox(box);
    }
    catch { }
    try {
        syncShotTextControlsFromBox(box);
    }
    catch { }
    shotTextArea = null;
    shotTextBox = null;
    shotTextAt = null;
    shotTextCaret = 0;
    shotTextResizing = false;
}
function setupShotTextEditing(box, ta) {
    if (!shotStage || !shotAnnotCanvas)
        return;
    box.dataset.shotText = '1';
    if (!box.dataset.textId)
        box.dataset.textId = `t_${Date.now()}_${++shotTextIdSeq}`;
    if (!box.querySelector('.shot-text-handle')) {
        for (const dir of ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w']) {
            const h = document.createElement('div');
            h.className = 'shot-text-handle';
            h.dataset.dir = dir;
            box.appendChild(h);
        }
    }
    const stageRect = shotStage.getBoundingClientRect();
    const boxRect = box.getBoundingClientRect();
    const taFontCssRaw = Number(String(ta.style.fontSize || '').replace(/px$/i, '')) || 0;
    if (!taFontCssRaw) {
        try {
            ta.style.fontSize = `${shotTextFontSizeCss}px`;
        }
        catch { }
    }
    try {
        const v = Number(String(ta.style.fontSize || '').replace(/px$/i, '')) || shotTextFontSizeCss;
        shotTextFontSizeCss = clamp(Math.round(v), 10, 80);
        if (shotTextSizeInput)
            shotTextSizeInput.value = String(shotTextFontSizeCss);
    }
    catch { }
    shotTextArea = ta;
    shotTextBox = box;
    try {
        setShotSelectedTextBox(box);
    }
    catch { }
    try {
        syncShotTextControlsFromBox(box);
    }
    catch { }
    shotTextAt = posOnShotFromEvent({ clientX: boxRect.left + 8, clientY: boxRect.top + 6 });
    shotTextCaret = 0;
    const edgeDirAt = (ev) => {
        const target = ev && ev.target ? ev.target : null;
        if (target && target.dataset && target.dataset.dir)
            return String(target.dataset.dir);
        const r = box.getBoundingClientRect();
        const x = Number(ev.clientX) - r.left;
        const y = Number(ev.clientY) - r.top;
        const m = 8;
        const onW = x <= m;
        const onE = x >= r.width - m;
        const onN = y <= m;
        const onS = y >= r.height - m;
        let d = '';
        if (onN)
            d += 'n';
        else if (onS)
            d += 's';
        if (onW)
            d += 'w';
        else if (onE)
            d += 'e';
        return d;
    };
    const cursorForDir = (dir) => {
        switch (String(dir || '')) {
            case 'n':
            case 's': return 'ns-resize';
            case 'e':
            case 'w': return 'ew-resize';
            case 'ne':
            case 'sw': return 'nesw-resize';
            case 'nw':
            case 'se': return 'nwse-resize';
            default: return 'text';
        }
    };
    if (box.dataset.shotTextResizeBound !== '1') {
        box.dataset.shotTextResizeBound = '1';
        const beginResize = (dir, ev) => {
            if (!shotStage)
                return;
            const stageB = shotStage.getBoundingClientRect();
            const boxB = box.getBoundingClientRect();
            const start = {
                dir: String(dir || ''),
                x: Number(ev.clientX) || 0,
                y: Number(ev.clientY) || 0,
                left: boxB.left - stageB.left,
                top: boxB.top - stageB.top,
                w: boxB.width,
                h: boxB.height,
                stageW: stageB.width,
                stageH: stageB.height,
            };
            const minW = 120;
            const minH = 34;
            shotTextResizing = true;
            try {
                ev.preventDefault();
            }
            catch { }
            try {
                ta.focus({ preventScroll: true });
            }
            catch {
                try {
                    ta.focus();
                }
                catch { }
            }
            const onMove = (mv) => {
                const dx = (Number(mv.clientX) || 0) - start.x;
                const dy = (Number(mv.clientY) || 0) - start.y;
                let nl = start.left;
                let nt = start.top;
                let nw = start.w;
                let nh = start.h;
                if (start.dir.includes('e'))
                    nw = start.w + dx;
                if (start.dir.includes('s'))
                    nh = start.h + dy;
                if (start.dir.includes('w')) {
                    nw = start.w - dx;
                    nl = start.left + dx;
                }
                if (start.dir.includes('n')) {
                    nh = start.h - dy;
                    nt = start.top + dy;
                }
                nw = clamp(nw, minW, start.stageW - nl);
                nh = clamp(nh, minH, start.stageH - nt);
                nl = clamp(nl, 0, start.stageW - nw);
                nt = clamp(nt, 0, start.stageH - nh);
                box.style.left = `${Math.round(nl)}px`;
                box.style.top = `${Math.round(nt)}px`;
                box.style.width = `${Math.round(nw)}px`;
                box.style.height = `${Math.round(nh)}px`;
            };
            const onUp = () => {
                shotTextResizing = false;
                window.removeEventListener('pointermove', onMove);
                window.removeEventListener('pointerup', onUp);
                window.removeEventListener('pointercancel', onUp);
            };
            window.addEventListener('pointermove', onMove);
            window.addEventListener('pointerup', onUp);
            window.addEventListener('pointercancel', onUp);
        };
        box.addEventListener('pointerdown', (e) => {
            if (!e || e.button !== 0)
                return;
            const dir = edgeDirAt(e);
            if (dir) {
                beginResize(dir, e);
                return;
            }
            try {
                ta.focus({ preventScroll: true });
            }
            catch {
                try {
                    ta.focus();
                }
                catch { }
            }
        });
        box.addEventListener('mousemove', (e) => {
            if (!e || shotTextResizing)
                return;
            const dir = edgeDirAt(e);
            box.style.cursor = cursorForDir(dir);
        });
    }
    const autoGrowHeight = () => {
        if (!shotStage || !box.isConnected || !ta.isConnected)
            return;
        const stageB = shotStage.getBoundingClientRect();
        const boxB = box.getBoundingClientRect();
        const topCss = boxB.top - stageB.top;
        const minH = 34;
        const maxH = Math.max(minH, stageB.height - topCss);
        const extra = (ta.scrollHeight || 0) - (ta.clientHeight || 0);
        if (extra <= 0)
            return;
        const nextH = clamp(boxB.height + extra, minH, maxH);
        if (nextH <= boxB.height + 0.5)
            return;
        box.style.height = `${Math.round(nextH)}px`;
        try {
            ta.scrollTop = ta.scrollHeight;
        }
        catch { }
    };
    ta.addEventListener('blur', () => {
        setTimeout(() => {
            if (!ta.isConnected)
                return;
            if (shotTextResizing)
                return;
            if (document.activeElement === ta)
                return;
            const t = String(ta.value || '');
            if (t.trim()) {
                cleanupTextArea(true);
                return;
            }
            if (shotTool !== 'text') {
                cleanupTextArea(false);
                return;
            }
        }, 0);
    });
    ta.addEventListener('input', () => {
        try {
            requestAnimationFrame(() => autoGrowHeight());
        }
        catch {
            setTimeout(() => autoGrowHeight(), 0);
        }
    });
    ta.addEventListener('keydown', (e) => {
        if (!e)
            return;
        if (e.key === 'Escape') {
            try {
                e.preventDefault();
            }
            catch { }
            ;
            cleanupTextArea(false);
            return;
        }
        if (e.key === 'Enter' && !e.shiftKey) {
            try {
                e.preventDefault();
            }
            catch { }
            ;
            cleanupTextArea(true);
            return;
        }
    });
    ta.addEventListener('mousedown', () => {
        try {
            shotTextCaret = String(ta.value || '').length;
            if (typeof ta.setSelectionRange === 'function')
                ta.setSelectionRange(shotTextCaret, shotTextCaret);
        }
        catch { }
    });
    const focusTa = () => {
        try {
            if (!ta.isConnected)
                return;
            try {
                if (window.tt && typeof window.tt.setOverlayFocusable === 'function')
                    window.tt.setOverlayFocusable(true);
            }
            catch { }
            try {
                if (window.tt && typeof window.tt.focusOverlay === 'function')
                    window.tt.focusOverlay();
            }
            catch { }
            try {
                ta.focus({ preventScroll: true });
            }
            catch {
                try {
                    ta.focus();
                }
                catch { }
            }
            try {
                shotTextCaret = String(ta.value || '').length;
                if (typeof ta.setSelectionRange === 'function')
                    ta.setSelectionRange(shotTextCaret, shotTextCaret);
            }
            catch { }
        }
        catch { }
    };
    focusTa();
    try {
        requestAnimationFrame(() => focusTa());
    }
    catch {
        setTimeout(() => focusTa(), 0);
    }
    setTimeout(() => focusTa(), 0);
    try {
        requestAnimationFrame(() => autoGrowHeight());
    }
    catch {
        setTimeout(() => autoGrowHeight(), 0);
    }
}
function startTextAreaAt(clientX, clientY) {
    if (!shotStage || !shotAnnotCanvas)
        return;
    cleanupTextArea(true);
    const stageRect = shotStage.getBoundingClientRect();
    const fontSizeCss = clamp(Math.round(Number(shotTextFontSizeCss) || 28), 10, 80);
    shotTextFontSizeCss = fontSizeCss;
    try {
        if (shotTextSizeInput)
            shotTextSizeInput.value = String(fontSizeCss);
    }
    catch { }
    const baseLeft = clamp(Math.round(clientX - stageRect.left), 0, Math.round(stageRect.width) - 10);
    const baseTop = clamp(Math.round(clientY - stageRect.top), 0, Math.round(stageRect.height) - 10);
    const defaultW = Math.max(160, Math.round(fontSizeCss * 7));
    const defaultH = Math.max(46, Math.round(fontSizeCss * 2.3));
    const left = clamp(baseLeft, 0, Math.round(stageRect.width) - defaultW);
    const top = clamp(baseTop, 0, Math.round(stageRect.height) - defaultH);
    const box = document.createElement('div');
    box.className = 'shot-textbox';
    box.dataset.shotText = '1';
    box.dataset.textId = `t_${Date.now()}_${++shotTextIdSeq}`;
    box.style.left = `${left}px`;
    box.style.top = `${top}px`;
    box.style.width = `${defaultW}px`;
    box.style.height = `${defaultH}px`;
    const ta = document.createElement('textarea');
    ta.className = 'shot-textarea';
    ta.style.fontSize = `${fontSizeCss}px`;
    ta.style.color = strokeColor;
    ta.setAttribute('spellcheck', 'false');
    box.appendChild(ta);
    shotStage.appendChild(box);
    ensureShotTextBoxBound(box);
    setupShotTextEditing(box, ta);
}
function shotOnPointerDown(e) {
    if (!shotEditorActive)
        return;
    if (e && typeof e.button === 'number' && e.button !== 0)
        return;
    activeShotPointerId = e && typeof e.pointerId === 'number' ? e.pointerId : null;
    try {
        if (activeShotPointerId !== null && e && e.target && e.target.setPointerCapture)
            e.target.setPointerCapture(activeShotPointerId);
    }
    catch { }
    bindShotWindowPointerListeners();
    if (shotTextArea && shotTextArea.isConnected) {
        if (shotTool === 'text') {
            cleanupTextArea(true);
            return;
        }
        cleanupTextArea(true);
    }
    try {
        clearShotSelectedTextBox();
    }
    catch { }
    const { x, y } = posOnShotFromEvent(e);
    shotIsDrawing = true;
    shotSx = x;
    shotSy = y;
    if (shotTool === 'pen') {
        beginShotStroke(x, y, pressureFromEvent(e), e && e.pointerType);
        return;
    }
    if (shotTool === 'eraser') {
        if (!shotAnnotCtx)
            return;
        shotPushHistory();
        shotAnnotCtx.globalCompositeOperation = 'destination-out';
        shotAnnotCtx.beginPath();
        shotAnnotCtx.moveTo(x, y);
        return;
    }
    if (shotTool === 'mosaic') {
        shotPushHistory();
        shotMosaicTouched = new Set();
        applyMosaicAt(x, y);
        return;
    }
    if (shotTool === 'text') {
        startTextAreaAt(e.clientX, e.clientY);
        shotIsDrawing = false;
        activeShotPointerId = null;
        return;
    }
    if (shotTool === 'arrow' || shotTool === 'rect' || shotTool === 'circle')
        return;
    shotIsDrawing = false;
    try {
        if (e && typeof e.preventDefault === 'function')
            e.preventDefault();
    }
    catch { }
}
function shotOnPointerMove(e) {
    if (!shotEditorActive)
        return;
    if (shotUsingWindowPointerListeners && e && e.currentTarget !== window)
        return;
    if (activeShotPointerId !== null && e && typeof e.pointerId === 'number' && e.pointerId !== activeShotPointerId)
        return;
    try {
        lastPointerPos = { x: Number(e && e.clientX) || 0, y: Number(e && e.clientY) || 0 };
        refreshEraserCursor(lastPointerPos);
    }
    catch { }
    if (!shotIsDrawing)
        return;
    const { x, y } = posOnShotFromEvent(e);
    if (shotTool === 'pen') {
        extendShotStroke(x, y, pressureFromEvent(e));
        return;
    }
    if (shotTool === 'eraser') {
        if (!shotAnnotCtx)
            return;
        shotAnnotCtx.lineTo(x, y);
        shotAnnotCtx.lineWidth = eraserWidth;
        shotAnnotCtx.lineCap = 'round';
        shotAnnotCtx.lineJoin = 'round';
        shotAnnotCtx.stroke();
        return;
    }
    if (shotTool === 'mosaic') {
        applyMosaicAt(x, y);
        return;
    }
    shotClearTemp();
    if (!shotTempCtx)
        return;
    if (shotTool === 'arrow') {
        drawArrow(shotTempCtx, { x: shotSx, y: shotSy }, { x, y });
        return;
    }
    if (shotTool === 'rect') {
        drawRect(shotTempCtx, { x: shotSx, y: shotSy }, { x, y });
        return;
    }
    if (shotTool === 'circle') {
        drawEllipse(shotTempCtx, { x: shotSx, y: shotSy }, { x, y });
        return;
    }
    try {
        if (e && typeof e.preventDefault === 'function')
            e.preventDefault();
    }
    catch { }
}
function shotOnPointerUp(e) {
    if (!shotEditorActive)
        return;
    if (!shotIsDrawing)
        return;
    if (shotUsingWindowPointerListeners && e && e.currentTarget !== window)
        return;
    if (activeShotPointerId !== null && e && typeof e.pointerId === 'number' && e.pointerId !== activeShotPointerId)
        return;
    const capturedId = activeShotPointerId;
    shotIsDrawing = false;
    activeShotPointerId = null;
    unbindShotWindowPointerListeners();
    try {
        if (capturedId !== null && e && e.target && e.target.releasePointerCapture)
            e.target.releasePointerCapture(capturedId);
    }
    catch { }
    if (shotTool === 'pen') {
        commitShotStroke();
        return;
    }
    if (shotTool === 'eraser') {
        if (!shotAnnotCtx)
            return;
        shotAnnotCtx.globalCompositeOperation = 'source-over';
        shotClearTemp();
        return;
    }
    if (shotTool === 'mosaic') {
        shotMosaicTouched = null;
        return;
    }
    const { x, y } = e ? posOnShotFromEvent(e) : { x: shotSx, y: shotSy };
    if (!shotAnnotCtx)
        return;
    if (shotTool === 'arrow') {
        shotClearTemp();
        shotPushHistory();
        drawArrow(shotAnnotCtx, { x: shotSx, y: shotSy }, { x, y });
        return;
    }
    if (shotTool === 'rect') {
        shotClearTemp();
        shotPushHistory();
        drawRect(shotAnnotCtx, { x: shotSx, y: shotSy }, { x, y });
        return;
    }
    if (shotTool === 'circle') {
        shotClearTemp();
        shotPushHistory();
        drawEllipse(shotAnnotCtx, { x: shotSx, y: shotSy }, { x, y });
        return;
    }
    try {
        if (e && typeof e.preventDefault === 'function')
            e.preventDefault();
    }
    catch { }
}
function positionShotToolbar() {
    if (!shotToolbar || !shotStage)
        return;
    const s = shotStage.getBoundingClientRect();
    const margin = 12;
    const tb = shotToolbar.getBoundingClientRect();
    const w = Math.max(1, tb.width || shotToolbar.offsetWidth || 1);
    const h = Math.max(1, tb.height || shotToolbar.offsetHeight || 1);
    let cx = s.left + s.width / 2;
    let top = s.bottom + margin;
    if (top + h > window.innerHeight - margin)
        top = s.top - margin - h;
    top = clamp(top, margin, window.innerHeight - margin - h);
    cx = clamp(cx, margin + w / 2, window.innerWidth - margin - w / 2);
    shotToolbar.style.left = `${Math.round(cx)}px`;
    shotToolbar.style.top = `${Math.round(top)}px`;
}
async function openShotEditor(pngBase64, regionRect) {
    ensureShotEditor();
    if (!shotEditorRoot || !shotStage || !shotToolbar || !shotBaseCanvas || !shotAnnotCanvas || !shotTempCanvas)
        return;
    shotCaptureTs = Date.now();
    cleanupTextArea(false);
    removeAllShotTexts();
    shotHistory = [];
    shotRect = regionRect && typeof regionRect === 'object' ? regionRect : { x: 0, y: 0, w: 1, h: 1 };
    const r = {
        x: Math.round(Number(shotRect.x) || 0),
        y: Math.round(Number(shotRect.y) || 0),
        w: Math.max(1, Math.round(Number(shotRect.w) || 1)),
        h: Math.max(1, Math.round(Number(shotRect.h) || 1)),
    };
    shotStage.style.left = `${r.x}px`;
    shotStage.style.top = `${r.y}px`;
    shotStage.style.width = `${r.w}px`;
    shotStage.style.height = `${r.h}px`;
    const img = new Image();
    img.decoding = 'async';
    img.src = `data:image/png;base64,${String(pngBase64 || '').trim()}`;
    try {
        if (typeof img.decode === 'function')
            await img.decode();
        else
            await new Promise((resolve, reject) => { img.onload = resolve; img.onerror = reject; });
    }
    catch { }
    const iw = Math.max(1, Number(img.naturalWidth) || r.w);
    const ih = Math.max(1, Number(img.naturalHeight) || r.h);
    shotBaseCanvas.width = iw;
    shotBaseCanvas.height = ih;
    shotAnnotCanvas.width = iw;
    shotAnnotCanvas.height = ih;
    shotTempCanvas.width = iw;
    shotTempCanvas.height = ih;
    try {
        shotBaseCtx.clearRect(0, 0, iw, ih);
    }
    catch { }
    try {
        shotAnnotCtx.clearRect(0, 0, iw, ih);
    }
    catch { }
    try {
        shotTempCtx.clearRect(0, 0, iw, ih);
    }
    catch { }
    try {
        shotBaseCtx.drawImage(img, 0, 0, iw, ih);
    }
    catch { }
    try {
        if (shotColorInput)
            shotColorInput.value = strokeColor;
        if (shotSizeInput)
            shotSizeInput.value = String(lineWidth);
        if (shotEraserInput)
            shotEraserInput.value = String(eraserWidth);
        if (shotToolbarColorSwatch)
            shotToolbarColorSwatch.style.background = strokeColor;
        if (shotTextSizeInput)
            shotTextSizeInput.value = String(shotTextFontSizeCss);
    }
    catch { }
    setShotTool('');
    shotEditorActive = true;
    shotEditorRoot.classList.add('active');
    try {
        if (toolbar)
            toolbar.style.display = 'none';
    }
    catch { }
    try {
        setDrawingMode(false);
    }
    catch { }
    try {
        if (window.tt) {
            if (typeof window.tt.setOverlayFocusable === 'function')
                window.tt.setOverlayFocusable(true);
            if (typeof window.tt.focusOverlay === 'function')
                window.tt.focusOverlay();
            window.tt.setDrawingCapture(true);
        }
    }
    catch { }
    positionShotToolbar();
}
function closeShotEditor() {
    if (!shotEditorActive)
        return;
    ensureShotEditor();
    cleanupTextArea(true);
    removeAllShotTexts();
    shotEditorActive = false;
    shotCaptureTs = 0;
    setEraserCursorVisible(false);
    shotIsDrawing = false;
    shotPathPoints = [];
    shotHistory = [];
    shotRect = null;
    if (shotEditorRoot)
        shotEditorRoot.classList.remove('active');
    try {
        if (toolbar)
            toolbar.style.display = '';
    }
    catch { }
    try {
        if (window.tt) {
            window.tt.setDrawingCapture(false);
            if (typeof window.tt.setOverlayFocusable === 'function')
                window.tt.setOverlayFocusable(false);
        }
    }
    catch { }
}
function mergedShotPngBase64() {
    if (!shotBaseCanvas || !shotAnnotCanvas)
        return '';
    try {
        cleanupTextArea(true);
    }
    catch { }
    const out = document.createElement('canvas');
    out.width = shotBaseCanvas.width;
    out.height = shotBaseCanvas.height;
    const ctx = out.getContext('2d', { alpha: true });
    if (!ctx)
        return '';
    try {
        ctx.drawImage(shotBaseCanvas, 0, 0);
    }
    catch { }
    try {
        ctx.drawImage(shotAnnotCanvas, 0, 0);
    }
    catch { }
    try {
        if (shotStage) {
            const stageRect = shotStage.getBoundingClientRect();
            const sx = out.width / Math.max(1, stageRect.width);
            const sy = out.height / Math.max(1, stageRect.height);
            const padCssX = 8;
            const padCssY = 6;
            const fontFamily = `system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial, "Microsoft YaHei UI", "Microsoft YaHei", sans-serif`;
            const boxes = listShotTextBoxes().filter(b => b.classList.contains('static'));
            for (const box of boxes) {
                const text = String(box.dataset.text || '');
                if (!text.trim())
                    continue;
                const color = String(box.dataset.color || strokeColor);
                const fontSizeCss = Math.max(12, Math.round(Number(box.dataset.fontSizeCss) || 14));
                const boxRect = box.getBoundingClientRect();
                const leftCss = boxRect.left - stageRect.left;
                const topCss = boxRect.top - stageRect.top;
                const wCss = boxRect.width;
                const x = (leftCss + padCssX) * sx;
                const y = (topCss + padCssY) * sy;
                const maxWidth = Math.max(10, (wCss - padCssX * 2) * sx);
                const fontSize = Math.max(12, Math.round(fontSizeCss * sx));
                const lineH = Math.round(fontSize * 1.25);
                ctx.save();
                ctx.fillStyle = color;
                ctx.textBaseline = 'top';
                ctx.font = `${fontSize}px ${fontFamily}`;
                const rawLines = text.replace(/\r\n/g, '\n').split('\n');
                let dy = 0;
                for (const raw of rawLines) {
                    if (!raw) {
                        dy += lineH;
                        continue;
                    }
                    const line = String(raw);
                    let buf = '';
                    for (const ch of Array.from(line)) {
                        const next = buf + ch;
                        if (buf && ctx.measureText(next).width > maxWidth) {
                            ctx.fillText(buf, x, y + dy);
                            dy += lineH;
                            buf = ch;
                        }
                        else {
                            buf = next;
                        }
                    }
                    if (buf) {
                        ctx.fillText(buf, x, y + dy);
                        dy += lineH;
                    }
                }
                ctx.restore();
            }
        }
    }
    catch { }
    let dataUrl = '';
    try {
        dataUrl = out.toDataURL('image/png');
    }
    catch {
        dataUrl = '';
    }
    const parts = String(dataUrl || '').split(',');
    return parts.length >= 2 ? String(parts[1] || '').trim() : '';
}
async function shotSave() {
    if (!window.tt || typeof window.tt.saveScreenshot !== 'function')
        return;
    const pngBase64 = mergedShotPngBase64();
    if (!pngBase64)
        return;
    const ts = Number(shotCaptureTs) || Date.now();
    const res = await window.tt.saveScreenshot(pngBase64, `TT_Screenshot_${formatTimestamp(ts)}.png`);
    if (res && res.error) {
        showIpcToast({ action: '保存', res });
        return;
    }
    if (res && res.canceled) {
        showIpcToast({ action: '保存', res });
        closeShotEditor();
        return;
    }
    showIpcToast({ action: '保存', res, successMessage: '已保存截图' });
    closeShotEditor();
}
async function shotDone() {
    if (!window.tt || typeof window.tt.copyScreenshot !== 'function')
        return;
    const pngBase64 = mergedShotPngBase64();
    if (!pngBase64)
        return;
    const res = await window.tt.copyScreenshot(pngBase64);
    if (res && typeof res === 'object') {
        if (res.ok) {
            bubble.showToast({ type: 'success', message: '已复制截图' });
            closeShotEditor();
            return;
        }
        bubble.showToast({ type: 'error', message: '复制失败', details: normalizeScreenshotError(res.error) });
        return;
    }
    if (res) {
        bubble.showToast({ type: 'success', message: '已复制截图' });
        closeShotEditor();
        return;
    }
    bubble.showToast({ type: 'error', message: '复制失败' });
}
async function shotPin() {
    const pngBase64 = mergedShotPngBase64();
    if (!pngBase64 || !shotRect) {
        closeShotEditor();
        return;
    }
    const r = { x: Math.round(Number(shotRect.x) || 0), y: Math.round(Number(shotRect.y) || 0), w: Math.max(1, Math.round(Number(shotRect.w) || 1)), h: Math.max(1, Math.round(Number(shotRect.h) || 1)) };
    try {
        if (window.tt && typeof window.tt.pinShot === 'function') {
            window.tt.pinShot({ pngBase64, rect: r, displayId });
        }
    }
    catch { }
    closeShotEditor();
}
let screenshotBusy = false;
async function doScreenshot(kind) {
    if (screenshotBusy)
        return;
    screenshotBusy = true;
    try {
        if (!window.tt || typeof window.tt.captureScreen !== 'function') {
            bubble.showText('桌面版才支持截图');
            return;
        }
        const mode = String(kind || 'save');
        const { pngBase64: fullBase64, error } = await window.tt.captureScreen({ displayId });
        if (!fullBase64) {
            bubble.showToast({ type: 'error', message: '截图失败', details: normalizeScreenshotError(error) });
            return;
        }
        bubble.showText('拖拽选择区域，右键取消');
        const region = await pickScreenshotRegion({ frozenPngBase64: fullBase64 });
        if (!region) {
            bubble.showText('已取消截图');
            return;
        }
        const { pngBase64, error: cropErr } = await cropPngBase64ByRegion(fullBase64, region);
        if (!pngBase64) {
            const errKey = cropErr || error;
            bubble.showToast({ type: 'error', message: '截图失败', details: normalizeScreenshotError(errKey) });
            return;
        }
        if (mode === 'clipboard') {
            if (!window.tt || typeof window.tt.copyScreenshot !== 'function')
                return;
            const ok = await window.tt.copyScreenshot(pngBase64);
            bubble.showText(ok ? '已复制截图' : '复制失败');
            return;
        }
        const r = region;
        await openShotEditor(pngBase64, r && r.rect ? r.rect : null);
    }
    catch { }
    finally {
        screenshotBusy = false;
    }
}
const toolbarPinnedStorageKey = displayId ? `tt.toolbarPinned.${displayId}` : 'tt.toolbarPinned';
function readToolbarPinned(defaultPinned) {
    try {
        const v = localStorage.getItem(toolbarPinnedStorageKey);
        if (v === null)
            return !!defaultPinned;
        if (v === '1' || v === 'true')
            return true;
        if (v === '0' || v === 'false')
            return false;
        return !!defaultPinned;
    }
    catch {
        return !!defaultPinned;
    }
}
function writeToolbarPinned(pinned) {
    try {
        localStorage.setItem(toolbarPinnedStorageKey, pinned ? '1' : '0');
    }
    catch { }
}
let currentSettings = { fadeTime: 1000, cssVars: {} };
let pendingSettingsTimer = null;
function scheduleSettingsPatch(patch) {
    currentSettings = ensureSettingsObj(currentSettings);
    const p = patch && typeof patch === 'object' ? patch : {};
    const next = { ...currentSettings, ...p };
    if (p.cssVars && typeof p.cssVars === 'object')
        next.cssVars = { ...currentSettings.cssVars, ...p.cssVars };
    currentSettings = ensureSettingsObj(next);
    try {
        updateRecordAudioControls();
    }
    catch { }
    if (!window.tt || typeof window.tt.setSettings !== 'function')
        return;
    if (pendingSettingsTimer)
        clearTimeout(pendingSettingsTimer);
    pendingSettingsTimer = setTimeout(async () => {
        pendingSettingsTimer = null;
        try {
            const saved = await window.tt.setSettings(currentSettings);
            currentSettings = ensureSettingsObj(saved);
            try {
                updateRecordAudioControls();
            }
            catch { }
        }
        catch { }
    }, 150);
}
function scheduleSettingsCssVars(nextCssVars) {
    scheduleSettingsPatch({ cssVars: nextCssVars });
}
function setCssVar(key, value) {
    try {
        if (value === null || typeof value === 'undefined') {
            root.style.removeProperty(key);
            appliedCssKeys.delete(key);
        }
        else {
            root.style.setProperty(key, String(value));
            appliedCssKeys.add(key);
        }
    }
    catch { }
}
function syncDrawVarsToSettings() {
    setCssVar('--draw-line-width', `${Math.max(1, Math.round(Number(lineWidth) || 1))}px`);
    setCssVar('--draw-eraser-width', `${Math.max(2, Math.round(Number(eraserWidth) || 2))}px`);
    if (strokeColor)
        setCssVar('--draw-stroke-color', String(strokeColor));
    scheduleSettingsCssVars({
        '--draw-line-width': `${Math.max(1, Math.round(Number(lineWidth) || 1))}px`,
        '--draw-eraser-width': `${Math.max(2, Math.round(Number(eraserWidth) || 2))}px`,
        '--draw-stroke-color': strokeColor ? String(strokeColor) : null,
    });
}
function applySettings(settings) {
    const s = ensureSettingsObj(settings);
    currentSettings = s;
    try {
        updateRecordAudioControls();
    }
    catch { }
    try {
        root.dataset.theme = s.uiTheme;
    }
    catch { }
    const cssVars = s && s.cssVars && typeof s.cssVars === 'object' ? s.cssVars : null;
    try {
        for (const k of appliedCssKeys)
            root.style.removeProperty(k);
    }
    catch { }
    appliedCssKeys = new Set();
    if (cssVars) {
        for (const [k, v] of Object.entries(cssVars)) {
            if (!k || typeof k !== 'string')
                continue;
            if (v === null || typeof v === 'undefined') {
                try {
                    root.style.removeProperty(k);
                }
                catch { }
            }
            else {
                try {
                    root.style.setProperty(k, String(v));
                    appliedCssKeys.add(k);
                }
                catch { }
            }
        }
    }
    if (s && typeof s.fadeTime === 'number' && Number.isFinite(s.fadeTime))
        bubbleFadeTime = Math.max(0, Math.round(s.fadeTime));
    try {
        if (mouse && typeof mouse.setMoveVisible === 'function')
            mouse.setMoveVisible(s.mouseDotMoveVisible !== false);
        if (mouse && typeof mouse.setClickVisible === 'function')
            mouse.setClickVisible(s.mouseDotClickVisible !== false);
        if (mouse && typeof mouse.setStyleMode === 'function')
            mouse.setStyleMode(s.mouseDotStyle);
    }
    catch { }
    try {
        const cs = getComputedStyle(root);
        const wRaw = String(cs.getPropertyValue('--draw-line-width') || '').trim();
        const cRaw = String(cs.getPropertyValue('--draw-stroke-color') || '').trim();
        const eRaw = String(cs.getPropertyValue('--draw-eraser-width') || '').trim();
        const stRaw = String(cs.getPropertyValue('--shot-text-font-size') || '').trim();
        const px = (v) => {
            const n = Number(String(v || '').trim().replace(/px$/i, ''));
            return Number.isFinite(n) ? n : null;
        };
        const nextLineWidth = px(wRaw);
        if (nextLineWidth !== null) {
            lineWidth = Math.max(1, Math.round(nextLineWidth));
            if (sizeInput)
                sizeInput.value = String(lineWidth);
            if (shotSizeInput)
                shotSizeInput.value = String(lineWidth);
        }
        if (cRaw) {
            strokeColor = cRaw;
            if (colorInput)
                colorInput.value = strokeColor;
            if (toolbarColorSwatch)
                toolbarColorSwatch.style.background = strokeColor;
            if (shotColorInput)
                shotColorInput.value = strokeColor;
            if (shotToolbarColorSwatch)
                shotToolbarColorSwatch.style.background = strokeColor;
        }
        const nextEraserWidth = px(eRaw);
        if (nextEraserWidth !== null) {
            eraserWidth = Math.max(2, Math.round(nextEraserWidth));
            if (smoothInput)
                smoothInput.value = String(eraserWidth);
            if (shotEraserInput)
                shotEraserInput.value = String(eraserWidth);
            syncEraserCursorSize();
        }
        const nextShotTextSize = px(stRaw);
        if (nextShotTextSize !== null) {
            shotTextFontSizeCss = clamp(Math.round(nextShotTextSize), 10, 80);
            if (shotTextSizeInput)
                shotTextSizeInput.value = String(shotTextFontSizeCss);
        }
    }
    catch { }
    try {
        applyToolbarSettings(s);
    }
    catch { }
    try {
        const shotBtn = toolbar && toolbar.querySelector('button[data-tool="screenshot"]');
        const hotkeys = s && s.hotkeys && typeof s.hotkeys === 'object' ? s.hotkeys : {};
        const saveKey = formatAcceleratorLabel(hotkeys.screenshotSave);
        if (shotBtn)
            shotBtn.title = saveKey ? `截图 (${saveKey})` : '截图';
        const refBtn = toolbar && toolbar.querySelector('button[data-tool="refimg"]');
        const refKey = formatAcceleratorLabel(hotkeys.toolRefimg);
        if (refBtn)
            refBtn.title = refKey ? `参考图 (${refKey})` : '参考图';
        const undoBtn = toolbar && toolbar.querySelector('button[data-tool="undo"]');
        const undoKey = formatAcceleratorLabel(hotkeys.undo);
        if (undoBtn)
            undoBtn.title = undoKey ? `撤销 (${undoKey})` : '撤销';
        currentHotkeysNorm = {
            undo: normalizeHotkeyString(hotkeys.undo),
            clear: normalizeHotkeyString(hotkeys.clear),
            toolPen: normalizeHotkeyString(hotkeys.toolPen),
            toolEraser: normalizeHotkeyString(hotkeys.toolEraser),
            toolArrow: normalizeHotkeyString(hotkeys.toolArrow),
            toolRect: normalizeHotkeyString(hotkeys.toolRect),
            toolCircle: normalizeHotkeyString(hotkeys.toolCircle),
            toolText: normalizeHotkeyString(hotkeys.toolText),
            toolMosaic: normalizeHotkeyString(hotkeys.toolMosaic),
            toolRefimg: normalizeHotkeyString(hotkeys.toolRefimg),
            toolRecord: normalizeHotkeyString(hotkeys.toolRecord),
        };
    }
    catch { }
}
let toolbarVisible = true;
let toolbarPinned = false;
function applyToolbarSettings(s) {
    toolbarVisible = s.toolbarVisible !== false;
    toolbarPinned = readToolbarPinned(!!s.toolbarPinned);
    if (!toolbarVisible) {
        try {
            if (drawingEnabled)
                setDrawingMode(false);
        }
        catch { }
        try {
            if (toolbar)
                toolbar.classList.remove('visible');
        }
        catch { }
        try {
            if (toolbar)
                toolbar.style.display = 'none';
        }
        catch { }
        try {
            if (hotzone)
                hotzone.style.display = 'none';
        }
        catch { }
        try {
            if (window.tt)
                window.tt.setDrawingCapture(false);
        }
        catch { }
    }
    else {
        try {
            if (toolbar)
                toolbar.style.display = '';
        }
        catch { }
        try {
            if (hotzone)
                hotzone.style.display = '';
        }
        catch { }
        if (toolbar && toolbarPinned)
            toolbar.classList.add('visible');
    }
    if (pinBtn)
        pinBtn.classList.toggle('active', toolbarPinned);
}
// 绘图（简化版，仅开关与撤销/清空示意）
const drawCanvas = document.getElementById('draw-canvas');
const tempCanvas = document.getElementById('temp-canvas');
const toolbar = document.getElementById('draw-toolbar');
const hotzone = document.getElementById('toolbar-hotzone');
const pinBtn = document.getElementById('toolbar-pin');
const colorInput = document.getElementById('color');
const sizeInput = document.getElementById('size');
const smoothInput = document.getElementById('smooth');
const drawCtx = drawCanvas.getContext('2d');
const tempCtx = tempCanvas.getContext('2d');
let toolbarColorSwatch = null;
let shotToolbarColorSwatch = null;
function resizeCanvas() {
    const dpr = window.devicePixelRatio || 1;
    const w = window.innerWidth, h = window.innerHeight;
    for (const c of [drawCanvas, tempCanvas]) {
        c.width = Math.ceil(w * dpr);
        c.height = Math.ceil(h * dpr);
        c.style.width = w + 'px';
        c.style.height = h + 'px';
    }
    drawCtx.setTransform(1, 0, 0, 1, 0, 0);
    tempCtx.setTransform(1, 0, 0, 1, 0, 0);
    drawCtx.scale(dpr, dpr);
    tempCtx.scale(dpr, dpr);
}
window.addEventListener('resize', resizeCanvas);
resizeCanvas();
const tooltipEnhancer = createTooltipEnhancer();
tooltipEnhancer.enhanceAll();
let tool = 'pen';
let strokeColor = (colorInput && colorInput.value) || '#ff5252';
let lineWidth = Number((sizeInput && sizeInput.value) || 4);
let eraserWidth = Number((smoothInput && smoothInput.value) || lineWidth);
let smoothing = 0.6;
let eraserCursor = null;
let lastPointerPos = null;
function ensureEraserCursor() {
    if (eraserCursor)
        return;
    eraserCursor = document.createElement('div');
    eraserCursor.className = 'eraser-cursor';
    eraserCursor.style.display = 'none';
    document.body.appendChild(eraserCursor);
}
function currentEraserCursorCssSize() {
    const base = Math.max(2, Math.round(Number(eraserWidth) || 0));
    if (!shotEditorActive || shotTool !== 'eraser' || !shotStage || !shotAnnotCanvas)
        return base;
    const r = shotStage.getBoundingClientRect();
    const scale = r.width / Math.max(1, shotAnnotCanvas.width);
    return Math.max(2, Math.round(base * scale));
}
function syncEraserCursorSize() {
    if (!eraserCursor)
        return;
    const s = currentEraserCursorCssSize();
    eraserCursor.style.width = s + 'px';
    eraserCursor.style.height = s + 'px';
}
function setEraserCursorVisible(visible) {
    if (!eraserCursor && !visible)
        return;
    ensureEraserCursor();
    syncEraserCursorSize();
    eraserCursor.style.display = visible ? 'block' : 'none';
}
function updateEraserCursorPos(pos) {
    if (!eraserCursor || !pos)
        return;
    eraserCursor.style.left = pos.x + 'px';
    eraserCursor.style.top = pos.y + 'px';
}
function refreshEraserCursor(pos) {
    const shouldShow = (drawingEnabled && tool === 'eraser') || (shotEditorActive && shotTool === 'eraser');
    if (!shouldShow) {
        setEraserCursorVisible(false);
        return;
    }
    const p = pos || lastPointerPos;
    if (!p) {
        setEraserCursorVisible(false);
        return;
    }
    setEraserCursorVisible(true);
    updateEraserCursorPos(p);
}
function setActiveTool(next) {
    tool = next;
    if (toolbar) {
        toolbar.querySelectorAll('button[data-tool]').forEach(btn => btn.classList.toggle('active', btn.dataset.tool === tool));
    }
    refreshEraserCursor();
}
let drawingEnabled = false;
function setDrawingMode(enabled) {
    drawingEnabled = !!enabled;
    drawCanvas.style.pointerEvents = drawingEnabled ? 'auto' : 'none';
    tempCanvas.style.pointerEvents = drawingEnabled ? 'auto' : 'none';
    drawCanvas.style.visibility = drawingEnabled ? 'visible' : 'hidden';
    tempCanvas.style.visibility = drawingEnabled ? 'visible' : 'hidden';
    try {
        if (window.tt)
            window.tt.setDrawingCapture(!!drawingEnabled);
    }
    catch { }
    if (!drawingEnabled) {
        isDrawing = false;
        pathPoints = [];
        clearTemp();
        if (toolbar)
            toolbar.querySelectorAll('button[data-tool]').forEach(btn => btn.classList.remove('active'));
        if (toolbar && !toolbarPinned)
            toolbar.classList.remove('visible');
    }
    refreshEraserCursor();
}
let isDrawing = false, sx = 0, sy = 0;
let pathPoints = [];
let history = [];
const getPos = (e) => ({ x: e.clientX, y: e.clientY });
function clearTemp() { tempCtx.clearRect(0, 0, tempCanvas.width, tempCanvas.height); }
let activeDrawPointerId = null;
let strokeUsePressure = false;
let drawUsingWindowPointerListeners = false;
function bindDrawWindowPointerListeners() {
    if (drawUsingWindowPointerListeners)
        return;
    drawUsingWindowPointerListeners = true;
    window.addEventListener('pointermove', onPointerMove);
    window.addEventListener('pointerup', onPointerUp);
    window.addEventListener('pointercancel', onPointerUp);
}
function unbindDrawWindowPointerListeners() {
    if (!drawUsingWindowPointerListeners)
        return;
    drawUsingWindowPointerListeners = false;
    window.removeEventListener('pointermove', onPointerMove);
    window.removeEventListener('pointerup', onPointerUp);
    window.removeEventListener('pointercancel', onPointerUp);
}
function pushHistory() {
    try {
        const img = drawCtx.getImageData(0, 0, drawCanvas.width, drawCanvas.height);
        history.push(img);
    }
    catch { }
}
function undo() {
    if (!history.length)
        return;
    const img = history.pop();
    clearTemp();
    drawCtx.clearRect(0, 0, drawCanvas.width, drawCanvas.height);
    try {
        drawCtx.putImageData(img, 0, 0);
    }
    catch { }
}
function setupStrokeStyle(ctx) {
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';
    ctx.strokeStyle = strokeColor;
    ctx.lineWidth = lineWidth;
}
function drawPolyline(ctx, pts) {
    if (!pts || pts.length < 2)
        return;
    ctx.beginPath();
    ctx.moveTo(pts[0].x, pts[0].y);
    for (let i = 1; i < pts.length; i++) {
        ctx.lineTo(pts[i].x, pts[i].y);
    }
    ctx.stroke();
}
function chaikinSmooth(pts, t) {
    let iters = Math.floor(Math.round(Math.max(0, Math.min(0.9, Number(t) || 0)) * 10) / 3);
    if (iters < 0)
        iters = 0;
    else if (iters > 3)
        iters = 3;
    if (!pts || pts.length < 3 || iters <= 0)
        return pts;
    let out = pts.slice();
    for (let k = 0; k < iters; k++) {
        const next = [];
        next.push(out[0]);
        for (let i = 0; i < out.length - 1; i++) {
            const p = out[i], q = out[i + 1];
            const p1 = { x: 0.75 * p.x + 0.25 * q.x, y: 0.75 * p.y + 0.25 * q.y };
            const p2 = { x: 0.25 * p.x + 0.75 * q.x, y: 0.25 * p.y + 0.75 * q.y };
            next.push(p1, p2);
        }
        next.push(out[out.length - 1]);
        out = next;
    }
    return out;
}
function smoothPolyline(pts, t) {
    // 优先使用原生几何库，其次使用 JS 回退
    const mod = (window.tt && window.tt.geometry) || null;
    if (mod && typeof mod.smoothPolyline === 'function') {
        try {
            return mod.smoothPolyline(pts, Number(t) || 0);
        }
        catch { /* fallthrough */ }
    }
    return chaikinSmooth(pts, Number(t) || 0);
}
function beginStroke(x, y, pressure, pointerType) {
    pathPoints = [{ x, y, p: pressure }];
    strokeUsePressure = String(pointerType || '') === 'pen';
}
function extendStroke(x, y, pressure) {
    pathPoints.push({ x, y, p: pressure });
    clearTemp();
    setupStrokeStyle(tempCtx);
    const smoothed = smoothPolyline(pathPoints.map((p) => ({ x: p.x, y: p.y })), smoothing);
    if (strokeUsePressure)
        drawPressurePolyline(tempCtx, smoothed, lineWidth, (r) => pressureAtRatio(pathPoints, r));
    else
        drawPolyline(tempCtx, smoothed);
}
function commitStroke() {
    clearTemp();
    setupStrokeStyle(drawCtx);
    const smoothed = smoothPolyline(pathPoints.map((p) => ({ x: p.x, y: p.y })), smoothing);
    pushHistory();
    if (strokeUsePressure)
        drawPressurePolyline(drawCtx, smoothed, lineWidth, (r) => pressureAtRatio(pathPoints, r));
    else
        drawPolyline(drawCtx, smoothed);
    pathPoints = [];
    strokeUsePressure = false;
}
function jsArrowHeadPoints(from, to, headLen = 12, headAngleDeg = 28) {
    const dx = to.x - from.x, dy = to.y - from.y;
    const theta = Math.atan2(dy, dx);
    const angle = headAngleDeg * Math.PI / 180;
    const l = Math.max(1, headLen);
    const left = { x: to.x - l * Math.cos(theta + angle), y: to.y - l * Math.sin(theta + angle) };
    const right = { x: to.x - l * Math.cos(theta - angle), y: to.y - l * Math.sin(theta - angle) };
    return { left, right };
}
function arrowHeadPoints(from, to, headLen = 12, headAngleDeg = 28) {
    const mod = (window.tt && window.tt.geometry) || null;
    if (mod && typeof mod.arrowHeadPoints === 'function') {
        try {
            return mod.arrowHeadPoints(from, to, headLen, headAngleDeg);
        }
        catch { }
    }
    return jsArrowHeadPoints(from, to, headLen, headAngleDeg);
}
function drawArrow(ctx, from, to) {
    setupStrokeStyle(ctx);
    const head = arrowHeadPoints(from, to);
    ctx.beginPath();
    ctx.moveTo(from.x, from.y);
    ctx.lineTo(to.x, to.y);
    ctx.moveTo(to.x, to.y);
    ctx.lineTo(head.left.x, head.left.y);
    ctx.moveTo(to.x, to.y);
    ctx.lineTo(head.right.x, head.right.y);
    ctx.stroke();
}
function drawRect(ctx, p1, p2) {
    setupStrokeStyle(ctx);
    const x = Math.min(p1.x, p2.x), y = Math.min(p1.y, p2.y);
    const w = Math.abs(p2.x - p1.x), h = Math.abs(p2.y - p1.y);
    ctx.beginPath();
    ctx.rect(x, y, w, h);
    ctx.stroke();
}
function drawEllipse(ctx, p1, p2) {
    setupStrokeStyle(ctx);
    const cx = (p1.x + p2.x) / 2, cy = (p1.y + p2.y) / 2;
    const rx = Math.abs(p2.x - p1.x) / 2, ry = Math.abs(p2.y - p1.y) / 2;
    ctx.beginPath();
    ctx.ellipse(cx, cy, rx, ry, 0, 0, Math.PI * 2);
    ctx.stroke();
}
function onPointerDown(e) {
    if (!drawingEnabled)
        return;
    if (e && typeof e.button === 'number' && e.button !== 0)
        return;
    activeDrawPointerId = e && typeof e.pointerId === 'number' ? e.pointerId : null;
    try {
        if (activeDrawPointerId !== null && e && e.target && e.target.setPointerCapture)
            e.target.setPointerCapture(activeDrawPointerId);
    }
    catch { }
    bindDrawWindowPointerListeners();
    const { x, y } = getPos(e);
    isDrawing = true;
    sx = x;
    sy = y;
    lastPointerPos = { x, y };
    refreshEraserCursor(lastPointerPos);
    if (tool === 'pen')
        beginStroke(x, y, pressureFromEvent(e), e.pointerType);
    if (tool === 'eraser') {
        pathPoints = [{ x, y }];
        setupStrokeStyle(drawCtx);
        drawCtx.globalCompositeOperation = 'destination-out';
        drawCtx.beginPath();
        drawCtx.moveTo(x, y);
        pushHistory();
    }
    try {
        if (e && typeof e.preventDefault === 'function')
            e.preventDefault();
    }
    catch { }
}
function onPointerMove(e) {
    if (!drawingEnabled)
        return;
    if (drawUsingWindowPointerListeners && e && e.currentTarget !== window)
        return;
    if (activeDrawPointerId !== null && e && typeof e.pointerId === 'number' && e.pointerId !== activeDrawPointerId)
        return;
    const { x, y } = getPos(e);
    lastPointerPos = { x, y };
    refreshEraserCursor(lastPointerPos);
    if (!isDrawing)
        return;
    if (tool === 'pen') {
        extendStroke(x, y, pressureFromEvent(e));
        return;
    }
    if (tool === 'eraser') {
        drawCtx.lineTo(x, y);
        drawCtx.lineWidth = eraserWidth;
        drawCtx.stroke();
        return;
    }
    clearTemp();
    if (tool === 'arrow') {
        drawArrow(tempCtx, { x: sx, y: sy }, { x, y });
        return;
    }
    if (tool === 'rect') {
        drawRect(tempCtx, { x: sx, y: sy }, { x, y });
        return;
    }
    if (tool === 'circle') {
        drawEllipse(tempCtx, { x: sx, y: sy }, { x, y });
        return;
    }
    try {
        if (e && typeof e.preventDefault === 'function')
            e.preventDefault();
    }
    catch { }
}
function onPointerUp(e) {
    if (!drawingEnabled || !isDrawing)
        return;
    if (drawUsingWindowPointerListeners && e && e.currentTarget !== window)
        return;
    if (activeDrawPointerId !== null && e && typeof e.pointerId === 'number' && e.pointerId !== activeDrawPointerId)
        return;
    const capturedId = activeDrawPointerId;
    isDrawing = false;
    activeDrawPointerId = null;
    unbindDrawWindowPointerListeners();
    try {
        if (capturedId !== null && e && e.target && e.target.releasePointerCapture)
            e.target.releasePointerCapture(capturedId);
    }
    catch { }
    if (tool === 'pen') {
        commitStroke();
        return;
    }
    if (tool === 'eraser') {
        drawCtx.globalCompositeOperation = 'source-over';
        pathPoints = [];
        clearTemp();
        return;
    }
    const { x, y } = e ? getPos(e) : { x: sx, y: sy };
    if (tool === 'arrow') {
        clearTemp();
        pushHistory();
        drawArrow(drawCtx, { x: sx, y: sy }, { x, y });
        return;
    }
    if (tool === 'rect') {
        clearTemp();
        pushHistory();
        drawRect(drawCtx, { x: sx, y: sy }, { x, y });
        return;
    }
    if (tool === 'circle') {
        clearTemp();
        pushHistory();
        drawEllipse(drawCtx, { x: sx, y: sy }, { x, y });
        return;
    }
    try {
        if (e && typeof e.preventDefault === 'function')
            e.preventDefault();
    }
    catch { }
}
drawCanvas.addEventListener('pointerdown', onPointerDown);
drawCanvas.addEventListener('pointermove', onPointerMove);
drawCanvas.addEventListener('pointerup', onPointerUp);
drawCanvas.addEventListener('pointercancel', onPointerUp);
drawCanvas.addEventListener('contextmenu', (e) => { try {
    e.preventDefault();
}
catch { } });
tempCanvas.addEventListener('pointerdown', onPointerDown);
tempCanvas.addEventListener('pointermove', onPointerMove);
tempCanvas.addEventListener('pointerup', onPointerUp);
tempCanvas.addEventListener('pointercancel', onPointerUp);
tempCanvas.addEventListener('contextmenu', (e) => { try {
    e.preventDefault();
}
catch { } });
// 在画布范围内保持捕获以允许绘图，离开时根据模式决定是否释放
const ensureCapture = (active) => { try {
    if (window.tt)
        window.tt.setDrawingCapture(!!active);
}
catch { } };
drawCanvas.addEventListener('mouseenter', () => ensureCapture(true));
tempCanvas.addEventListener('mouseenter', () => ensureCapture(true));
drawCanvas.addEventListener('mouseleave', () => ensureCapture(drawingEnabled));
tempCanvas.addEventListener('mouseleave', () => ensureCapture(drawingEnabled));
drawCanvas.addEventListener('mouseenter', (e) => {
    try {
        lastPointerPos = { x: e.clientX, y: e.clientY };
    }
    catch { }
    refreshEraserCursor(lastPointerPos);
});
tempCanvas.addEventListener('mouseenter', (e) => {
    try {
        lastPointerPos = { x: e.clientX, y: e.clientY };
    }
    catch { }
    refreshEraserCursor(lastPointerPos);
});
drawCanvas.addEventListener('mouseleave', () => setEraserCursorVisible(false));
tempCanvas.addEventListener('mouseleave', () => setEraserCursorVisible(false));
if (toolbar) {
    if (pinBtn) {
        pinBtn.addEventListener('click', async (e) => {
            try {
                e.preventDefault();
            }
            catch { }
            if (!toolbarVisible)
                return;
            toolbarPinned = !toolbarPinned;
            writeToolbarPinned(toolbarPinned);
            if (pinBtn)
                pinBtn.classList.toggle('active', toolbarPinned);
            if (toolbarPinned)
                toolbar.classList.add('visible');
            else if (!drawingEnabled && toolbar && !toolbar.matches(':hover'))
                toolbar.classList.remove('visible');
        });
    }
    toolbar.addEventListener('click', (e) => {
        const t = e && e.target && e.target.closest ? e.target.closest('button[data-tool]') : null;
        if (!t)
            return;
        const toolId = t.dataset.tool;
        if (toolId === 'screenshot') {
            doScreenshot('save');
            return;
        }
        if (toolId === 'refimg') {
            openRefImagePicker();
            return;
        }
        if (toolId === 'record') {
            startRecordRegionPick();
            return;
        }
        if (toolId === 'undo') {
            undo();
            return;
        }
        if (toolId === 'clear') {
            drawCtx.clearRect(0, 0, drawCanvas.width, drawCanvas.height);
            clearTemp();
            history = [];
            return;
        }
        if (toolId === tool && drawingEnabled) {
            setDrawingMode(false);
            return;
        }
        setActiveTool(toolId);
        setDrawingMode(true);
    });
    if (colorInput)
        colorInput.addEventListener('input', (e) => {
            strokeColor = e.target.value;
            if (toolbarColorSwatch)
                toolbarColorSwatch.style.background = strokeColor;
            try {
                if (shotColorInput)
                    shotColorInput.value = strokeColor;
            }
            catch { }
            try {
                if (shotToolbarColorSwatch)
                    shotToolbarColorSwatch.style.background = strokeColor;
            }
            catch { }
            syncDrawVarsToSettings();
        });
    // 完成选择或面板提交时，自动关闭原生颜色面板（无需额外点击）
    if (colorInput)
        colorInput.addEventListener('change', (e) => {
            try {
                strokeColor = e.target.value;
                if (toolbarColorSwatch)
                    toolbarColorSwatch.style.background = strokeColor;
                try {
                    if (shotColorInput)
                        shotColorInput.value = strokeColor;
                }
                catch { }
                try {
                    if (shotToolbarColorSwatch)
                        shotToolbarColorSwatch.style.background = strokeColor;
                }
                catch { }
                syncDrawVarsToSettings();
                colorInput.blur();
            }
            catch { }
        });
    if (sizeInput)
        sizeInput.addEventListener('input', (e) => {
            lineWidth = Number(e.target.value) || 4;
            try {
                if (shotSizeInput)
                    shotSizeInput.value = String(lineWidth);
            }
            catch { }
            syncDrawVarsToSettings();
        });
    if (smoothInput)
        smoothInput.addEventListener('input', (e) => {
            eraserWidth = Number(e.target.value) || eraserWidth;
            try {
                if (shotEraserInput)
                    shotEraserInput.value = String(eraserWidth);
            }
            catch { }
            syncEraserCursorSize();
            syncDrawVarsToSettings();
        });
    if (hotzone) {
        // 根据面板实际宽度，将热点区限制到面板投影范围内（居中）
        const updateHotzoneRegion = () => {
            try {
                const w = (toolbar && toolbar.offsetWidth) || 0;
                const left = Math.max(0, Math.round((window.innerWidth - w) / 2));
                hotzone.style.width = w + 'px';
                hotzone.style.left = left + 'px';
                hotzone.style.right = 'auto';
            }
            catch { }
        };
        // 初始与窗口尺寸变化时同步
        updateHotzoneRegion();
        window.addEventListener('resize', updateHotzoneRegion);
        hotzone.addEventListener('mouseenter', () => { if (!toolbarVisible)
            return; toolbar.classList.add('visible'); if (window.tt)
            window.tt.setDrawingCapture(true); });
        hotzone.addEventListener('mouseleave', () => { if (!drawingEnabled) {
            if (!toolbarPinned)
                toolbar.classList.remove('visible');
            if (window.tt)
                window.tt.setDrawingCapture(false);
        } });
    }
    toolbar.addEventListener('mouseenter', () => { if (!toolbarVisible)
        return; toolbar.classList.add('visible'); if (window.tt)
        window.tt.setDrawingCapture(true); });
    toolbar.addEventListener('mouseleave', () => { if (!drawingEnabled) {
        if (!toolbarPinned)
            toolbar.classList.remove('visible');
        if (window.tt)
            window.tt.setDrawingCapture(false);
    } });
}
// 自定义颜色选择器：点击色块弹出面板，鼠标离开面板即关闭
try {
    if (toolbar) {
        const origColorInput = colorInput;
        if (origColorInput) {
            // 隐藏原生颜色输入，改用自定义色块按钮
            origColorInput.style.display = 'none';
            const swatch = document.createElement('button');
            swatch.id = 'color-swatch';
            swatch.title = '颜色';
            swatch.style.width = '32px';
            swatch.style.height = '24px';
            swatch.style.border = '2px solid #ccc';
            swatch.style.borderRadius = '6px';
            swatch.style.background = strokeColor;
            swatch.style.cursor = 'pointer';
            swatch.style.padding = '0';
            swatch.style.outline = 'none';
            origColorInput.parentNode.insertBefore(swatch, origColorInput);
            toolbarColorSwatch = swatch;
            const panel = document.createElement('div');
            panel.id = 'color-panel';
            panel.style.position = 'fixed';
            panel.style.zIndex = '1006';
            panel.style.background = '#ffffff';
            panel.style.borderRadius = '8px';
            panel.style.boxShadow = '0 6px 20px rgba(0,0,0,0.35)';
            panel.style.padding = '8px';
            panel.style.display = 'none';
            panel.style.userSelect = 'none';
            const canvas = document.createElement('canvas');
            canvas.width = 240;
            canvas.height = 140;
            canvas.style.display = 'block';
            canvas.style.cursor = 'crosshair';
            canvas.style.marginBottom = '8px';
            const hueRange = document.createElement('input');
            hueRange.type = 'range';
            hueRange.min = '0';
            hueRange.max = '360';
            hueRange.value = '0';
            hueRange.style.width = '240px';
            panel.appendChild(canvas);
            panel.appendChild(hueRange);
            document.body.appendChild(panel);
            const ctx = canvas.getContext('2d');
            let hue = 0;
            let sat = 1;
            let val = 1;
            const hsvToRgb = (h, s, v) => {
                // h: 0..1, s: 0..1, v: 0..1
                const f = (n) => {
                    const k = (n + h * 6) % 6;
                    return v - v * s * Math.max(Math.min(k, 4 - k, 1), 0);
                };
                const r = Math.round(f(5) * 255);
                const g = Math.round(f(3) * 255);
                const b = Math.round(f(1) * 255);
                return { r, g, b };
            };
            const rgbToCss = ({ r, g, b }) => `rgb(${r}, ${g}, ${b})`;
            function drawSV() {
                const base = hsvToRgb(hue / 360, 1, 1);
                const w = canvas.width, h = canvas.height;
                const gradX = ctx.createLinearGradient(0, 0, w, 0);
                gradX.addColorStop(0, '#ffffff');
                gradX.addColorStop(1, rgbToCss(base));
                ctx.fillStyle = gradX;
                ctx.fillRect(0, 0, w, h);
                const gradY = ctx.createLinearGradient(0, 0, 0, h);
                gradY.addColorStop(0, 'rgba(0,0,0,0)');
                gradY.addColorStop(1, 'rgba(0,0,0,1)');
                ctx.fillStyle = gradY;
                ctx.fillRect(0, 0, w, h);
                // knob
                const x = sat * w;
                const y = (1 - val) * h;
                ctx.beginPath();
                ctx.arc(x, y, 6, 0, Math.PI * 2);
                ctx.lineWidth = 2;
                ctx.strokeStyle = '#fff';
                ctx.stroke();
            }
            function updateColor() {
                const css = rgbToCss(hsvToRgb(hue / 360, sat, val));
                strokeColor = css;
                swatch.style.background = css;
                try {
                    if (origColorInput)
                        origColorInput.value = css;
                }
                catch { }
                try {
                    if (shotColorInput)
                        shotColorInput.value = css;
                }
                catch { }
                try {
                    if (shotToolbarColorSwatch)
                        shotToolbarColorSwatch.style.background = css;
                }
                catch { }
                syncDrawVarsToSettings();
            }
            function posFromEvent(e) {
                const rect = canvas.getBoundingClientRect();
                const x = Math.max(0, Math.min(rect.width, e.clientX - rect.left));
                const y = Math.max(0, Math.min(rect.height, e.clientY - rect.top));
                return { x, y };
            }
            let dragging = false;
            canvas.addEventListener('mousedown', (e) => {
                dragging = true;
                const p = posFromEvent(e);
                sat = p.x / canvas.width;
                val = 1 - p.y / canvas.height;
                updateColor();
                drawSV();
            });
            window.addEventListener('mousemove', (e) => {
                if (!dragging)
                    return;
                const p = posFromEvent(e);
                sat = p.x / canvas.width;
                val = 1 - p.y / canvas.height;
                updateColor();
                drawSV();
            });
            window.addEventListener('mouseup', () => { dragging = false; });
            hueRange.addEventListener('input', (e) => {
                const t = e && e.target ? e.target : null;
                hue = Number(t && t.value) || 0;
                updateColor();
                drawSV();
            });
            let colorPanelActivated = false;
            function showPanel() {
                const r = swatch.getBoundingClientRect();
                panel.style.left = `${Math.round(r.left)}px`;
                panel.style.top = `${Math.round(r.bottom + 6)}px`;
                panel.style.display = 'block';
                drawSV();
                try {
                    if (toolbar)
                        toolbar.classList.add('visible');
                    // 仅当当前没有处于绘图模式时，由颜色面板激活工具
                    colorPanelActivated = !drawingEnabled;
                    if (colorPanelActivated) {
                        setDrawingMode(true);
                        if (window.tt)
                            window.tt.setDrawingCapture(true);
                    }
                }
                catch { }
            }
            function hidePanel() {
                panel.style.display = 'none';
                try {
                    // 仅当颜色面板曾经激活了工具时，离开面板才取消激活
                    if (colorPanelActivated) {
                        setDrawingMode(false);
                        if (window.tt)
                            window.tt.setDrawingCapture(false);
                    }
                    colorPanelActivated = false;
                }
                catch { }
            }
            swatch.addEventListener('click', showPanel);
            panel.addEventListener('mouseleave', hidePanel);
            document.addEventListener('keydown', (e) => { if (e.key === 'Escape')
                hidePanel(); });
            // 初始化
            updateColor();
            drawSV();
        }
    }
}
catch { }
function ensureShotColorPicker() {
    if (shotColorPickerReady)
        return;
    if (!shotToolbar)
        return;
    const origColorInput = shotColorInput;
    if (!origColorInput)
        return;
    origColorInput.style.display = 'none';
    const swatch = document.createElement('button');
    swatch.id = 'shot-color-swatch';
    swatch.title = '颜色';
    swatch.style.width = '32px';
    swatch.style.height = '24px';
    swatch.style.border = '2px solid #ccc';
    swatch.style.borderRadius = '6px';
    swatch.style.background = strokeColor;
    swatch.style.cursor = 'pointer';
    swatch.style.padding = '0';
    swatch.style.outline = 'none';
    origColorInput.parentNode.insertBefore(swatch, origColorInput);
    shotToolbarColorSwatch = swatch;
    shotColorPickerReady = true;
    const panel = document.createElement('div');
    panel.id = 'shot-color-panel';
    panel.style.position = 'fixed';
    panel.style.zIndex = '2301';
    panel.style.background = '#ffffff';
    panel.style.borderRadius = '8px';
    panel.style.boxShadow = '0 6px 20px rgba(0,0,0,0.35)';
    panel.style.padding = '8px';
    panel.style.display = 'none';
    panel.style.userSelect = 'none';
    const canvas = document.createElement('canvas');
    canvas.width = 240;
    canvas.height = 140;
    canvas.style.display = 'block';
    canvas.style.cursor = 'crosshair';
    canvas.style.marginBottom = '8px';
    const hueRange = document.createElement('input');
    hueRange.type = 'range';
    hueRange.min = '0';
    hueRange.max = '360';
    hueRange.value = '0';
    hueRange.style.width = '240px';
    panel.appendChild(canvas);
    panel.appendChild(hueRange);
    document.body.appendChild(panel);
    const ctx = canvas.getContext('2d');
    let hue = 0;
    let sat = 1;
    let val = 1;
    const hsvToRgb = (h, s, v) => {
        const f = (n) => {
            const k = (n + h * 6) % 6;
            return v - v * s * Math.max(Math.min(k, 4 - k, 1), 0);
        };
        const r = Math.round(f(5) * 255);
        const g = Math.round(f(3) * 255);
        const b = Math.round(f(1) * 255);
        return { r, g, b };
    };
    const rgbToCss = ({ r, g, b }) => `rgb(${r}, ${g}, ${b})`;
    function drawSV() {
        const base = hsvToRgb(hue / 360, 1, 1);
        const w = canvas.width, h = canvas.height;
        const gradX = ctx.createLinearGradient(0, 0, w, 0);
        gradX.addColorStop(0, '#ffffff');
        gradX.addColorStop(1, rgbToCss(base));
        ctx.fillStyle = gradX;
        ctx.fillRect(0, 0, w, h);
        const gradY = ctx.createLinearGradient(0, 0, 0, h);
        gradY.addColorStop(0, 'rgba(0,0,0,0)');
        gradY.addColorStop(1, 'rgba(0,0,0,1)');
        ctx.fillStyle = gradY;
        ctx.fillRect(0, 0, w, h);
        const x = sat * w;
        const y = (1 - val) * h;
        ctx.beginPath();
        ctx.arc(x, y, 6, 0, Math.PI * 2);
        ctx.lineWidth = 2;
        ctx.strokeStyle = '#fff';
        ctx.stroke();
    }
    function updateColor() {
        const css = rgbToCss(hsvToRgb(hue / 360, sat, val));
        strokeColor = css;
        swatch.style.background = css;
        try {
            if (origColorInput)
                origColorInput.value = css;
        }
        catch { }
        try {
            if (colorInput)
                colorInput.value = css;
        }
        catch { }
        try {
            if (toolbarColorSwatch)
                toolbarColorSwatch.style.background = css;
        }
        catch { }
        syncDrawVarsToSettings();
        try {
            const box = getShotActiveOrSelectedTextBox();
            if (box)
                applyShotTextStyleToBox(box, { color: strokeColor });
        }
        catch { }
    }
    function posFromEvent(e) {
        const rect = canvas.getBoundingClientRect();
        const x = Math.max(0, Math.min(rect.width, e.clientX - rect.left));
        const y = Math.max(0, Math.min(rect.height, e.clientY - rect.top));
        return { x, y };
    }
    let dragging = false;
    canvas.addEventListener('mousedown', (e) => {
        dragging = true;
        const p = posFromEvent(e);
        sat = p.x / canvas.width;
        val = 1 - p.y / canvas.height;
        updateColor();
        drawSV();
    });
    window.addEventListener('mousemove', (e) => {
        if (!dragging)
            return;
        const p = posFromEvent(e);
        sat = p.x / canvas.width;
        val = 1 - p.y / canvas.height;
        updateColor();
        drawSV();
    });
    window.addEventListener('mouseup', () => { dragging = false; });
    hueRange.addEventListener('input', (e) => {
        const t = e && e.target ? e.target : null;
        hue = Number(t && t.value) || 0;
        updateColor();
        drawSV();
    });
    function showPanel() {
        const r = swatch.getBoundingClientRect();
        panel.style.left = `${Math.round(r.left)}px`;
        panel.style.top = `${Math.round(r.bottom + 6)}px`;
        panel.style.display = 'block';
        drawSV();
    }
    function hidePanel() { panel.style.display = 'none'; }
    swatch.addEventListener('click', showPanel);
    panel.addEventListener('mouseleave', hidePanel);
    document.addEventListener('keydown', (e) => { if (e.key === 'Escape')
        hidePanel(); });
    updateColor();
    drawSV();
}
const pressedKeys = new Map();
let capsLockOn = false;
// 组合显示锁定：当出现非修饰键时，锁定当前组合标签，直到全部松开
let lockedChordActive = false;
let lockedChordLabel = '';
const MOD_NAMES = new Set(['Shift', 'Ctrl', 'Alt', 'Win', 'Cmd', 'Meta', 'Caps Lock']);
// 鼠标气泡去抖：防止一次点击产生两次气泡（down/up）导致 ×2
let lastMouseBubbleKey = null;
let lastMouseBubbleTs = 0;
function shouldShowMouseBubble(keyId) {
    const now = Date.now();
    if (lastMouseBubbleKey === keyId && (now - lastMouseBubbleTs) < 120)
        return false;
    lastMouseBubbleKey = keyId;
    lastMouseBubbleTs = now;
    return true;
}
function hasNonModifierPressed(map) {
    for (const v of map.values()) {
        const n = v && v.name;
        if (n && !MOD_NAMES.has(n))
            return true;
    }
    return false;
}
function activateToolFromHotkey(nextTool) {
    if (drawingEnabled && tool === nextTool) {
        setDrawingMode(false);
        return;
    }
    try {
        if (toolbar)
            toolbar.classList.add('visible');
    }
    catch { }
    setActiveTool(nextTool);
    setDrawingMode(true);
}
function onKeyPayload(payload) {
    if (!payload || typeof payload !== 'object')
        return;
    const { type, data } = payload;
    if (!data)
        return;
    if (type === 'down') {
        const id = getKeyIdFromEvent(data);
        const name = nameFromEvent(data);
        if (name === 'Esc' && screenshotSelectActive && typeof screenshotSelectCancel === 'function') {
            screenshotSelectCancel();
            return;
        }
        if (name === 'Esc' && recordSelectActive) {
            cancelRecordUi();
            return;
        }
        if ((name === 'Enter' || name === 'Numpad Enter') && screenshotSelectActive && screenshotSelectSession && screenshotSelectSession.created) {
            try {
                const s = screenshotSelectSession;
                const r = s && s.rect ? s.rect : null;
                if (r && (Number(r.w) || 0) >= 2 && (Number(r.h) || 0) >= 2)
                    s.done({ rect: r, viewport: { width: s.vw, height: s.vh } });
            }
            catch { }
            return;
        }
        if (name === 'Esc' && shotEditorActive) {
            if (shotTextArea && shotTextArea.isConnected)
                cleanupTextArea(false);
            else
                closeShotEditor();
            return;
        }
        if (name === 'Esc' && !shotEditorActive && pinnedShotIdsInWindow && pinnedShotIdsInWindow.length) {
            try {
                if (window.tt && typeof window.tt.popPinnedShot === 'function')
                    window.tt.popPinnedShot();
            }
            catch { }
            bubble.showText('已关闭钉住');
            return;
        }
        if (name === 'Esc') {
            setDrawingMode(false);
            try {
                if (window.tt)
                    window.tt.setDrawingCapture(false);
            }
            catch { }
        }
        if (name === 'Caps Lock') {
            capsLockOn = !capsLockOn;
        }
        const rawcode = typeof data.rawcode === 'number' ? data.rawcode : null;
        const keycode = typeof data.keycode === 'number' ? data.keycode : null;
        pressedKeys.set(id, { name, rawcode, keycode });
        try {
            if (!screenshotSelectActive) {
                const chord = chordFromPressed(pressedKeys);
                if (shotEditorActive) {
                    if ((name === 'Delete' || name === 'Del' || name === 'Backspace') && !(shotTextArea && shotTextArea.isConnected && document.activeElement === shotTextArea)) {
                        const box = getShotActiveOrSelectedTextBox();
                        if (box) {
                            try {
                                shotPushHistory();
                            }
                            catch { }
                            if (shotTextBox === box && shotTextArea && shotTextArea.isConnected)
                                cleanupTextArea(false);
                            else {
                                try {
                                    if (shotSelectedTextBox === box)
                                        clearShotSelectedTextBox();
                                }
                                catch { }
                                try {
                                    box.remove();
                                }
                                catch { }
                            }
                            return;
                        }
                    }
                    if (chord === 'Ctrl+Z' || (chord && currentHotkeysNorm && chord === currentHotkeysNorm.undo)) {
                        shotUndo();
                        return;
                    }
                    if (chord && currentHotkeysNorm && chord === currentHotkeysNorm.clear) {
                        shotClear();
                        return;
                    }
                    if (chord && currentHotkeysNorm) {
                        if (chord === currentHotkeysNorm.toolPen) {
                            setShotTool('pen');
                            return;
                        }
                        if (chord === currentHotkeysNorm.toolEraser) {
                            setShotTool('eraser');
                            return;
                        }
                        if (chord === currentHotkeysNorm.toolArrow) {
                            setShotTool('arrow');
                            return;
                        }
                        if (chord === currentHotkeysNorm.toolRect) {
                            setShotTool('rect');
                            return;
                        }
                        if (chord === currentHotkeysNorm.toolCircle) {
                            setShotTool('circle');
                            return;
                        }
                        if (chord === currentHotkeysNorm.toolText) {
                            setShotTool('text');
                            return;
                        }
                        if (chord === currentHotkeysNorm.toolMosaic) {
                            setShotTool('mosaic');
                            return;
                        }
                    }
                    return;
                }
                if (chord && currentHotkeysNorm && toolbarVisible) {
                    if (chord === currentHotkeysNorm.toolPen) {
                        activateToolFromHotkey('pen');
                        return;
                    }
                    if (chord === currentHotkeysNorm.toolEraser) {
                        activateToolFromHotkey('eraser');
                        return;
                    }
                    if (chord === currentHotkeysNorm.toolArrow) {
                        activateToolFromHotkey('arrow');
                        return;
                    }
                    if (chord === currentHotkeysNorm.toolRect) {
                        activateToolFromHotkey('rect');
                        return;
                    }
                    if (chord === currentHotkeysNorm.toolCircle) {
                        activateToolFromHotkey('circle');
                        return;
                    }
                    if (chord === currentHotkeysNorm.toolRefimg) {
                        openRefImagePicker();
                        return;
                    }
                    if (chord === currentHotkeysNorm.toolRecord) {
                        if (recordSelectActive) {
                            cancelRecordUi();
                            return;
                        }
                        startRecordRegionPick();
                        return;
                    }
                }
                if (chord && currentHotkeysNorm && drawingEnabled) {
                    if (chord === currentHotkeysNorm.undo) {
                        undo();
                        return;
                    }
                    if (chord === currentHotkeysNorm.clear) {
                        drawCtx.clearRect(0, 0, drawCanvas.width, drawCanvas.height);
                        clearTemp();
                        history = [];
                        return;
                    }
                }
            }
        }
        catch { }
        // 如果尚未锁定且已包含非修饰键，则锁定当前组合
        if (!lockedChordActive && hasNonModifierPressed(pressedKeys)) {
            lockedChordLabel = buildLabelFromPressed(pressedKeys, capsLockOn);
            lockedChordActive = !!lockedChordLabel;
        }
        // 已锁定时，按下新键应扩展组合标签（支持 Q+W+E 等并发组合）
        if (lockedChordActive) {
            lockedChordLabel = buildLabelFromPressed(pressedKeys, capsLockOn);
        }
        // 锁定后持续展示锁定的组合；未锁定时展示实时组合（仅修饰键阶段）
        const toShow = lockedChordActive ? lockedChordLabel : buildLabelFromPressed(pressedKeys, capsLockOn);
        if (toShow)
            bubble.showHtml(`<span class="kb-text">${toShow}</span>`, `k:${toShow}`);
    }
    else if (type === 'up') {
        const id = getKeyIdFromEvent(data);
        if (!pressedKeys.delete(id)) {
            const rawcode = typeof data.rawcode === 'number' ? data.rawcode : null;
            const keycode = typeof data.keycode === 'number' ? data.keycode : null;
            const name = nameFromEvent(data);
            for (const [k, v] of pressedKeys.entries()) {
                if (!v)
                    continue;
                if (rawcode !== null && v.rawcode === rawcode) {
                    pressedKeys.delete(k);
                    break;
                }
                if (keycode !== null && v.keycode === keycode) {
                    pressedKeys.delete(k);
                    break;
                }
                if (name && v.name === name) {
                    pressedKeys.delete(k);
                    break;
                }
            }
        }
        // 若全部松开，解除锁定；保持已有气泡让其按原定时淡出
        if (pressedKeys.size === 0) {
            lockedChordActive = false;
            lockedChordLabel = '';
            return;
        }
        // 锁定期间不降级显示为剩余按键；未锁定时仍显示实时组合（仅修饰键阶段）
        if (!lockedChordActive) {
            const chord = buildLabelFromPressed(pressedKeys, capsLockOn);
            if (chord)
                bubble.showText(chord);
        }
        else {
            // 维持锁定标签以延长可见性（可选：不重置淡出计时）
            bubble.showText(lockedChordLabel);
        }
    }
}
function onMouseEvent(payload) {
    if (!payload || typeof payload !== 'object')
        return;
    if (screenshotSelectActive && screenshotSelectSession) {
        const s = screenshotSelectSession;
        const t = payload.type;
        const btn = Number(payload.data && payload.data.button) || 0;
        const pos = payload.pos && typeof payload.pos === 'object' ? payload.pos : null;
        if (!pos)
            return;
        if (t === 'down') {
            if (btn && btn !== 1) {
                if (typeof screenshotSelectCancel === 'function')
                    screenshotSelectCancel();
                return;
            }
            const p = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
            const r0 = s && s.rect ? s.rect : { x: 0, y: 0, w: 0, h: 0 };
            const inside = !!(s.created && p.x >= r0.x && p.y >= r0.y && p.x <= (r0.x + r0.w) && p.y <= (r0.y + r0.h));
            const dir = inside ? String(s.edgeDirAt(p, r0) || '') : '';
            const now = Date.now();
            if (inside && !dir) {
                const lp = s.lastClickPos;
                if (lp && (now - (Number(s.lastClickTs) || 0)) < 350 && Math.abs((Number(lp.x) || 0) - p.x) <= 4 && Math.abs((Number(lp.y) || 0) - p.y) <= 4) {
                    if ((Number(r0.w) || 0) >= 2 && (Number(r0.h) || 0) >= 2)
                        s.done({ rect: r0, viewport: { width: s.vw, height: s.vh } });
                    return;
                }
                s.lastClickTs = now;
                s.lastClickPos = p;
            }
            s.start = p;
            s.dragging = true;
            s.anchorRect = { x: r0.x, y: r0.y, w: r0.w, h: r0.h };
            s.resizeDir = '';
            s.moveOffset = null;
            if (inside && dir) {
                s.mode = 'resize';
                s.resizeDir = dir;
                return;
            }
            if (inside) {
                s.mode = 'move';
                s.moveOffset = { x: p.x - r0.x, y: p.y - r0.y };
                return;
            }
            s.mode = 'create';
            s.created = false;
            s.rect = { x: p.x, y: p.y, w: 0, h: 0 };
            setSelectBoxRect(s.rect);
            return;
        }
        if (t === 'up') {
            if (btn && btn !== 1)
                return;
            if (!s.dragging || !s.start) {
                if (!s.created && typeof screenshotSelectCancel === 'function')
                    screenshotSelectCancel();
                return;
            }
            s.dragging = false;
            if (s.mode === 'create') {
                const r = s.normalize(s.start, pos);
                const next = s.clampRect(r);
                if (next.w < 2 || next.h < 2) {
                    const prev = s.anchorRect;
                    if (prev && (Number(prev.w) || 0) >= 2 && (Number(prev.h) || 0) >= 2) {
                        s.rect = prev;
                        s.created = true;
                        setSelectBoxRect(prev);
                        s.mode = '';
                        s.start = null;
                        s.anchorRect = null;
                        s.resizeDir = '';
                        s.moveOffset = null;
                        return;
                    }
                    if (typeof screenshotSelectCancel === 'function')
                        screenshotSelectCancel();
                    return;
                }
                s.rect = next;
                s.created = true;
                setSelectBoxRect(next);
                s.mode = '';
                s.start = null;
                s.anchorRect = null;
                s.resizeDir = '';
                s.moveOffset = null;
                return;
            }
            s.mode = '';
            s.start = null;
            s.anchorRect = null;
            s.resizeDir = '';
            s.moveOffset = null;
            return;
        }
        return;
    }
    if (recordSelectActive && recordSelectSession) {
        if (recordState !== 'idle' && recordState !== 'stopped')
            return;
        const s = recordSelectSession;
        const t = payload.type;
        const btn = Number(payload.data && payload.data.button) || 0;
        const pos = payload.pos && typeof payload.pos === 'object' ? payload.pos : null;
        if (!pos)
            return;
        if (t === 'down') {
            if (btn && btn !== 1) {
                if (typeof recordSelectCancel === 'function')
                    recordSelectCancel();
                return;
            }
            const p = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
            if (recordUiHit(p))
                return;
            const r0 = s && s.rect ? s.rect : { x: 0, y: 0, w: 0, h: 0 };
            const inside = !!(s.created && p.x >= r0.x && p.y >= r0.y && p.x <= (r0.x + r0.w) && p.y <= (r0.y + r0.h));
            const dir = inside ? String(s.edgeDirAt(p, r0) || '') : '';
            s.start = p;
            s.dragging = true;
            s.anchorRect = { x: r0.x, y: r0.y, w: r0.w, h: r0.h };
            s.resizeDir = '';
            s.moveOffset = null;
            if (inside && dir) {
                s.mode = 'resize';
                s.resizeDir = dir;
                return;
            }
            if (inside) {
                s.mode = 'move';
                s.moveOffset = { x: p.x - r0.x, y: p.y - r0.y };
                return;
            }
            s.mode = 'create';
            s.created = false;
            s.rect = { x: p.x, y: p.y, w: 0, h: 0 };
            setRecordBoxRect(s.rect);
            return;
        }
        if (t === 'up') {
            if (btn && btn !== 1)
                return;
            if (!s.dragging || !s.start) {
                if (!s.created && typeof recordSelectCancel === 'function')
                    recordSelectCancel();
                return;
            }
            s.dragging = false;
            if (s.mode === 'create') {
                const r = s.normalize(s.start, pos);
                const next = s.clampRect(r);
                if (next.w < 2 || next.h < 2) {
                    const prev = s.anchorRect;
                    if (prev && (Number(prev.w) || 0) >= 2 && (Number(prev.h) || 0) >= 2) {
                        s.rect = prev;
                        s.created = true;
                        setRecordBoxRect(prev);
                        setRecordToolbarVisible(true);
                        setRecordHotzoneVisible(false);
                        s.mode = '';
                        s.start = null;
                        s.anchorRect = null;
                        s.resizeDir = '';
                        s.moveOffset = null;
                        return;
                    }
                    if (typeof recordSelectCancel === 'function')
                        recordSelectCancel();
                    return;
                }
                s.rect = next;
                s.created = true;
                setRecordBoxRect(next);
                setRecordToolbarVisible(true);
                setRecordHotzoneVisible(false);
                s.mode = '';
                s.start = null;
                s.anchorRect = null;
                s.resizeDir = '';
                s.moveOffset = null;
                return;
            }
            if (s.created) {
                setRecordToolbarVisible(true);
                setRecordHotzoneVisible(false);
            }
            s.mode = '';
            s.start = null;
            s.anchorRect = null;
            s.resizeDir = '';
            s.moveOffset = null;
            return;
        }
        return;
    }
    if (payload.type !== 'down')
        return;
    mouse.pulseClick(payload);
    if (payload.bubble) {
        const label = mouseButtonLabel(payload.data);
        const keyId = `m${(payload.data && payload.data.button) || 0}`;
        if (shouldShowMouseBubble(keyId)) {
            const svg = mouseButtonIconSVG(payload.data && payload.data.button, label);
            bubble.showHtml(`${svg}<span class="kb-text">${label}</span>`, keyId);
        }
    }
}
if (window.tt) {
    window.tt.onKeyEvent(onKeyPayload);
    window.tt.onMouseEvent(onMouseEvent);
    try {
        if (typeof window.tt.onPinnedShotsState === 'function')
            window.tt.onPinnedShotsState((p) => applyPinnedShotsState(p));
    }
    catch { }
    try {
        if (typeof window.tt.onRefImagesState === 'function')
            window.tt.onRefImagesState((p) => applyRefImagesState(p));
    }
    catch { }
    window.tt.onMouseMove((pos) => {
        mouse.update(pos);
        if (screenshotSelectActive && screenshotSelectSession) {
            const s = screenshotSelectSession;
            if (!pos || typeof pos !== 'object')
                return;
            const p = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
            const rect0 = s && s.rect ? s.rect : { x: 0, y: 0, w: 0, h: 0 };
            if (screenshotSelectLayer && screenshotSelectLayer.isConnected) {
                try {
                    if (!s.created || (Number(rect0.w) || 0) < 2 || (Number(rect0.h) || 0) < 2) {
                        screenshotSelectLayer.style.cursor = 'crosshair';
                    }
                    else {
                        const inside = p.x >= rect0.x && p.y >= rect0.y && p.x <= (rect0.x + rect0.w) && p.y <= (rect0.y + rect0.h);
                        if (!inside)
                            screenshotSelectLayer.style.cursor = 'crosshair';
                        else {
                            const dir = String(s.edgeDirAt(p, rect0) || '');
                            screenshotSelectLayer.style.cursor = dir ? s.cursorForDir(dir) : 'move';
                        }
                    }
                }
                catch { }
            }
            if (!s.dragging || !s.start)
                return;
            if (s.mode === 'create') {
                const r = s.normalize(s.start, p);
                const next = s.clampRect(r);
                s.rect = next;
                setSelectBoxRect(next);
                return;
            }
            if (s.mode === 'move') {
                const ar = s.anchorRect;
                const off = s.moveOffset;
                if (!ar || !off)
                    return;
                const w = Math.max(2, Math.round(Number(ar.w) || 0));
                const h = Math.max(2, Math.round(Number(ar.h) || 0));
                const maxX = Math.max(0, s.vw - w);
                const maxY = Math.max(0, s.vh - h);
                const nx = Math.max(0, Math.min(maxX, Math.round(p.x - (Number(off.x) || 0))));
                const ny = Math.max(0, Math.min(maxY, Math.round(p.y - (Number(off.y) || 0))));
                const next = { x: nx, y: ny, w, h };
                s.rect = next;
                s.created = true;
                setSelectBoxRect(next);
                return;
            }
            if (s.mode === 'resize') {
                const ar = s.anchorRect;
                const dir = String(s.resizeDir || '');
                if (!ar || !dir)
                    return;
                const min = 2;
                let x = Math.round(Number(ar.x) || 0);
                let y = Math.round(Number(ar.y) || 0);
                let w = Math.max(min, Math.round(Number(ar.w) || 0));
                let h = Math.max(min, Math.round(Number(ar.h) || 0));
                const r = { x, y, w, h };
                if (dir.includes('e'))
                    r.w = Math.max(min, Math.min(s.vw - r.x, Math.round(p.x - r.x)));
                if (dir.includes('s'))
                    r.h = Math.max(min, Math.min(s.vh - r.y, Math.round(p.y - r.y)));
                if (dir.includes('w')) {
                    const maxX = r.x + r.w - min;
                    const nx = Math.max(0, Math.min(maxX, Math.round(p.x)));
                    r.w = Math.max(min, (r.x + r.w) - nx);
                    r.x = nx;
                }
                if (dir.includes('n')) {
                    const maxY = r.y + r.h - min;
                    const ny = Math.max(0, Math.min(maxY, Math.round(p.y)));
                    r.h = Math.max(min, (r.y + r.h) - ny);
                    r.y = ny;
                }
                const next = s.clampRect(r);
                s.rect = next;
                s.created = true;
                setSelectBoxRect(next);
                return;
            }
        }
        if (activeRefImageId && !refPickerActive && !refCropActive) {
            if (!pos || typeof pos !== 'object')
                return;
            const p = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
            updateRefToolbarAutoVisibility(p);
        }
        if (recordSelectActive && recordSelectSession) {
            if (!pos || typeof pos !== 'object')
                return;
            ensureRecordElements();
            const p = { x: Math.round(Number(pos.x) || 0), y: Math.round(Number(pos.y) || 0) };
            updateRecordToolbarAutoVisibility(p);
            if (recordState !== 'idle' && recordState !== 'stopped') {
                try {
                    if (recordLayer && recordLayer.isConnected)
                        recordLayer.style.cursor = 'default';
                }
                catch { }
                return;
            }
            const s = recordSelectSession;
            const rect0 = s && s.rect ? s.rect : { x: 0, y: 0, w: 0, h: 0 };
            if (recordLayer && recordLayer.isConnected) {
                try {
                    if (!s.created || (Number(rect0.w) || 0) < 2 || (Number(rect0.h) || 0) < 2) {
                        recordLayer.style.cursor = 'crosshair';
                    }
                    else {
                        const inside = p.x >= rect0.x && p.y >= rect0.y && p.x <= (rect0.x + rect0.w) && p.y <= (rect0.y + rect0.h);
                        if (!inside)
                            recordLayer.style.cursor = 'crosshair';
                        else {
                            const dir = String(s.edgeDirAt(p, rect0) || '');
                            recordLayer.style.cursor = dir ? s.cursorForDir(dir) : 'move';
                        }
                    }
                }
                catch { }
            }
            if (!s.dragging || !s.start)
                return;
            if (s.mode === 'create') {
                const r = s.normalize(s.start, p);
                const next = s.clampRect(r);
                s.rect = next;
                setRecordBoxRect(next);
                return;
            }
            if (s.mode === 'move') {
                const ar = s.anchorRect;
                const off = s.moveOffset;
                if (!ar || !off)
                    return;
                const w = Math.max(2, Math.round(Number(ar.w) || 0));
                const h = Math.max(2, Math.round(Number(ar.h) || 0));
                const maxX = Math.max(0, s.vw - w);
                const maxY = Math.max(0, s.vh - h);
                const nx = Math.max(0, Math.min(maxX, Math.round(p.x - (Number(off.x) || 0))));
                const ny = Math.max(0, Math.min(maxY, Math.round(p.y - (Number(off.y) || 0))));
                const next = { x: nx, y: ny, w, h };
                s.rect = next;
                s.created = true;
                setRecordBoxRect(next);
                return;
            }
            if (s.mode === 'resize') {
                const ar = s.anchorRect;
                const dir = String(s.resizeDir || '');
                if (!ar || !dir)
                    return;
                const min = 2;
                let x = Math.round(Number(ar.x) || 0);
                let y = Math.round(Number(ar.y) || 0);
                let w = Math.max(min, Math.round(Number(ar.w) || 0));
                let h = Math.max(min, Math.round(Number(ar.h) || 0));
                const r = { x, y, w, h };
                if (dir.includes('e'))
                    r.w = Math.max(min, Math.min(s.vw - r.x, Math.round(p.x - r.x)));
                if (dir.includes('s'))
                    r.h = Math.max(min, Math.min(s.vh - r.y, Math.round(p.y - r.y)));
                if (dir.includes('w')) {
                    const maxX = r.x + r.w - min;
                    const nx = Math.max(0, Math.min(maxX, Math.round(p.x)));
                    r.w = Math.max(min, (r.x + r.w) - nx);
                    r.x = nx;
                }
                if (dir.includes('n')) {
                    const maxY = r.y + r.h - min;
                    const ny = Math.max(0, Math.min(maxY, Math.round(p.y)));
                    r.h = Math.max(min, (r.y + r.h) - ny);
                    r.y = ny;
                }
                const next = s.clampRect(r);
                s.rect = next;
                s.created = true;
                setRecordBoxRect(next);
            }
        }
    });
    window.tt.onMouseVisible((visible) => mouse.setVisible(visible));
    try {
        if (typeof window.tt.onScreenshotCommand === 'function') {
            window.tt.onScreenshotCommand((p) => {
                const pp = p;
                const kind = pp && typeof pp === 'object' ? pp.kind : 'save';
                doScreenshot(kind);
            });
        }
    }
    catch { }
    try {
        if (typeof window.tt.onRecordCommand === 'function') {
            window.tt.onRecordCommand((p) => {
                const pp = p;
                const action = pp && typeof pp === 'object' ? String(pp.action || '').trim() : '';
                if (!action)
                    return;
                if (action === 'toggle') {
                    if (!recordSelectActive && (!recordSelectSession || !recordSelectSession.created)) {
                        startRecordRegionPick();
                        return;
                    }
                    toggleRecordStartPause();
                    return;
                }
                if (action === 'saveGif') {
                    if (!recordSelectSession || !recordSelectSession.created) {
                        bubble.showText('请先拖拽选择录屏区域');
                        return;
                    }
                    saveRecordAs('gif');
                    return;
                }
                if (action === 'saveMp4') {
                    if (!recordSelectSession || !recordSelectSession.created) {
                        bubble.showText('请先拖拽选择录屏区域');
                        return;
                    }
                    saveRecordAs('mp4');
                }
            });
        }
    }
    catch { }
    try {
        const initSettings = await window.tt.getSettings();
        applySettings(initSettings);
        window.tt.onSettingsUpdated((s) => applySettings(s));
    }
    catch { }
}
// 浏览器预览降级（无全局输入）：展示示例气泡
if (!window.tt) {
    const demo = ['Ctrl + Shift + P', 'Alt + Tab', 'F5', 'A', 'Enter'];
    let i = 0;
    setInterval(() => { bubble.showText(demo[i % demo.length]); i += 1; }, 1200);
}
//# sourceMappingURL=main.js.map