// 键盘/鼠标事件解析与标签构建（精简版）
const VK_NAME = new Map([
    [8, 'Backspace'], [9, 'Tab'], [13, 'Enter'], [16, 'Shift'], [17, 'Ctrl'], [18, 'Alt'],
    [27, 'Esc'], [32, 'Space'], [33, 'PageUp'], [34, 'PageDown'], [35, 'End'], [36, 'Home'],
    [37, 'Left'], [38, 'Up'], [39, 'Right'], [40, 'Down'], [44, 'PrtSc'], [45, 'Insert'], [46, 'Delete'],
    [91, 'Win'], [92, 'Win'],
]);
for (let i = 0; i < 10; i++)
    VK_NAME.set(48 + i, String(i));
for (let i = 0; i < 26; i++)
    VK_NAME.set(65 + i, String.fromCharCode(65 + i));
for (let i = 1; i <= 12; i++)
    VK_NAME.set(111 + i, `F${i}`);
for (let i = 0; i < 10; i++)
    VK_NAME.set(96 + i, `Num${i}`);
VK_NAME.set(106, 'Num*');
VK_NAME.set(107, 'Num+');
VK_NAME.set(109, 'Num-');
VK_NAME.set(110, 'Num.');
VK_NAME.set(111, 'Num/');
VK_NAME.set(144, 'Num Lock');
const RAW_MOD_SET = new Set([16, 17, 18, 91, 92, 160, 161, 162, 163, 164, 165]);
const MASK_SHIFT = 1, MASK_CTRL = 2, MASK_ALT = 4, MASK_META = 8;
const KC_NAME = new Map([
    [1, 'Esc'], [14, 'Backspace'], [15, 'Tab'], [28, 'Enter'], [57, 'Space'],
    [58, 'Caps Lock'],
    [12, '-'], [13, '='], [26, '['], [27, ']'], [43, '\\'], [39, ';'], [40, "'"], [41, '`'], [51, ','], [52, '.'], [53, '/'],
    [71, 'Num7'], [72, 'Num8'], [73, 'Num9'],
    [75, 'Num4'], [77, 'Num6'],
    [76, 'Num5'],
    [79, 'Num1'], [80, 'Num2'], [81, 'Num3'],
    [82, 'Num0'], [83, 'Num.'],
    [55, 'Num*'], [78, 'Num+'], [74, 'Num-'],
    [309, 'Num/'],
    [284, 'NumEnter'],
    [69, 'Num Lock'],
    [199, 'Home'], [207, 'End'], [201, 'PageUp'], [209, 'PageDown'],
    [200, 'Up'], [208, 'Down'], [203, 'Left'], [205, 'Right'],
    [3612, 'NumEnter'], [3637, 'Num/'],
    [3655, 'Home'], [3663, 'End'], [3657, 'PageUp'], [3665, 'PageDown'],
    [3656, 'Up'], [3664, 'Down'], [3659, 'Left'], [3661, 'Right'],
    [2, '1'], [3, '2'], [4, '3'], [5, '4'], [6, '5'], [7, '6'], [8, '7'], [9, '8'], [10, '9'], [11, '0'],
    [59, 'F1'], [60, 'F2'], [61, 'F3'], [62, 'F4'], [63, 'F5'], [64, 'F6'], [65, 'F7'], [66, 'F8'], [67, 'F9'], [68, 'F10'], [87, 'F11'], [88, 'F12'],
    [16, 'Q'], [17, 'W'], [18, 'E'], [19, 'R'], [20, 'T'], [21, 'Y'], [22, 'U'], [23, 'I'], [24, 'O'], [25, 'P'],
    [30, 'A'], [31, 'S'], [32, 'D'], [33, 'F'], [34, 'G'], [35, 'H'], [36, 'J'], [37, 'K'], [38, 'L'],
    [44, 'Z'], [45, 'X'], [46, 'C'], [47, 'V'], [48, 'B'], [49, 'N'], [50, 'M'],
    [42, 'Shift'], [54, 'Shift'], [29, 'Ctrl'], [157, 'Ctrl'], [56, 'Alt'], [184, 'Alt'],
]);
const KC_NUMPAD_SET = new Set([71, 72, 73, 75, 76, 77, 79, 80, 81, 82, 83]);
const KC_EXT_OFFSET_1 = 3584;
const KC_EXT_OFFSET_2 = 57344;
const KC_EXT_NAME = new Map([
    [53, 'Num/'],
    [28, 'NumEnter'],
    [55, 'PrtSc'],
    [82, 'Insert'], [83, 'Delete'],
    [71, 'Home'], [79, 'End'], [73, 'PageUp'], [81, 'PageDown'],
    [72, 'Up'], [80, 'Down'], [75, 'Left'], [77, 'Right'],
    [91, 'Win'], [92, 'Win'],
    [29, 'Ctrl'], [54, 'Shift'], [56, 'Alt'],
]);
function toNum(v) { const n = Number(v); return Number.isFinite(n) ? n : null; }
export function isModifierEvent(e) {
    const rc = toNum(e && e.rawcode);
    const kc = toNum(e && e.keycode);
    if (typeof rc === 'number' && RAW_MOD_SET.has(rc))
        return true;
    if (kc === 42 || kc === 54)
        return true;
    if (kc === 29 || kc === 157)
        return true;
    if (kc === 56 || kc === 184)
        return true;
    return false;
}
function modifiersFromEvent(e) {
    const m = Number(e && e.mask) || 0;
    return {
        ctrl: !!(m & MASK_CTRL) || !!e.ctrlKey,
        shift: !!(m & MASK_SHIFT) || !!e.shiftKey,
        alt: !!(m & MASK_ALT) || !!e.altKey,
        meta: !!(m & MASK_META) || !!e.metaKey,
    };
}
export function baseKeyName(e) {
    const rc = toNum(e && e.rawcode);
    const kcRaw = toNum(e && e.keycode);
    const offsets = [KC_EXT_OFFSET_1, KC_EXT_OFFSET_2];
    if (kcRaw !== null) {
        for (const off of offsets) {
            if (kcRaw >= off && kcRaw < off + 256) {
                const sc = kcRaw - off;
                if (KC_EXT_NAME.has(sc))
                    return KC_EXT_NAME.get(sc);
                break;
            }
        }
    }
    const kc = kcRaw;
    if (kc !== null && KC_NUMPAD_SET.has(kc) && KC_NAME.has(kc))
        return KC_NAME.get(kc);
    if (kc !== null && ((kc >= 96 && kc <= 111) || kc === 144) && VK_NAME.has(kc))
        return VK_NAME.get(kc);
    if (rc !== null && VK_NAME.has(rc))
        return VK_NAME.get(rc);
    if (kc !== null && KC_NAME.has(kc))
        return KC_NAME.get(kc);
    if (e && e.keychar && Number(e.keychar) > 0) {
        const ch = String.fromCharCode(Number(e.keychar));
        return e.shiftKey ? ch.toUpperCase() : ch;
    }
    if (rc !== null)
        return `#${rc}`;
    if (kcRaw !== null)
        return `KC${kcRaw}`;
    return 'Key';
}
export function labelFromEvent(e) {
    const mods = modifiersFromEvent(e);
    const parts = [];
    if (mods.ctrl)
        parts.push('Ctrl');
    if (mods.shift)
        parts.push('Shift');
    if (mods.alt)
        parts.push('Alt');
    if (mods.meta)
        parts.push('Win');
    const base = baseKeyName(e);
    if (isModifierEvent(e))
        return parts.length ? parts.join(' + ') : base;
    return parts.length ? parts.join(' + ') + ' + ' + base : base;
}
export function getKeyIdFromEvent(e) {
    if (typeof e.rawcode === 'number')
        return `r${e.rawcode}`;
    if (typeof e.keycode === 'number')
        return `k${e.keycode}`;
    if (typeof e.keychar === 'number' && e.keychar > 0)
        return `c${e.keychar}`;
    return `u${Date.now()}`;
}
export function nameFromEvent(e) { return baseKeyName(e); }
export const MOD_ORDER = ['Ctrl', 'Shift', 'Alt', 'Win'];
const SHIFTED_SYMBOLS = new Map([
    ['1', '!'], ['2', '@'], ['3', '#'], ['4', '$'], ['5', '%'], ['6', '^'], ['7', '&'], ['8', '*'], ['9', '('], ['0', ')'],
    ['-', '_'], ['=', '+'], ['[', '{'], [']', '}'], ['\\', '|'], [';', ':'], ['\'', '"'], [',', '<'], ['.', '>'], ['/', '?'], ['`', '~']
]);
function isAsciiLetter(name) { return typeof name === 'string' && name.length === 1 && name >= 'A' && name <= 'Z'; }
function isAsciiDigit(name) { return typeof name === 'string' && name.length === 1 && name >= '0' && name <= '9'; }
function isAsciiPunct(name) { return typeof name === 'string' && SHIFTED_SYMBOLS.has(name); }
function computePrintable(name, mods, capsLockOn) {
    if (name === 'Space')
        return 'Space';
    if (isAsciiLetter(name)) {
        const shift = mods.includes('Shift');
        const upper = (!!shift) !== (!!capsLockOn);
        return upper ? name : name.toLowerCase();
    }
    if (isAsciiDigit(name)) {
        const shift = mods.includes('Shift');
        if (shift && SHIFTED_SYMBOLS.has(name))
            return SHIFTED_SYMBOLS.get(name);
        return name;
    }
    if (isAsciiPunct(name)) {
        const shift = mods.includes('Shift');
        if (shift)
            return SHIFTED_SYMBOLS.get(name);
        return name;
    }
    return null;
}
export function buildLabelFromPressed(map, capsLockOn = false) {
    const nameSet = new Set();
    map.forEach(v => { const n = v && v.name ? v.name : ''; if (n)
        nameSet.add(n); });
    const mods = MOD_ORDER.filter(m => nameSet.has(m));
    const others = Array.from(nameSet).filter(n => !MOD_ORDER.includes(n));
    if (others.length === 1) {
        const printable = computePrintable(others[0], mods, !!capsLockOn);
        const hasNonShiftMods = mods.some(m => m !== 'Shift');
        if (printable && !hasNonShiftMods)
            return printable;
    }
    const othersFinal = others.map(n => {
        const p = computePrintable(n, mods, !!capsLockOn);
        return p || n;
    });
    const parts = mods.concat(othersFinal);
    return parts.length ? parts.join(' + ') : '';
}
export function mouseButtonLabel(evt) {
    const b = Number(evt && evt.button);
    switch (b) {
        case 1: return '左键';
        case 2: return '右键';
        case 3: return '中键';
        case 4: return '后退';
        case 5: return '前进';
        default: return b ? `鼠标${b}` : '鼠标';
    }
}
export function mouseButtonIconSVG(button, label) {
    const stroke = 'stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" fill="none"';
    const sizeAttr = 'style="width:var(--mouse-icon-size, 18px);height:var(--mouse-icon-size, 18px)"';
    const defs = `<defs><clipPath id="mouse-clip"><rect x="3" y="1" width="14" height="18" rx="7"/></clipPath></defs>`;
    const outline = `<rect x="3" y="1" width="14" height="18" rx="7" ${stroke}/><line x1="10" y1="1" x2="10" y2="10" ${stroke}/>`;
    let highlight = '';
    switch (Number(button)) {
        case 1:
            highlight = '<rect x="3" y="1" width="7" height="10" fill="currentColor" clip-path="url(#mouse-clip)"/>';
            break;
        case 2:
            highlight = '<rect x="10" y="1" width="7" height="10" fill="currentColor" clip-path="url(#mouse-clip)"/>';
            break;
        case 3:
            highlight = '<rect x="9" y="1" width="2" height="9" fill="currentColor" clip-path="url(#mouse-clip)"/>';
            break;
        case 4: return `<svg viewBox="0 0 20 20" ${sizeAttr} aria-label="${label}" role="img"><circle cx="10" cy="10" r="8" ${stroke}/><polyline points="11,6 7,10 11,14" ${stroke}/></svg>`;
        case 5: return `<svg viewBox="0 0 20 20" ${sizeAttr} aria-label="${label}" role="img"><circle cx="10" cy="10" r="8" ${stroke}/><polyline points="9,6 13,10 9,14" ${stroke}/></svg>`;
        default: return `<svg viewBox="0 0 20 20" ${sizeAttr} aria-label="${label || '鼠标'}" role="img"><circle cx="10" cy="10" r="8" ${stroke}/></svg>`;
    }
    return `<svg viewBox="0 0 20 20" ${sizeAttr} aria-label="${label}" role="img">${defs}${highlight}${outline}</svg>`;
}
//# sourceMappingURL=keymap.js.map