// main.ts
// Electron 主进程：为每个显示器创建透明置顶窗口，使用 uiohook-napi 捕获全局键鼠事件
const { app, BrowserWindow, Tray, Menu, nativeImage, clipboard, dialog, globalShortcut, screen, ipcMain, desktopCapturer } = require('electron');
const path = require('path');
const fs = require('fs');
const { spawn, spawnSync } = require('child_process');
const { uIOhook } = require('uiohook-napi');
app.commandLine.appendSwitch('disable-gpu-shader-disk-cache');
const MAX_PNG_BYTES = 25 * 1024 * 1024;
const MAX_REF_IMAGE_BYTES = 40 * 1024 * 1024;
const MAX_RECORD_BYTES = 512 * 1024 * 1024;
function estimateBase64Bytes(raw) {
    const s = String(raw || '').trim();
    if (!s)
        return 0;
    const pad = s.endsWith('==') ? 2 : s.endsWith('=') ? 1 : 0;
    return Math.max(0, Math.floor((s.length * 3) / 4) - pad);
}
function isBase64ish(raw) {
    const s = String(raw || '').trim();
    if (!s)
        return false;
    return /^[A-Za-z0-9+/]+={0,2}$/.test(s);
}
function parseBase64ToBuffer(raw, maxBytes) {
    const s = String(raw || '').trim();
    if (!s)
        return null;
    const est = estimateBase64Bytes(s);
    const limit = Math.max(1, Number(maxBytes) || 1);
    if (!Number.isFinite(est) || est <= 0 || est > limit)
        return null;
    if (!isBase64ish(s))
        return null;
    try {
        const buf = Buffer.from(s, 'base64');
        if (!buf || !buf.length)
            return null;
        if (buf.length > limit)
            return null;
        return buf;
    }
    catch {
        return null;
    }
}
function normalizeDataUrl(raw, maxBytes) {
    const s = String(raw || '').trim();
    if (!s)
        return '';
    const idx = s.indexOf(';base64,');
    if (!s.startsWith('data:') || idx < 0)
        return '';
    const head = s.slice(0, idx + ';base64,'.length);
    const b64 = s.slice(idx + ';base64,'.length);
    const buf = parseBase64ToBuffer(b64, maxBytes);
    if (!buf)
        return '';
    return `${head}${buf.toString('base64')}`;
}
const gotSingleInstanceLock = app.requestSingleInstanceLock();
if (!gotSingleInstanceLock) {
    try {
        app.quit();
    }
    catch (err) {
        console.error('[main] Failed to quit app:', err);
    }
    process.exit(0);
}
app.on('second-instance', () => { try {
    refreshTray();
}
catch (err) {
    console.error('[main] Failed to refresh tray:', err);
} });
let wins = new Map();
let tray;
let settingsWin;
let lastSettingsWillMoveLogAt = 0;
let suppressOverlayTopmost = false;
let toolHoldCount = 0;
let restoreSettingsAfterTool = false;
let overlayVisible = true;
let mouseButtonVisible = true;
let mouseDotMoveVisible = true;
let mouseDotClickVisible = true;
let screenshotPickLock = null;
let recordPickLock = null;
const readyWins = new Set();
let debugKeysCount = 0;
const keyRouteById = new Map();
const keyRouteTimestamps = new Map();
const assetsIconDir = path.join(__dirname, 'assets', 'icons');
const rendererCaptureByBwId = new Map();
const pinnedRectsByBwId = new Map();
const pinnedHoverByBwId = new Map();
const interactiveRectsByBwId = new Map();
const interactiveHoverByBwId = new Map();
const pinnedShotRectsByBwId = new Map();
const refImageRectsByBwId = new Map();
const pinnedShotsById = new Map();
let pinnedShotOrder = [];
let pinnedDrag = null;
const refImagesById = new Map();
let refImageOrder = [];
let refDrag = null;
let settingsDrag = null;
let pinnedBroadcastQueued = false;
const KEY_ROUTE_TTL = 30000;
const CLEANUP_INTERVAL = 30000;
const MAX_PINNED_SHOTS = 50;
const MAX_REF_IMAGES = 50;
const MAX_KEY_ROUTES = 1000;
let cleanupInterval = null;
function cleanupExpiredKeyRoutes() {
    try {
        const now = Date.now();
        const expiredKeys = [];
        for (const [key, timestamp] of keyRouteTimestamps.entries()) {
            if (now - timestamp > KEY_ROUTE_TTL) {
                expiredKeys.push(key);
            }
        }
        for (const key of expiredKeys) {
            keyRouteById.delete(key);
            keyRouteTimestamps.delete(key);
        }
        if (keyRouteById.size > MAX_KEY_ROUTES) {
            const entries = Array.from(keyRouteTimestamps.entries())
                .sort((a, b) => a[1] - b[1]);
            const toRemove = entries.slice(0, keyRouteById.size - MAX_KEY_ROUTES);
            for (const [key] of toRemove) {
                keyRouteById.delete(key);
                keyRouteTimestamps.delete(key);
            }
        }
        if (expiredKeys.length > 0) {
            console.log(`[cleanup] Removed ${expiredKeys.length} expired key routes`);
        }
    }
    catch (err) {
        console.error('[cleanup] Failed to cleanup key routes:', err);
    }
}
function cleanupPinnedShots() {
    try {
        if (pinnedShotsById.size <= MAX_PINNED_SHOTS)
            return;
        const toRemove = pinnedShotOrder.slice(0, pinnedShotOrder.length - MAX_PINNED_SHOTS);
        for (const id of toRemove) {
            pinnedShotsById.delete(id);
        }
        pinnedShotOrder = pinnedShotOrder.slice(toRemove.length);
        console.log(`[cleanup] Removed ${toRemove.length} old pinned shots`);
    }
    catch (err) {
        console.error('[cleanup] Failed to cleanup pinned shots:', err);
    }
}
function cleanupRefImages() {
    try {
        if (refImagesById.size <= MAX_REF_IMAGES)
            return;
        const toRemove = refImageOrder.slice(0, refImageOrder.length - MAX_REF_IMAGES);
        for (const id of toRemove) {
            refImagesById.delete(id);
        }
        refImageOrder = refImageOrder.slice(toRemove.length);
        console.log(`[cleanup] Removed ${toRemove.length} old ref images`);
    }
    catch (err) {
        console.error('[cleanup] Failed to cleanup ref images:', err);
    }
}
function cleanupDestroyedWindowData() {
    try {
        const validIds = new Set();
        for (const [displayId, bw] of wins.entries()) {
            if (bw && !bw.isDestroyed()) {
                validIds.add(bw.id);
            }
        }
        for (const [bwId] of pinnedRectsByBwId.entries()) {
            if (!validIds.has(bwId))
                pinnedRectsByBwId.delete(bwId);
        }
        for (const [bwId] of pinnedHoverByBwId.entries()) {
            if (!validIds.has(bwId))
                pinnedHoverByBwId.delete(bwId);
        }
        for (const [bwId] of interactiveRectsByBwId.entries()) {
            if (!validIds.has(bwId))
                interactiveRectsByBwId.delete(bwId);
        }
        for (const [bwId] of interactiveHoverByBwId.entries()) {
            if (!validIds.has(bwId))
                interactiveHoverByBwId.delete(bwId);
        }
        for (const [bwId] of pinnedShotRectsByBwId.entries()) {
            if (!validIds.has(bwId))
                pinnedShotRectsByBwId.delete(bwId);
        }
        for (const [bwId] of refImageRectsByBwId.entries()) {
            if (!validIds.has(bwId))
                refImageRectsByBwId.delete(bwId);
        }
        for (const [bwId] of rendererCaptureByBwId.entries()) {
            if (!validIds.has(bwId))
                rendererCaptureByBwId.delete(bwId);
        }
    }
    catch (err) {
        console.error('[cleanup] Failed to cleanup destroyed window data:', err);
    }
}
function runAllCleanup() {
    cleanupExpiredKeyRoutes();
    cleanupPinnedShots();
    cleanupRefImages();
    cleanupDestroyedWindowData();
}
function startCleanupInterval() {
    if (cleanupInterval)
        return;
    cleanupInterval = setInterval(() => {
        runAllCleanup();
    }, CLEANUP_INTERVAL);
}
function stopCleanupInterval() {
    if (cleanupInterval) {
        clearInterval(cleanupInterval);
        cleanupInterval = null;
    }
}
function setOverlaysTopmostEnabled(enabled) {
    suppressOverlayTopmost = !enabled;
    try {
        updateWindowVisibility();
    }
    catch (err) {
        console.error('[main] Failed to update window visibility:', err);
    }
}
function beginToolHold() {
    toolHoldCount = Math.max(0, toolHoldCount + 1);
    if (toolHoldCount !== 1)
        return;
    restoreSettingsAfterTool = false;
    if (settingsWin && !settingsWin.isDestroyed()) {
        try {
            const visible = settingsWin.isVisible && settingsWin.isVisible();
            const minimized = settingsWin.isMinimized && settingsWin.isMinimized();
            restoreSettingsAfterTool = !!visible && !minimized;
            if (restoreSettingsAfterTool)
                settingsWin.hide();
        }
        catch { }
    }
    setOverlaysTopmostEnabled(true);
}
function endToolHold() {
    toolHoldCount = Math.max(0, toolHoldCount - 1);
    if (toolHoldCount !== 0)
        return;
    if (settingsWin && !settingsWin.isDestroyed()) {
        if (restoreSettingsAfterTool) {
            try {
                settingsWin.show();
            }
            catch { }
            try {
                settingsWin.focus();
            }
            catch { }
        }
        setOverlaysTopmostEnabled(false);
    }
    else {
        setOverlaysTopmostEnabled(true);
    }
    restoreSettingsAfterTool = false;
}
function normalizePinnedRects(rects) {
    if (!Array.isArray(rects))
        return [];
    const out = [];
    for (const r0 of rects) {
        const r = (r0 && typeof r0 === 'object') ? r0 : null;
        if (!r)
            continue;
        const x = Math.round(Number(r.x) || 0);
        const y = Math.round(Number(r.y) || 0);
        const w = Math.max(1, Math.round(Number(r.w) || 0));
        const h = Math.max(1, Math.round(Number(r.h) || 0));
        out.push({ x, y, w, h });
    }
    return out;
}
function pointInRect(p, r) {
    if (!p || !r)
        return false;
    return p.x >= r.x && p.x < r.x + r.w && p.y >= r.y && p.y < r.y + r.h;
}
function computePinnedHover(bw, rects) {
    if (!bw || bw.isDestroyed() || !rects || !rects.length) {
        try {
            if (bw && !bw.isDestroyed())
                pinnedHoverByBwId.set(bw.id, false);
        }
        catch { }
        return false;
    }
    try {
        const pt = screen.getCursorScreenPoint();
        const b = bw.getBounds();
        const localPos = { x: pt.x - b.x, y: pt.y - b.y };
        const inside = rects.some(r => pointInRect(localPos, r));
        pinnedHoverByBwId.set(bw.id, inside);
        return inside;
    }
    catch {
        return pinnedHoverByBwId.get(bw.id) || false;
    }
}
function computeInteractiveHover(bw, rects) {
    if (!bw || bw.isDestroyed() || !rects || !rects.length) {
        try {
            if (bw && !bw.isDestroyed())
                interactiveHoverByBwId.set(bw.id, false);
        }
        catch { }
        return false;
    }
    try {
        const pt = screen.getCursorScreenPoint();
        const b = bw.getBounds();
        const localPos = { x: pt.x - b.x, y: pt.y - b.y };
        const inside = rects.some(r => pointInRect(localPos, r));
        interactiveHoverByBwId.set(bw.id, inside);
        return inside;
    }
    catch {
        return interactiveHoverByBwId.get(bw.id) || false;
    }
}
function normalizePinnedShotRect(r0) {
    const r = (r0 && typeof r0 === 'object') ? r0 : null;
    if (!r)
        return null;
    const x = Math.round(Number(r.x) || 0);
    const y = Math.round(Number(r.y) || 0);
    const w = Math.max(1, Math.round(Number(r.w) || 1));
    const h = Math.max(1, Math.round(Number(r.h) || 1));
    return { x, y, w, h };
}
function getVirtualBounds() {
    const displays = screen.getAllDisplays();
    if (!displays || !displays.length)
        return { x: 0, y: 0, width: 1, height: 1 };
    let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
    for (const d of displays) {
        const b = d && d.bounds ? d.bounds : null;
        if (!b)
            continue;
        minX = Math.min(minX, b.x);
        minY = Math.min(minY, b.y);
        maxX = Math.max(maxX, b.x + b.width);
        maxY = Math.max(maxY, b.y + b.height);
    }
    if (!Number.isFinite(minX) || !Number.isFinite(minY) || !Number.isFinite(maxX) || !Number.isFinite(maxY))
        return { x: 0, y: 0, width: 1, height: 1 };
    return { x: minX, y: minY, width: Math.max(1, maxX - minX), height: Math.max(1, maxY - minY) };
}
function clampRectToBounds(rect, bounds, minVisible) {
    const r = rect && typeof rect === 'object' ? rect : null;
    const b = bounds && typeof bounds === 'object' ? bounds : null;
    if (!r || !b)
        return null;
    const width = Math.max(1, Math.round(Number(r.width) || 1));
    const height = Math.max(1, Math.round(Number(r.height) || 1));
    const minRaw = Number(minVisible);
    const minVisibleX = Number.isFinite(minRaw) && minRaw > 0 && minRaw <= 1
        ? Math.max(1, Math.round(width * minRaw))
        : Math.max(1, Math.min(width, Math.round(Number(minVisible) || 1)));
    const minVisibleY = Number.isFinite(minRaw) && minRaw > 0 && minRaw <= 1
        ? Math.max(1, Math.round(height * minRaw))
        : Math.max(1, Math.min(height, Math.round(Number(minVisible) || 1)));
    const minX = Math.round(Number(b.x) || 0);
    const minY = Math.round(Number(b.y) || 0);
    const boundWidth = Math.max(1, Math.round(Number(b.width) || 1));
    const boundHeight = Math.max(1, Math.round(Number(b.height) || 1));
    const minXRaw = minX - (width - minVisibleX);
    const minYRaw = minY - (height - minVisibleY);
    const maxXRaw = minX + boundWidth - minVisibleX;
    const maxYRaw = minY + boundHeight - minVisibleY;
    const maxX = Math.max(minX, maxXRaw);
    const maxY = Math.max(minY, maxYRaw);
    const rx = Math.round(Number(r.x) || 0);
    const ry = Math.round(Number(r.y) || 0);
    const x = Math.min(Math.max(rx, minXRaw), maxX);
    const y = Math.min(Math.max(ry, minYRaw), maxY);
    return { x, y, width, height };
}
function getDisplayIdForRect(rect) {
    const r = rect && typeof rect === 'object' ? rect : null;
    const cx = (Number(r && r.x) || 0) + (Number(r && r.w) || 1) / 2;
    const cy = (Number(r && r.y) || 0) + (Number(r && r.h) || 1) / 2;
    try {
        const d = screen.getDisplayNearestPoint({ x: Math.round(cx), y: Math.round(cy) });
        return d && typeof d.id !== 'undefined' ? d.id : null;
    }
    catch {
        return null;
    }
}
function queueBroadcastPinnedShots() {
    if (pinnedBroadcastQueued)
        return;
    pinnedBroadcastQueued = true;
    setImmediate(() => {
        pinnedBroadcastQueued = false;
        broadcastPinnedShots();
    });
}
function broadcastPinnedShots() {
    for (const [displayId, bw] of wins.entries()) {
        if (!bw || bw.isDestroyed())
            continue;
        const d = screen.getAllDisplays().find(x => String(x && x.id) === String(displayId));
        const b = d && d.bounds ? d.bounds : null;
        if (!b)
            continue;
        const shots = [];
        for (const s of pinnedShotsById.values()) {
            const targetId = getDisplayIdForRect(s.rect);
            if (String(s.lastTargetId || '') !== String(targetId || '')) {
                s.sentTo = new Set();
                s.lastTargetId = targetId;
            }
            if (String(targetId) !== String(displayId))
                continue;
            const lr = { x: Math.round(s.rect.x - b.x), y: Math.round(s.rect.y - b.y), w: Math.round(s.rect.w), h: Math.round(s.rect.h) };
            const item = { id: s.id, rect: lr };
            if (!s.sentTo)
                s.sentTo = new Set();
            if (!s.sentTo.has(displayId)) {
                item.pngBase64 = s.pngBase64;
                s.sentTo.add(displayId);
            }
            shots.push(item);
        }
        safeSendTo(bw, 'pinned-shots-state', { shots });
        const shotRects = shots.map(x => x.rect);
        pinnedShotRectsByBwId.set(bw.id, shotRects);
        const images = [];
        for (const s of refImagesById.values()) {
            const targetId = getDisplayIdForRect(s.rect);
            if (String(s.lastTargetId || '') !== String(targetId || '')) {
                s.sentTo = new Set();
                s.lastTargetId = targetId;
            }
            if (String(targetId) !== String(displayId))
                continue;
            const lr = { x: Math.round(s.rect.x - b.x), y: Math.round(s.rect.y - b.y), w: Math.round(s.rect.w), h: Math.round(s.rect.h) };
            const z = refImageOrder.indexOf(s.id);
            const item = {
                id: s.id,
                rect: lr,
                z: z >= 0 ? z : 0,
                rotation: Number(s.rotation) || 0,
                flipH: !!s.flipH,
                flipV: !!s.flipV,
                grayscale: !!s.grayscale,
                invert: !!s.invert,
                locked: !!s.locked,
                showGrid: !!s.showGrid,
                brightness: Number(s.brightness) || 1,
                contrast: Number(s.contrast) || 1,
                opacity: Number(s.opacity) || 1
            };
            if (!s.sentTo)
                s.sentTo = new Set();
            if (!s.sentTo.has(displayId)) {
                item.dataUrl = s.dataUrl;
                s.sentTo.add(displayId);
            }
            images.push(item);
        }
        safeSendTo(bw, 'ref-images-state', { images });
        const refRects = images.map(x => x.rect);
        refImageRectsByBwId.set(bw.id, refRects);
        const merged = shotRects.length ? shotRects.slice() : [];
        if (refRects.length)
            merged.push(...refRects);
        pinnedRectsByBwId.set(bw.id, merged);
    }
    updateWindowVisibility();
}
function setScreenshotPickLock(bw, active, displayId) {
    if (!bw || bw.isDestroyed())
        return;
    if (active) {
        screenshotPickLock = { bwId: bw.id, displayId: String(displayId || '').trim() };
        try {
            bw.show();
        }
        catch (err) {
            console.error('[screenshot-pick] Failed to show window:', err);
        }
        try {
            bw.setIgnoreMouseEvents(false);
        }
        catch (err) {
            console.error('[screenshot-pick] Failed to set ignore mouse events:', err);
        }
    }
    else {
        if (screenshotPickLock && screenshotPickLock.bwId === bw.id)
            screenshotPickLock = null;
        try {
            bw.setIgnoreMouseEvents(true, { forward: true });
        }
        catch (err) {
            console.error('[screenshot-pick] Failed to restore ignore mouse events:', err);
        }
    }
}
function setRecordPickLock(bw, active, displayId, capture) {
    if (!bw || bw.isDestroyed())
        return;
    if (active) {
        recordPickLock = { bwId: bw.id, displayId: String(displayId || '').trim(), capture: !!capture };
        try {
            bw.show();
        }
        catch (err) {
            console.error('[record-pick] Failed to show window:', err);
        }
        try {
            if (capture)
                bw.setIgnoreMouseEvents(false);
            else
                bw.setIgnoreMouseEvents(true, { forward: true });
        }
        catch (err) {
            console.error('[record-pick] Failed to set ignore mouse events:', err);
        }
    }
    else {
        if (recordPickLock && recordPickLock.bwId === bw.id)
            recordPickLock = null;
        try {
            bw.setIgnoreMouseEvents(true, { forward: true });
        }
        catch (err) {
            console.error('[record-pick] Failed to restore ignore mouse events:', err);
        }
    }
}
function resolveAppIconPath() {
    const candidates = ['icon.ico', 'icon.png', 'app.ico', 'app.png', 'tt.ico', 'tt.png', 'active.ico', 'active.png'];
    for (const name of candidates) {
        const p = path.join(assetsIconDir, name);
        if (fs.existsSync(p))
            return p;
    }
    return null;
}
let settings = {
    fadeTime: 1000,
    cssVars: {},
    uiTheme: 'light',
    overlayVisible: true,
    mouseButtonVisible: true,
    toolbarVisible: true,
    toolbarPinned: false,
    mouseDotMoveVisible: true,
    mouseDotClickVisible: true,
    mouseDotStyle: 'trail',
    mouseDotProfiles: null,
    autoLaunch: true,
    recordSystemAudio: true,
    recordMicAudio: true,
    hotkeys: {
        screenshotSave: 'Ctrl+Alt+S',
        screenshotClipboard: 'Alt+S',
        recordToggle: 'Ctrl+Alt+R',
        recordSaveGif: 'Ctrl+Alt+G',
        recordSaveMp4: 'Ctrl+Alt+M',
        undo: 'Ctrl+Z',
    },
};
let settingsWritePending = null;
let settingsWriteScheduled = false;
let settingsWriteChain = Promise.resolve();
let settingsWriteResolve = null;
function scheduleSettingsWrite() {
    const p = settingsFilePath();
    settingsWritePending = { p, json: JSON.stringify(settings, null, 2) };
    if (settingsWriteScheduled)
        return;
    settingsWriteScheduled = true;
    settingsWriteChain = settingsWriteChain.then(async () => {
        while (settingsWritePending) {
            const job = settingsWritePending;
            settingsWritePending = null;
            try {
                await fs.promises.mkdir(path.dirname(job.p), { recursive: true });
            }
            catch (err) {
                console.error('[settings] Failed to create directory:', err);
            }
            try {
                await fs.promises.writeFile(job.p, job.json, 'utf8');
            }
            catch (err) {
                console.error('[settings] Failed to write settings file:', err);
            }
        }
        settingsWriteScheduled = false;
        if (settingsWriteResolve) {
            settingsWriteResolve();
            settingsWriteResolve = null;
        }
    });
}
async function flushSettingsWrite() {
    if (!settingsWriteScheduled && !settingsWritePending)
        return;
    await new Promise((resolve) => {
        if (!settingsWriteScheduled && !settingsWritePending) {
            resolve(undefined);
            return;
        }
        settingsWriteResolve = resolve;
    });
}
function hardenWebContents(wc, allowedHtmlBasename) {
    if (!wc || wc.isDestroyed())
        return;
    try {
        wc.setWindowOpenHandler(() => ({ action: 'deny' }));
    }
    catch { }
    const allow = (url) => {
        try {
            const u = new URL(String(url || ''));
            if (u.protocol !== 'file:')
                return false;
            const pathname = String(u.pathname || '').toLowerCase();
            const target = `/${String(allowedHtmlBasename || '').toLowerCase()}`;
            return pathname.endsWith(target);
        }
        catch {
            return false;
        }
    };
    wc.on('will-navigate', (e, url) => { if (!allow(url))
        e.preventDefault(); });
    wc.on('will-redirect', (e, url) => { if (!allow(url))
        e.preventDefault(); });
}
function isMouseDotActive() {
    return !!(mouseDotMoveVisible || mouseDotClickVisible);
}
function quoteArg(s) {
    return `"${String(s).replace(/"/g, '\\"')}"`;
}
const WINDOWS_RUN_KEY = 'HKCU\\Software\\Microsoft\\Windows\\CurrentVersion\\Run';
const WINDOWS_AUTO_LAUNCH_VALUE = 'TT';
function buildWindowsAutoLaunchCommand() {
    const exe = process.execPath;
    if (!exe)
        return null;
    if (app.isPackaged)
        return `${quoteArg(exe)} --open-as-hidden`;
    let appPath = null;
    try {
        appPath = app.getAppPath();
    }
    catch { }
    if (!appPath)
        return `${quoteArg(exe)} --open-as-hidden`;
    return `${quoteArg(exe)} ${quoteArg(appPath)} --open-as-hidden`;
}
function listWindowsRunValues() {
    try {
        const cp = spawnSync('reg.exe', ['query', WINDOWS_RUN_KEY], { windowsHide: true, encoding: 'utf8' });
        if (!cp || cp.status !== 0)
            return [];
        const out = String(cp.stdout || '');
        const lines = out.split(/\\r?\\n/);
        const res = [];
        for (const line of lines) {
            const m = line.match(/^\\s*(.+?)\\s+(REG_[A-Z0-9_]+)\\s+(.*)$/i);
            if (!m)
                continue;
            res.push({ name: String(m[1] || '').trim(), type: String(m[2] || '').trim(), data: String(m[3] || '').trim() });
        }
        return res;
    }
    catch {
        return [];
    }
}
function deleteWindowsRunValue(valueName) {
    try {
        const args = ['delete', WINDOWS_RUN_KEY, '/v', String(valueName), '/f'];
        const cp = spawn('reg.exe', args, { windowsHide: true, detached: true, stdio: 'ignore' });
        cp.unref();
    }
    catch { }
}
function setWindowsRunValue(valueName, data) {
    try {
        const args = ['add', WINDOWS_RUN_KEY, '/v', String(valueName), '/t', 'REG_SZ', '/d', String(data || ''), '/f'];
        const cp = spawn('reg.exe', args, { windowsHide: true, detached: true, stdio: 'ignore' });
        cp.unref();
    }
    catch { }
}
function isProjectElectronAutoLaunchValue(data) {
    try {
        const s = String(data || '').toLowerCase();
        if (!s.includes('\\\\node_modules\\\\electron\\\\dist\\\\electron.exe'))
            return false;
        let appPath = null;
        try {
            appPath = app.getAppPath();
        }
        catch { }
        if (!appPath)
            return true;
        return s.includes(String(appPath).toLowerCase());
    }
    catch {
        return false;
    }
}
function cleanupLegacyWindowsAutoLaunch() {
    if (process.platform !== 'win32')
        return;
    const values = listWindowsRunValues();
    for (const v of values) {
        if (!v || !v.name)
            continue;
        const name = String(v.name);
        const data = String(v.data || '');
        if (/^electron\\.app\\./i.test(name) && isProjectElectronAutoLaunchValue(data))
            deleteWindowsRunValue(name);
        if (isProjectElectronAutoLaunchValue(data))
            deleteWindowsRunValue(name);
    }
}
let winAutoLaunchNeedsRepair = false;
function getAutoLaunchFromOS() {
    if (process.platform === 'win32') {
        try {
            const expected = buildWindowsAutoLaunchCommand();
            const values = listWindowsRunValues();
            const v = values.find(x => x && String(x.name).toLowerCase() === WINDOWS_AUTO_LAUNCH_VALUE.toLowerCase());
            if (!v) {
                winAutoLaunchNeedsRepair = false;
                return false;
            }
            const actual = String(v.data || '').trim();
            if (!expected) {
                winAutoLaunchNeedsRepair = false;
                return true;
            }
            winAutoLaunchNeedsRepair = actual !== expected;
            return true;
        }
        catch {
            return false;
        }
    }
    try {
        return !!app.getLoginItemSettings().openAtLogin;
    }
    catch {
        return false;
    }
}
function setAutoLaunchOnOS(enabled) {
    const on = !!enabled;
    if (process.platform === 'win32') {
        cleanupLegacyWindowsAutoLaunch();
        const cmd = buildWindowsAutoLaunchCommand();
        if (on)
            setWindowsRunValue(WINDOWS_AUTO_LAUNCH_VALUE, cmd);
        else {
            deleteWindowsRunValue(WINDOWS_AUTO_LAUNCH_VALUE);
            deleteWindowsRunValue('tt');
        }
        return;
    }
    try {
        app.setLoginItemSettings({ openAtLogin: on, openAsHidden: true });
    }
    catch { }
}
let autoLaunchApplied = null;
let autoLaunchPending = null;
let autoLaunchTimer = null;
function requestAutoLaunchOnOS(enabled) {
    autoLaunchPending = !!enabled;
    if (autoLaunchTimer) {
        try {
            clearTimeout(autoLaunchTimer);
        }
        catch { }
        autoLaunchTimer = null;
    }
    autoLaunchTimer = setTimeout(() => {
        autoLaunchTimer = null;
        const target = !!autoLaunchPending;
        autoLaunchPending = null;
        if (autoLaunchApplied === null)
            autoLaunchApplied = getAutoLaunchFromOS();
        if (target === autoLaunchApplied && !(target && winAutoLaunchNeedsRepair))
            return;
        setAutoLaunchOnOS(target);
        autoLaunchApplied = target;
        winAutoLaunchNeedsRepair = false;
        refreshTray();
    }, 260);
}
function normalizeSettings(v) {
    if (!v || typeof v !== 'object')
        v = {};
    const { mouseDotVisible: _legacyMouseDotVisible, hotkeys: rawHotkeys } = v;
    const normalizeCssVars = (raw) => {
        if (!raw || typeof raw !== 'object' || Array.isArray(raw))
            return {};
        const entries = Object.entries(raw);
        const out = {};
        let kept = 0;
        for (const [k0, v0] of entries) {
            if (kept >= 240)
                break;
            const k = String(k0 || '').trim();
            if (!/^--[a-z0-9-]{1,64}$/i.test(k))
                continue;
            if (v0 === null) {
                out[k] = null;
                kept += 1;
                continue;
            }
            if (typeof v0 !== 'string')
                continue;
            const val = String(v0).trim();
            if (!val || val.length > 128)
                continue;
            out[k] = val;
            kept += 1;
        }
        return out;
    };
    const cssVars = normalizeCssVars(v.cssVars);
    const uiTheme = String(v.uiTheme || '').trim().toLowerCase() === 'dark' ? 'dark' : 'light';
    const normalizeAccelerator = (raw, fallback, allowEmpty) => {
        const s = String(raw || '').trim();
        if (!s)
            return allowEmpty ? '' : fallback;
        const cleaned = s.replace(/\\s+/g, '');
        const parts = cleaned.split('+').map(x => String(x || '').trim()).filter(Boolean);
        if (!parts.length)
            return allowEmpty ? '' : fallback;
        const normalized = parts.map((p) => {
            const low = p.toLowerCase();
            if (low === 'control')
                return 'Ctrl';
            if (low === 'cmdorctrl')
                return 'CommandOrControl';
            if (low === 'commandorcontrol')
                return 'CommandOrControl';
            if (low === 'cmd')
                return 'Command';
            if (low === 'ctrl')
                return 'Ctrl';
            if (low === 'alt' || low === 'option')
                return 'Alt';
            if (low === 'shift')
                return 'Shift';
            if (low === 'win' || low === 'meta' || low === 'super')
                return 'Super';
            if (p.length === 1)
                return p.toUpperCase();
            return p[0].toUpperCase() + p.slice(1);
        });
        return normalized.join('+');
    };
    const hotkeysObj = rawHotkeys && typeof rawHotkeys === 'object' ? rawHotkeys : {};
    const hasOwn = (obj, k) => obj && Object.prototype.hasOwnProperty.call(obj, k);
    const screenshotSave = hasOwn(hotkeysObj, 'screenshotSave')
        ? normalizeAccelerator(hotkeysObj.screenshotSave, 'Ctrl+Alt+S', true)
        : normalizeAccelerator(undefined, 'Ctrl+Alt+S', false);
    const screenshotClipboard = hasOwn(hotkeysObj, 'screenshotClipboard')
        ? normalizeAccelerator(hotkeysObj.screenshotClipboard, 'Alt+S', true)
        : normalizeAccelerator(undefined, 'Alt+S', false);
    const recordToggle = hasOwn(hotkeysObj, 'recordToggle')
        ? normalizeAccelerator(hotkeysObj.recordToggle, 'Ctrl+Alt+R', true)
        : normalizeAccelerator(undefined, 'Ctrl+Alt+R', false);
    const recordSaveGif = hasOwn(hotkeysObj, 'recordSaveGif')
        ? normalizeAccelerator(hotkeysObj.recordSaveGif, 'Ctrl+Alt+G', true)
        : normalizeAccelerator(undefined, 'Ctrl+Alt+G', false);
    const recordSaveMp4 = hasOwn(hotkeysObj, 'recordSaveMp4')
        ? normalizeAccelerator(hotkeysObj.recordSaveMp4, 'Ctrl+Alt+M', true)
        : normalizeAccelerator(undefined, 'Ctrl+Alt+M', false);
    const undo = hasOwn(hotkeysObj, 'undo')
        ? normalizeAccelerator(hotkeysObj.undo, 'Ctrl+Z', true)
        : normalizeAccelerator(undefined, 'Ctrl+Z', false);
    const toggleOverlay = hasOwn(hotkeysObj, 'toggleOverlay')
        ? normalizeAccelerator(hotkeysObj.toggleOverlay, '', true)
        : '';
    const toggleMouseButtons = hasOwn(hotkeysObj, 'toggleMouseButtons')
        ? normalizeAccelerator(hotkeysObj.toggleMouseButtons, '', true)
        : '';
    const toggleMouseDot = hasOwn(hotkeysObj, 'toggleMouseDot')
        ? normalizeAccelerator(hotkeysObj.toggleMouseDot, '', true)
        : '';
    const openSettings = hasOwn(hotkeysObj, 'openSettings')
        ? normalizeAccelerator(hotkeysObj.openSettings, '', true)
        : '';
    const toolText = hasOwn(hotkeysObj, 'toolText')
        ? normalizeAccelerator(hotkeysObj.toolText, '', true)
        : '';
    const toolMosaic = hasOwn(hotkeysObj, 'toolMosaic')
        ? normalizeAccelerator(hotkeysObj.toolMosaic, '', true)
        : '';
    const toolRefimg = hasOwn(hotkeysObj, 'toolRefimg')
        ? normalizeAccelerator(hotkeysObj.toolRefimg, '', true)
        : '';
    const toolRecord = hasOwn(hotkeysObj, 'toolRecord')
        ? normalizeAccelerator(hotkeysObj.toolRecord, '', true)
        : '';
    const toolPen = hasOwn(hotkeysObj, 'toolPen')
        ? normalizeAccelerator(hotkeysObj.toolPen, '', true)
        : '';
    const toolEraser = hasOwn(hotkeysObj, 'toolEraser')
        ? normalizeAccelerator(hotkeysObj.toolEraser, '', true)
        : '';
    const toolArrow = hasOwn(hotkeysObj, 'toolArrow')
        ? normalizeAccelerator(hotkeysObj.toolArrow, '', true)
        : '';
    const toolRect = hasOwn(hotkeysObj, 'toolRect')
        ? normalizeAccelerator(hotkeysObj.toolRect, '', true)
        : '';
    const toolCircle = hasOwn(hotkeysObj, 'toolCircle')
        ? normalizeAccelerator(hotkeysObj.toolCircle, '', true)
        : '';
    const clear = hasOwn(hotkeysObj, 'clear')
        ? normalizeAccelerator(hotkeysObj.clear, '', true)
        : '';
    const fadeTime = typeof v.fadeTime === 'number' && Number.isFinite(v.fadeTime) ? Math.max(0, Math.round(v.fadeTime)) : 1000;
    const overlayVisible = typeof v.overlayVisible === 'boolean' ? v.overlayVisible : true;
    const mouseButtonVisible = typeof v.mouseButtonVisible === 'boolean' ? v.mouseButtonVisible : true;
    const toolbarVisible = typeof v.toolbarVisible === 'boolean' ? v.toolbarVisible : true;
    const toolbarPinned = typeof v.toolbarPinned === 'boolean' ? v.toolbarPinned : false;
    const legacyDot = typeof v.mouseDotVisible === 'boolean' ? v.mouseDotVisible : undefined;
    const mouseDotMoveVisible = typeof v.mouseDotMoveVisible === 'boolean' ? v.mouseDotMoveVisible : (legacyDot ?? true);
    const mouseDotClickVisible = typeof v.mouseDotClickVisible === 'boolean' ? v.mouseDotClickVisible : (legacyDot ?? true);
    const styleRaw = String(v.mouseDotStyle || '').trim().toLowerCase();
    const mouseDotStyle = (styleRaw === 'fixed' || styleRaw === 'nebula' || styleRaw === 'firecracker') ? styleRaw : 'trail';
    const mouseDotProfiles = v.mouseDotProfiles && typeof v.mouseDotProfiles === 'object' ? v.mouseDotProfiles : null;
    const autoLaunch = typeof v.autoLaunch === 'boolean' ? v.autoLaunch : getAutoLaunchFromOS();
    const recordSystemAudio = typeof v.recordSystemAudio === 'boolean' ? v.recordSystemAudio : true;
    const recordMicAudio = typeof v.recordMicAudio === 'boolean' ? v.recordMicAudio : true;
    const micIdRaw = typeof v.recordMicDeviceId === 'string' ? v.recordMicDeviceId : '';
    const recordMicDeviceId = String(micIdRaw || '').trim() || null;
    return {
        fadeTime,
        cssVars,
        uiTheme,
        overlayVisible,
        mouseButtonVisible,
        toolbarVisible,
        toolbarPinned,
        mouseDotMoveVisible,
        mouseDotClickVisible,
        mouseDotStyle,
        mouseDotProfiles,
        autoLaunch,
        recordSystemAudio,
        recordMicAudio,
        recordMicDeviceId,
        hotkeys: {
            screenshotSave,
            screenshotClipboard,
            recordToggle,
            recordSaveGif,
            recordSaveMp4,
            undo,
            clear,
            toggleOverlay,
            toggleMouseButtons,
            toggleMouseDot,
            openSettings,
            toolPen,
            toolEraser,
            toolArrow,
            toolRect,
            toolCircle,
            toolText,
            toolMosaic,
            toolRefimg,
            toolRecord,
        },
    };
}
function settingsFilePath() {
    try {
        return path.join(app.getPath('userData'), 'settings.json');
    }
    catch {
        return path.join(__dirname, 'settings.json');
    }
}
async function loadSettings() {
    const p = settingsFilePath();
    try {
        const raw = await fs.promises.readFile(p, 'utf8');
        const obj = JSON.parse(raw);
        settings = normalizeSettings(obj);
    }
    catch {
        settings = normalizeSettings(settings);
    }
    overlayVisible = !!settings.overlayVisible;
    mouseButtonVisible = !!settings.mouseButtonVisible;
    mouseDotMoveVisible = !!settings.mouseDotMoveVisible;
    mouseDotClickVisible = !!settings.mouseDotClickVisible;
    if (autoLaunchApplied === null)
        autoLaunchApplied = getAutoLaunchFromOS();
    if (!!settings.autoLaunch !== autoLaunchApplied || (settings.autoLaunch && winAutoLaunchNeedsRepair)) {
        requestAutoLaunchOnOS(!!settings.autoLaunch);
    }
    return settings;
}
function saveSettings(next) {
    settings = normalizeSettings(next);
    scheduleSettingsWrite();
    return settings;
}
function resetSettings() {
    settings = saveSettings({
        fadeTime: 1000,
        cssVars: {},
        uiTheme: 'light',
        overlayVisible: true,
        mouseButtonVisible: true,
        toolbarVisible: true,
        toolbarPinned: false,
        mouseDotMoveVisible: true,
        mouseDotClickVisible: true,
        mouseDotStyle: 'trail',
        mouseDotProfiles: null,
        autoLaunch: true,
        recordSystemAudio: true,
        recordMicAudio: true,
        recordMicDeviceId: null,
        hotkeys: {
            screenshotSave: 'Ctrl+Alt+S',
            screenshotClipboard: 'Alt+S',
            recordToggle: 'Ctrl+Alt+R',
            recordSaveGif: 'Ctrl+Alt+G',
            recordSaveMp4: 'Ctrl+Alt+M',
            undo: 'Ctrl+Z',
        },
    });
    overlayVisible = !!settings.overlayVisible;
    mouseButtonVisible = !!settings.mouseButtonVisible;
    mouseDotMoveVisible = !!settings.mouseDotMoveVisible;
    mouseDotClickVisible = !!settings.mouseDotClickVisible;
    return settings;
}
function broadcastSettings() {
    safeBroadcast('settings-updated', settings);
    if (settingsWin && !settingsWin.isDestroyed()) {
        try {
            settingsWin.webContents.send('settings-updated', settings);
        }
        catch { }
    }
}
function normalizeOverlayBounds(display) {
    const b = display && display.bounds ? display.bounds : { x: 0, y: 0, width: 0, height: 0 };
    const sf = Number(display && display.scaleFactor) || 1;
    const pad = Math.max(1, Math.ceil(sf));
    return {
        x: Math.floor(b.x),
        y: Math.floor(b.y),
        width: Math.ceil(b.width) + pad,
        height: Math.ceil(b.height) + pad,
    };
}
function destroyWindows() {
    try {
        readyWins.clear();
    }
    catch { }
    for (const bw of wins.values()) {
        try {
            if (bw && !bw.isDestroyed())
                bw.destroy();
        }
        catch { }
    }
    wins.clear();
}
function resolveTrayImage(active = true) {
    const candidates = active
        ? ['active.png', 'active.ico', 'tray-on.png', 'tray-on.ico', 'tray.png', 'tray.ico']
        : ['inactive.png', 'inactive.ico', 'tray-off.png', 'tray-off.ico', 'tray.png', 'tray.ico'];
    for (const name of candidates) {
        const p = path.join(assetsIconDir, name);
        if (fs.existsSync(p)) {
            const img = nativeImage.createFromPath(p);
            if (img && !img.isEmpty())
                return img;
        }
    }
    const appIconPath = resolveAppIconPath();
    if (appIconPath) {
        const img = nativeImage.createFromPath(appIconPath);
        if (img && !img.isEmpty())
            return img;
    }
    return null;
}
function buildTrayImage(active = true) {
    const fileImg = resolveTrayImage(active);
    if (fileImg)
        return fileImg;
    const base64 = 'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAAMUlEQVR4AWOgGAX4z0QMDAzgH4YgQFQCYZgGkA1gkFQJYFZgM9gQKQZQGwA0AAAGNdG0G8KpLFAAAAABJRU5ErkJggg==';
    return nativeImage.createFromDataURL(`data:image/png;base64,${base64}`);
}
function buildContextMenu() {
    const autoLaunch = !!settings.autoLaunch;
    return Menu.buildFromTemplate([
        { label: '键盘按键', type: 'checkbox', checked: overlayVisible, click: () => toggleOverlayVisibility() },
        { label: '鼠标按键', type: 'checkbox', checked: mouseButtonVisible, click: () => toggleMouseButtonVisibility() },
        { label: '鼠标指示点', type: 'checkbox', checked: isMouseDotActive(), click: () => toggleMouseDotVisibility() },
        { type: 'separator' },
        { label: '设置', click: () => openSettingsWindow() },
        { type: 'separator' },
        { label: '开机启动', type: 'checkbox', checked: autoLaunch, click: (mi) => { settings = saveSettings({ ...settings, autoLaunch: !!mi.checked }); applyRuntimeFromSettings(settings); broadcastSettings(); refreshTray(); } },
        { type: 'separator' },
        { label: '退出', click: () => { try {
                uIOhook.stop();
            }
            catch { } app.quit(); } },
    ]);
}
function refreshTray() {
    if (!tray)
        return;
    const activeState = !!(overlayVisible || mouseButtonVisible);
    tray.setImage(buildTrayImage(activeState));
    tray.setContextMenu(buildContextMenu());
    tray.setToolTip('TT - 键鼠可视化');
}
function createTray() { try {
    tray = new Tray(buildTrayImage(true));
    tray.on('click', () => toggleActiveState());
    refreshTray();
}
catch (err) {
    console.error('[main] Failed to create tray:', err);
} }
function persistRuntimeVisibility() {
    settings = saveSettings({ ...settings, overlayVisible, mouseButtonVisible, mouseDotMoveVisible, mouseDotClickVisible });
    broadcastSettings();
}
function toggleOverlayVisibility() { overlayVisible = !overlayVisible; updateWindowVisibility(); persistRuntimeVisibility(); refreshTray(); }
function toggleMouseDotVisibility() {
    const next = !isMouseDotActive();
    mouseDotMoveVisible = next;
    mouseDotClickVisible = next;
    safeBroadcast('mouse-visibility', isMouseDotActive());
    updateWindowVisibility();
    persistRuntimeVisibility();
    refreshTray();
}
function toggleMouseButtonVisibility() { mouseButtonVisible = !mouseButtonVisible; updateWindowVisibility(); persistRuntimeVisibility(); refreshTray(); }
function toggleActiveState() {
    const next = !(overlayVisible || mouseButtonVisible);
    overlayVisible = next;
    mouseButtonVisible = next;
    updateWindowVisibility();
    persistRuntimeVisibility();
    refreshTray();
}
function updateWindowVisibility() {
    for (const bw of wins.values()) {
        const forcedVisible = !!(screenshotPickLock && screenshotPickLock.bwId === bw.id) || !!(recordPickLock && recordPickLock.bwId === bw.id);
        const pinnedRects = pinnedRectsByBwId.get(bw.id);
        const pinnedForced = !!(pinnedRects && Array.isArray(pinnedRects) && pinnedRects.length);
        const interactiveRects = interactiveRectsByBwId.get(bw.id);
        const interactiveForced = !!(interactiveRects && Array.isArray(interactiveRects) && interactiveRects.length);
        if (overlayVisible || mouseButtonVisible || isMouseDotActive() || forcedVisible || pinnedForced || interactiveForced) {
            bw.show();
            try {
                if (suppressOverlayTopmost)
                    bw.setAlwaysOnTop(false);
                else
                    bw.setAlwaysOnTop(true, 'screen-saver', 1);
            }
            catch { }
            if (screenshotPickLock && screenshotPickLock.bwId === bw.id) {
                try {
                    bw.setIgnoreMouseEvents(false);
                }
                catch { }
            }
            else if (recordPickLock && recordPickLock.bwId === bw.id) {
                if (rendererCaptureByBwId.get(bw.id)) {
                    try {
                        bw.setIgnoreMouseEvents(false);
                    }
                    catch { }
                }
                else if (recordPickLock.capture) {
                    try {
                        bw.setIgnoreMouseEvents(false);
                    }
                    catch { }
                }
                else {
                    if (suppressOverlayTopmost) {
                        try {
                            bw.setIgnoreMouseEvents(true, { forward: true });
                        }
                        catch { }
                    }
                    else {
                        const pinnedHover = pinnedForced ? computePinnedHover(bw, pinnedRects) : false;
                        const interactiveHover = interactiveForced ? computeInteractiveHover(bw, interactiveRects) : false;
                        try {
                            if (pinnedHover || interactiveHover)
                                bw.setIgnoreMouseEvents(false);
                            else
                                bw.setIgnoreMouseEvents(true, { forward: true });
                        }
                        catch { }
                    }
                }
            }
            else if (rendererCaptureByBwId.get(bw.id)) {
                try {
                    bw.setIgnoreMouseEvents(false);
                }
                catch { }
            }
            else {
                if (suppressOverlayTopmost) {
                    try {
                        bw.setIgnoreMouseEvents(true, { forward: true });
                    }
                    catch { }
                }
                else {
                    const pinnedHover = pinnedForced ? computePinnedHover(bw, pinnedRects) : false;
                    const interactiveHover = interactiveForced ? computeInteractiveHover(bw, interactiveRects) : false;
                    if (pinnedHover || interactiveHover) {
                        try {
                            bw.setIgnoreMouseEvents(false);
                        }
                        catch { }
                    }
                    else {
                        bw.setIgnoreMouseEvents(true, { forward: true });
                    }
                }
            }
        }
        else {
            bw.hide();
        }
    }
}
function createOverlayWindowForDisplay(display) {
    const bounds = normalizeOverlayBounds(display);
    const bw = new BrowserWindow({
        x: bounds.x, y: bounds.y, width: bounds.width, height: bounds.height,
        frame: false, transparent: true, alwaysOnTop: true, resizable: false,
        skipTaskbar: true, focusable: false,
        webPreferences: { preload: path.join(__dirname, 'preload.js'), nodeIntegration: false, contextIsolation: true, sandbox: false, navigateOnDragDrop: false }
    });
    try {
        hardenWebContents(bw.webContents, 'index.html');
    }
    catch { }
    try {
        if (suppressOverlayTopmost)
            bw.setAlwaysOnTop(false);
        else
            bw.setAlwaysOnTop(true, 'screen-saver', 1);
    }
    catch { }
    try {
        bw.setVisibleOnAllWorkspaces(true, { visibleOnFullScreen: true });
    }
    catch { }
    bw.on('blur', () => {
        if (suppressOverlayTopmost)
            return;
        try {
            bw.setAlwaysOnTop(true, 'screen-saver', 1);
        }
        catch { }
    });
    bw.setIgnoreMouseEvents(true, { forward: true });
    try {
        bw.loadFile('index.html', { query: { displayId: String(display && display.id) } });
    }
    catch {
        bw.loadFile('index.html');
    }
    bw.webContents.once('did-finish-load', () => {
        readyWins.add(bw.id);
        safeSendTo(bw, 'mouse-visibility', isMouseDotActive());
        safeSendTo(bw, 'settings-updated', settings);
        try {
            bw.setBounds(bounds, false);
        }
        catch { }
        queueBroadcastPinnedShots();
    });
    bw.on('closed', () => { readyWins.delete(bw.id); });
    return bw;
}
function openSettingsWindow() {
    if (settingsWin && !settingsWin.isDestroyed()) {
        try {
            settingsWin.show();
            settingsWin.focus();
        }
        catch (err) {
            console.error('[main] Failed to show/focus settings window:', err);
        }
        return;
    }
    const appIconPath = resolveAppIconPath();
    settingsWin = new BrowserWindow({
        width: 800,
        height: 600,
        show: true,
        frame: false,
        backgroundColor: '#0f1115',
        autoHideMenuBar: true,
        ...(appIconPath ? { icon: appIconPath } : {}),
        webPreferences: { preload: path.join(__dirname, 'preload.js'), nodeIntegration: false, contextIsolation: true, sandbox: false, navigateOnDragDrop: false },
    });
    try {
        hardenWebContents(settingsWin.webContents, 'settings.html');
    }
    catch { }
    try {
        settingsWin.setAlwaysOnTop(false);
    }
    catch { }
    try {
        settingsWin.setMenu(null);
    }
    catch { }
    try {
        settingsWin.loadFile('settings.html');
    }
    catch { }
    settingsWin.on('closed', () => {
        settingsWin = null;
        if (toolHoldCount === 0)
            setOverlaysTopmostEnabled(true);
    });
}
function getCaptureThumbnailSize(displayId) {
    let d = null;
    try {
        const idStr = String(displayId || '').trim();
        d = screen.getAllDisplays().find(x => String(x && x.id) === idStr) || screen.getPrimaryDisplay();
    }
    catch { }
    const sf = Math.max(1, Number(d && d.scaleFactor) || 1);
    const b = d && d.bounds ? d.bounds : { width: 1920, height: 1080 };
    let w = Math.max(1, Math.round((Number(b.width) || 1920) * sf));
    let h = Math.max(1, Math.round((Number(b.height) || 1080) * sf));
    const maxDim = 4096;
    const scale = Math.min(1, maxDim / Math.max(w, h));
    if (scale < 1) {
        w = Math.max(1, Math.round(w * scale));
        h = Math.max(1, Math.round(h * scale));
    }
    return { width: w, height: h };
}
async function captureAllDisplays() {
    if (!desktopCapturer || typeof desktopCapturer.getSources !== 'function')
        return { displays: [], error: 'desktopCapturer_unavailable' };
    const displays = screen.getAllDisplays();
    const result = [];
    const maxDim = 4096;
    for (const display of displays) {
        const sf = Math.max(1, Number(display.scaleFactor) || 1);
        const b = display.bounds ? display.bounds : { x: 0, y: 0, width: 1920, height: 1080 };
        let w = Math.max(1, Math.round((Number(b.width) || 1920) * sf));
        let h = Math.max(1, Math.round((Number(b.height) || 1080) * sf));
        const scale = Math.min(1, maxDim / Math.max(w, h));
        if (scale < 1) {
            w = Math.max(1, Math.round(w * scale));
            h = Math.max(1, Math.round(h * scale));
        }
        try {
            const sources = await desktopCapturer.getSources({ types: ['screen'], thumbnailSize: { width: w, height: h } });
            const src = sources.find(s => String(s.display_id || '') === String(display.id)) || sources[0];
            const thumb = src && src.thumbnail;
            const png = thumb && !thumb.isEmpty() ? thumb.toPNG() : null;
            const pngBase64 = png ? Buffer.from(png).toString('base64') : '';
            result.push({
                displayId: String(display.id),
                bounds: { x: Math.round(b.x), y: Math.round(b.y), width: Math.round(b.width), height: Math.round(b.height) },
                pngBase64,
                error: pngBase64 ? '' : 'empty_png'
            });
        }
        catch {
            result.push({
                displayId: String(display.id),
                bounds: { x: Math.round(b.x), y: Math.round(b.y), width: Math.round(b.width), height: Math.round(b.height) },
                pngBase64: '',
                error: 'capture_failed'
            });
        }
    }
    return { displays: result, error: '' };
}
async function captureScreenInMain(displayId) {
    if (!desktopCapturer || typeof desktopCapturer.getSources !== 'function')
        return { pngBase64: '', error: 'desktopCapturer_unavailable' };
    const idStr = String(displayId || '').trim();
    const sizes = [
        getCaptureThumbnailSize(displayId),
        { width: 1920, height: 1080 },
        { width: 1280, height: 720 },
    ];
    const seen = new Set();
    let last = { pngBase64: '', error: 'empty_png' };
    for (const sz of sizes) {
        const tw = Math.max(1, Number(sz && sz.width) || 1);
        const th = Math.max(1, Number(sz && sz.height) || 1);
        const key = `${tw}x${th}`;
        if (seen.has(key))
            continue;
        seen.add(key);
        try {
            const sources = await desktopCapturer.getSources({ types: ['screen'], thumbnailSize: { width: tw, height: th } });
            if (!sources || !sources.length)
                return { pngBase64: '', error: 'no_sources' };
            const src = sources.find(s => String(s.display_id || '') === idStr) || sources[0];
            const thumb = src && src.thumbnail;
            const png = thumb && !thumb.isEmpty() ? thumb.toPNG() : null;
            const pngBase64 = png ? Buffer.from(png).toString('base64') : '';
            const res = { pngBase64, error: pngBase64 ? '' : 'empty_png' };
            if (res.pngBase64)
                return res;
            last = res;
        }
        catch {
            return { pngBase64: '', error: 'getSources_failed' };
        }
        try {
            await new Promise((r) => setTimeout(r, 40));
        }
        catch { }
    }
    return last;
}
function createWindows() {
    const displays = screen.getAllDisplays();
    destroyWindows();
    for (const d of displays) {
        wins.set(d.id, createOverlayWindowForDisplay(d));
    }
}
function safeSendTo(bw, channel, ...args) {
    if (!bw || bw.isDestroyed())
        return;
    const wc = bw.webContents;
    if (!wc || wc.isDestroyed() || wc.isLoadingMainFrame())
        return;
    if (!readyWins.has(bw.id))
        return;
    try {
        if (bw.isDestroyed() || wc.isDestroyed())
            return;
        wc.send(channel, ...args);
    }
    catch (err) {
        console.error(`[safeSendTo] Failed to send to window ${bw.id}:`, err);
    }
}
function safeBroadcast(channel, ...args) { for (const bw of wins.values())
    safeSendTo(bw, channel, ...args); }
function applyRuntimeFromSettings(s) {
    const next = s && typeof s === 'object' ? s : null;
    if (!next)
        return;
    overlayVisible = !!next.overlayVisible;
    mouseButtonVisible = !!next.mouseButtonVisible;
    mouseDotMoveVisible = !!next.mouseDotMoveVisible;
    mouseDotClickVisible = !!next.mouseDotClickVisible;
    if (autoLaunchApplied === null)
        autoLaunchApplied = getAutoLaunchFromOS();
    if (!!next.autoLaunch !== autoLaunchApplied)
        requestAutoLaunchOnOS(!!next.autoLaunch);
    try {
        applyGlobalShortcutsFromSettings(next);
    }
    catch { }
    safeBroadcast('mouse-visibility', isMouseDotActive());
    updateWindowVisibility();
    refreshTray();
}
function sendMouseToDisplay(channel, payload) {
    const pt = screen.getCursorScreenPoint();
    const d = screen.getAllDisplays().find(di => pt.x >= di.bounds.x && pt.x < di.bounds.x + di.bounds.width && pt.y >= di.bounds.y && pt.y < di.bounds.y + di.bounds.height);
    const bw = d ? wins.get(d.id) : null;
    if (!bw || bw.isDestroyed())
        return;
    const localPos = { x: pt.x - d.bounds.x, y: pt.y - d.bounds.y };
    if (channel === 'mouse-move') {
        safeSendTo(bw, channel, localPos);
        return;
    }
    const p = (payload && typeof payload === 'object') ? { ...payload, pos: localPos } : { pos: localPos };
    safeSendTo(bw, channel, p);
}
let mouseMoveDispatchQueued = false;
let mouseMoveDispatchPick = false;
let mouseMoveDispatchDisplay = false;
let lastHoverCheckAt = 0;
let lastMouseMoveDispatchAt = 0;
const MIN_MOUSE_MOVE_INTERVAL = 8;
function scheduleMouseMoveDispatch(pick, display) {
    if (pick)
        mouseMoveDispatchPick = true;
    if (display)
        mouseMoveDispatchDisplay = true;
    if (mouseMoveDispatchQueued)
        return;
    mouseMoveDispatchQueued = true;
    const now = Date.now();
    const elapsed = now - lastMouseMoveDispatchAt;
    if (elapsed < MIN_MOUSE_MOVE_INTERVAL) {
        setTimeout(() => {
            mouseMoveDispatchQueued = false;
            lastMouseMoveDispatchAt = Date.now();
            const doPick = mouseMoveDispatchPick;
            const doDisplay = mouseMoveDispatchDisplay;
            mouseMoveDispatchPick = false;
            mouseMoveDispatchDisplay = false;
            if (doPick && (screenshotPickLock || recordPickLock))
                sendMouseToPickLock('mouse-move', undefined);
            if (doDisplay && mouseDotMoveVisible)
                sendMouseToDisplay('mouse-move', undefined);
        }, MIN_MOUSE_MOVE_INTERVAL - elapsed);
        return;
    }
    setImmediate(() => {
        mouseMoveDispatchQueued = false;
        lastMouseMoveDispatchAt = Date.now();
        const doPick = mouseMoveDispatchPick;
        const doDisplay = mouseMoveDispatchDisplay;
        mouseMoveDispatchPick = false;
        mouseMoveDispatchDisplay = false;
        if (doPick && (screenshotPickLock || recordPickLock))
            sendMouseToPickLock('mouse-move', undefined);
        if (doDisplay && mouseDotMoveVisible)
            sendMouseToDisplay('mouse-move', undefined);
    });
}
function sendMouseToPickLock(channel, payload) {
    const lock = screenshotPickLock || recordPickLock;
    if (!lock)
        return false;
    const bw = Array.from(wins.values()).find(x => x && !x.isDestroyed() && x.id === lock.bwId);
    if (!bw)
        return false;
    const idStr = String(lock.displayId || '').trim();
    const pt = screen.getCursorScreenPoint();
    const d = screen.getAllDisplays().find(x => String(x && x.id) === idStr) || screen.getPrimaryDisplay();
    const b = d && d.bounds ? d.bounds : { x: 0, y: 0, width: 1, height: 1 };
    const clamp = (v, min, max) => Math.max(min, Math.min(max, v));
    const localPos = {
        x: clamp(pt.x - b.x, 0, Math.max(0, b.width - 1)),
        y: clamp(pt.y - b.y, 0, Math.max(0, b.height - 1)),
    };
    if (channel === 'mouse-move') {
        safeSendTo(bw, channel, localPos);
        return true;
    }
    const p = (payload && typeof payload === 'object') ? { ...payload, pos: localPos } : { pos: localPos };
    safeSendTo(bw, channel, p);
    return true;
}
function getDisplayAtCursor() {
    const pt = screen.getCursorScreenPoint();
    const displays = screen.getAllDisplays();
    return displays.find(di => pt.x >= di.bounds.x && pt.x < di.bounds.x + di.bounds.width && pt.y >= di.bounds.y && pt.y < di.bounds.y + di.bounds.height) || null;
}
function getOverlayWindowAtCursor() {
    const d = getDisplayAtCursor();
    return d ? wins.get(d.id) : null;
}
function getPickLockWindow() {
    const lock = screenshotPickLock || recordPickLock;
    if (!lock)
        return null;
    return Array.from(wins.values()).find(x => x && !x.isDestroyed() && x.id === lock.bwId) || null;
}
function getKeyRouteId(event) {
    if (event && typeof event.rawcode === 'number')
        return `r${event.rawcode}`;
    if (event && typeof event.keycode === 'number')
        return `k${event.keycode}`;
    if (event && typeof event.keychar === 'number' && event.keychar > 0)
        return `c${event.keychar}`;
    return null;
}
function sendKeyEventToCursor(payload) {
    const d = getDisplayAtCursor();
    const bw = d ? wins.get(d.id) : null;
    if (!bw || bw.isDestroyed())
        return null;
    safeSendTo(bw, 'key-event', payload);
    return { bw, displayId: d.id };
}
function sendKeyEventToDisplayId(displayId, payload) {
    const bw = wins.get(displayId);
    if (!bw || bw.isDestroyed())
        return null;
    safeSendTo(bw, 'key-event', payload);
    return bw;
}
function routeKeyDownEvent(event) {
    const routed = sendKeyEventToCursor({ type: 'down', data: event });
    const id = getKeyRouteId(event);
    if (routed && id) {
        keyRouteById.set(id, routed.displayId);
        keyRouteTimestamps.set(id, Date.now());
    }
    return routed ? routed.bw : null;
}
function routeKeyUpEvent(event) {
    const payload = { type: 'up', data: event };
    const id = getKeyRouteId(event);
    if (id && keyRouteById.has(id)) {
        const displayId = keyRouteById.get(id);
        keyRouteById.delete(id);
        const bw = sendKeyEventToDisplayId(displayId, payload);
        if (bw)
            return bw;
    }
    const routed = sendKeyEventToCursor(payload);
    return routed ? routed.bw : null;
}
function triggerScreenshot(kind) {
    try {
        console.log(`[screenshot] trigger kind=${String(kind || 'save')}`);
    }
    catch { }
    const bw = getOverlayWindowAtCursor() || Array.from(wins.values())[0];
    if (!bw)
        return;
    safeSendTo(bw, 'screenshot-command', { kind: String(kind || 'save') });
}
function triggerRecord(action) {
    const act = String(action || '').trim();
    if (!act)
        return;
    try {
        console.log(`[record] trigger action=${act}`);
    }
    catch { }
    const lockBw = recordPickLock
        ? Array.from(wins.values()).find(x => x && !x.isDestroyed() && x.id === recordPickLock.bwId)
        : null;
    const bw = lockBw || getOverlayWindowAtCursor() || Array.from(wins.values())[0];
    if (!bw)
        return;
    safeSendTo(bw, 'record-command', { action: act });
}
function applyGlobalShortcutsFromSettings(s) {
    try {
        globalShortcut.unregisterAll();
    }
    catch { }
    const hotkeys = s && s.hotkeys && typeof s.hotkeys === 'object' ? s.hotkeys : {};
    const overlayKey = String(hotkeys.toggleOverlay || '').trim();
    const mouseButtonsKey = String(hotkeys.toggleMouseButtons || '').trim();
    const mouseDotKey = String(hotkeys.toggleMouseDot || '').trim();
    const openSettingsKey = String(hotkeys.openSettings || '').trim();
    const saveKey = String(hotkeys.screenshotSave || '').trim();
    const clipKey = String(hotkeys.screenshotClipboard || '').trim();
    const recordToggleKey = String(hotkeys.recordToggle || '').trim();
    const recordSaveGifKey = String(hotkeys.recordSaveGif || '').trim();
    const recordSaveMp4Key = String(hotkeys.recordSaveMp4 || '').trim();
    if (overlayKey) {
        try {
            globalShortcut.register(overlayKey, () => toggleOverlayVisibility());
            if (!globalShortcut.isRegistered(overlayKey))
                console.log(`[hotkey] register_failed ${overlayKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${overlayKey}`);
            }
            catch { }
        }
    }
    if (mouseButtonsKey) {
        try {
            globalShortcut.register(mouseButtonsKey, () => toggleMouseButtonVisibility());
            if (!globalShortcut.isRegistered(mouseButtonsKey))
                console.log(`[hotkey] register_failed ${mouseButtonsKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${mouseButtonsKey}`);
            }
            catch { }
        }
    }
    if (mouseDotKey) {
        try {
            globalShortcut.register(mouseDotKey, () => toggleMouseDotVisibility());
            if (!globalShortcut.isRegistered(mouseDotKey))
                console.log(`[hotkey] register_failed ${mouseDotKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${mouseDotKey}`);
            }
            catch { }
        }
    }
    if (openSettingsKey) {
        try {
            globalShortcut.register(openSettingsKey, () => openSettingsWindow());
            if (!globalShortcut.isRegistered(openSettingsKey))
                console.log(`[hotkey] register_failed ${openSettingsKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${openSettingsKey}`);
            }
            catch { }
        }
    }
    if (saveKey) {
        try {
            globalShortcut.register(saveKey, () => triggerScreenshot('save'));
            if (!globalShortcut.isRegistered(saveKey))
                console.log(`[hotkey] register_failed ${saveKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${saveKey}`);
            }
            catch { }
        }
    }
    if (clipKey) {
        try {
            globalShortcut.register(clipKey, () => triggerScreenshot('clipboard'));
            if (!globalShortcut.isRegistered(clipKey))
                console.log(`[hotkey] register_failed ${clipKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${clipKey}`);
            }
            catch { }
        }
    }
    if (recordToggleKey) {
        try {
            globalShortcut.register(recordToggleKey, () => triggerRecord('toggle'));
            if (!globalShortcut.isRegistered(recordToggleKey))
                console.log(`[hotkey] register_failed ${recordToggleKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${recordToggleKey}`);
            }
            catch { }
        }
    }
    if (recordSaveGifKey) {
        try {
            globalShortcut.register(recordSaveGifKey, () => triggerRecord('saveGif'));
            if (!globalShortcut.isRegistered(recordSaveGifKey))
                console.log(`[hotkey] register_failed ${recordSaveGifKey}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${recordSaveGifKey}`);
            }
            catch { }
        }
    }
    if (recordSaveMp4Key) {
        try {
            globalShortcut.register(recordSaveMp4Key, () => triggerRecord('saveMp4'));
            if (!globalShortcut.isRegistered(recordSaveMp4Key))
                console.log(`[hotkey] register_failed ${recordSaveMp4Key}`);
        }
        catch {
            try {
                console.log(`[hotkey] register_error ${recordSaveMp4Key}`);
            }
            catch { }
        }
    }
}
function registerGlobalShortcuts() {
    applyGlobalShortcutsFromSettings(settings);
}
app.whenReady().then(async () => {
    cleanupLegacyWindowsAutoLaunch();
    await loadSettings();
    createWindows();
    createTray();
    registerGlobalShortcuts();
    updateWindowVisibility();
    startCleanupInterval();
    ipcMain.handle('get-settings', () => settings);
    ipcMain.handle('set-settings', (_evt, next) => {
        if (!next || typeof next !== 'object' || Array.isArray(next))
            return settings;
        try {
            const keys = Object.keys(next);
            if (keys.length > 2000)
                return settings;
        }
        catch {
            return settings;
        }
        const saved = saveSettings(next);
        applyRuntimeFromSettings(saved);
        broadcastSettings();
        return saved;
    });
    ipcMain.handle('pick-ref-images', async (evt) => {
        const parent = BrowserWindow.fromWebContents(evt.sender);
        try {
            const res = await dialog.showOpenDialog(parent || undefined, {
                title: '选择参考图',
                properties: ['openFile', 'multiSelections'],
                filters: [{ name: '图片', extensions: ['png', 'jpg', 'jpeg', 'webp', 'gif', 'bmp'] }],
            });
            if (!res || res.canceled || !res.filePaths || !res.filePaths.length)
                return [];
            const extToMime = (ext0) => {
                const ext = String(ext0 || '').trim().toLowerCase();
                if (ext === '.png')
                    return 'image/png';
                if (ext === '.jpg' || ext === '.jpeg')
                    return 'image/jpeg';
                if (ext === '.webp')
                    return 'image/webp';
                if (ext === '.gif')
                    return 'image/gif';
                if (ext === '.bmp')
                    return 'image/bmp';
                return 'application/octet-stream';
            };
            const out = [];
            for (const fp0 of res.filePaths) {
                const fp = String(fp0 || '').trim();
                if (!fp)
                    continue;
                try {
                    const buf = await fs.promises.readFile(fp);
                    if (!buf || !buf.length || buf.length > MAX_REF_IMAGE_BYTES)
                        continue;
                    const mime = extToMime(path.extname(fp));
                    const dataUrl = `data:${mime};base64,${buf.toString('base64')}`;
                    out.push({ name: path.basename(fp), dataUrl });
                }
                catch { }
            }
            return out;
        }
        catch {
            return [];
        }
    });
    ipcMain.handle('reset-settings', () => {
        const s = resetSettings();
        applyRuntimeFromSettings(s);
        broadcastSettings();
        return s;
    });
    ipcMain.on('draw-capture', (evt, capture) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        rendererCaptureByBwId.set(bw.id, !!capture);
        if (capture)
            beginToolHold();
        if (screenshotPickLock && screenshotPickLock.bwId === bw.id && !capture)
            return;
        if (recordPickLock && recordPickLock.bwId === bw.id && recordPickLock.capture && !capture)
            return;
        bw.setIgnoreMouseEvents(!capture, { forward: true });
        if (!capture)
            endToolHold();
    });
    ipcMain.on('pinned-shots-updated', (evt, payload) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        const p = (payload && typeof payload === 'object') ? payload : {};
        const rects = normalizePinnedRects(p.rects);
        pinnedShotRectsByBwId.set(bw.id, rects);
        const other = refImageRectsByBwId.get(bw.id);
        const merged = rects.length ? rects.slice() : [];
        if (other && other.length)
            merged.push(...other);
        pinnedRectsByBwId.set(bw.id, merged);
        updateWindowVisibility();
    });
    ipcMain.on('pin-shot-create', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const pngBase64 = String(p.pngBase64 || '').trim();
        const rectLocal = normalizePinnedShotRect(p.rect);
        const displayId = String(p.displayId || '').trim();
        if (!pngBase64 || !rectLocal)
            return;
        if (!isBase64ish(pngBase64) || estimateBase64Bytes(pngBase64) > MAX_PNG_BYTES)
            return;
        const d = screen.getAllDisplays().find(x => String(x && x.id) === displayId) || screen.getPrimaryDisplay();
        const b = d && d.bounds ? d.bounds : { x: 0, y: 0 };
        const id = `ps_${Date.now()}_${Math.random().toString(16).slice(2)}`;
        const rect = { x: rectLocal.x + b.x, y: rectLocal.y + b.y, w: rectLocal.w, h: rectLocal.h };
        pinnedShotsById.set(id, { id, rect, pngBase64, sentTo: new Set() });
        pinnedShotOrder = pinnedShotOrder.concat([id]);
        broadcastPinnedShots();
    });
    ipcMain.on('pin-shot-remove', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        if (!id)
            return;
        if (!pinnedShotsById.has(id))
            return;
        pinnedShotsById.delete(id);
        pinnedShotOrder = pinnedShotOrder.filter(x => x !== id);
        if (pinnedDrag && pinnedDrag.id === id)
            pinnedDrag = null;
        broadcastPinnedShots();
    });
    ipcMain.on('pin-shot-pop-last', () => {
        const id = pinnedShotOrder.length ? String(pinnedShotOrder[pinnedShotOrder.length - 1] || '').trim() : '';
        if (!id)
            return;
        if (!pinnedShotsById.has(id)) {
            pinnedShotOrder = pinnedShotOrder.slice(0, -1);
            return;
        }
        pinnedShotsById.delete(id);
        pinnedShotOrder = pinnedShotOrder.slice(0, -1);
        if (pinnedDrag && pinnedDrag.id === id)
            pinnedDrag = null;
        broadcastPinnedShots();
    });
    ipcMain.on('pin-shot-drag-start', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        const ox = Math.round(Number(p.offsetX) || 0);
        const oy = Math.round(Number(p.offsetY) || 0);
        if (!id || !pinnedShotsById.has(id))
            return;
        pinnedDrag = { id, offsetX: ox, offsetY: oy };
    });
    ipcMain.on('pin-shot-scale', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        const deltaY = Number(p.deltaY) || 0;
        if (!id || !pinnedShotsById.has(id))
            return;
        if (!Number.isFinite(deltaY) || deltaY === 0)
            return;
        const s = pinnedShotsById.get(id);
        const r = s && s.rect ? s.rect : null;
        if (!r)
            return;
        const vb = getVirtualBounds();
        const factor = deltaY > 0 ? 0.92 : 1.08;
        const minSize = 24;
        const cx = (Number(r.x) || 0) + (Number(r.w) || 1) / 2;
        const cy = (Number(r.y) || 0) + (Number(r.h) || 1) / 2;
        const nextW0 = Math.round((Number(r.w) || 1) * factor);
        const nextH0 = Math.round((Number(r.h) || 1) * factor);
        const maxW = Math.max(minSize, Math.round(Number(vb.width) || minSize));
        const maxH = Math.max(minSize, Math.round(Number(vb.height) || minSize));
        const nextW = Math.max(minSize, Math.min(maxW, nextW0));
        const nextH = Math.max(minSize, Math.min(maxH, nextH0));
        const nx = Math.round(cx - nextW / 2);
        const ny = Math.round(cy - nextH / 2);
        r.w = nextW;
        r.h = nextH;
        r.x = Math.max(vb.x, Math.min(vb.x + vb.width - r.w, nx));
        r.y = Math.max(vb.y, Math.min(vb.y + vb.height - r.h, ny));
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-create', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const dataUrl = normalizeDataUrl(p.dataUrl, MAX_REF_IMAGE_BYTES);
        const rectLocal = normalizePinnedShotRect(p.rect);
        const displayId = String(p.displayId || '').trim();
        let id = String(p.id || '').trim();
        if (!id)
            id = `ref_${Date.now()}_${Math.random().toString(16).slice(2)}`;
        if (!dataUrl || !rectLocal)
            return;
        const d = screen.getAllDisplays().find(x => String(x && x.id) === displayId) || screen.getPrimaryDisplay();
        const b = d && d.bounds ? d.bounds : { x: 0, y: 0 };
        const rect = { x: rectLocal.x + b.x, y: rectLocal.y + b.y, w: rectLocal.w, h: rectLocal.h };
        const rotation = Number(p.rotation) || 0;
        refImagesById.set(id, { id, rect, dataUrl, originalDataUrl: dataUrl, rotation, sentTo: new Set() });
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        broadcastPinnedShots();
    });
    ipcMain.on('ref-image-remove', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        if (!id || !refImagesById.has(id))
            return;
        refImagesById.delete(id);
        refImageOrder = refImageOrder.filter(x => x !== id);
        if (refDrag && refDrag.id === id)
            refDrag = null;
        broadcastPinnedShots();
    });
    ipcMain.on('ref-image-drag-start', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        const ox = Math.round(Number(p.offsetX) || 0);
        const oy = Math.round(Number(p.offsetY) || 0);
        if (!id || !refImagesById.has(id))
            return;
        refDrag = { id, offsetX: ox, offsetY: oy };
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('settings-drag-start', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const ox = Math.round(Number(p.offsetX) || 0);
        const oy = Math.round(Number(p.offsetY) || 0);
        if (!settingsWin || settingsWin.isDestroyed())
            return;
        const bounds = settingsWin.getBounds();
        settingsDrag = { offsetX: ox, offsetY: oy, width: bounds.width, height: bounds.height };
    });
    ipcMain.on('ref-image-scale', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        const deltaY = Number(p.deltaY) || 0;
        const rx0 = Number(p.rx);
        const ry0 = Number(p.ry);
        const rx = Number.isFinite(rx0) ? Math.max(0, Math.min(1, rx0)) : 0.5;
        const ry = Number.isFinite(ry0) ? Math.max(0, Math.min(1, ry0)) : 0.5;
        if (!id || !refImagesById.has(id))
            return;
        if (!Number.isFinite(deltaY) || deltaY === 0)
            return;
        const s = refImagesById.get(id);
        const r = s && s.rect ? s.rect : null;
        if (!r)
            return;
        const vb = getVirtualBounds();
        const factor = deltaY > 0 ? 0.92 : 1.08;
        const minSize = 24;
        const ax = (Number(r.x) || 0) + (Number(r.w) || 1) * rx;
        const ay = (Number(r.y) || 0) + (Number(r.h) || 1) * ry;
        const nextW0 = Math.round((Number(r.w) || 1) * factor);
        const nextH0 = Math.round((Number(r.h) || 1) * factor);
        const maxW = Math.max(minSize, Math.round(Number(vb.width) || minSize));
        const maxH = Math.max(minSize, Math.round(Number(vb.height) || minSize));
        const nextW = Math.max(minSize, Math.min(maxW, nextW0));
        const nextH = Math.max(minSize, Math.min(maxH, nextH0));
        r.w = nextW;
        r.h = nextH;
        r.x = Math.max(vb.x, Math.min(vb.x + vb.width - r.w, Math.round(ax - rx * r.w)));
        r.y = Math.max(vb.y, Math.min(vb.y + vb.height - r.h, Math.round(ay - ry * r.h)));
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-rotate', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        const delta = Number(p.delta) || 0;
        if (!id || !refImagesById.has(id))
            return;
        if (!Number.isFinite(delta) || delta === 0)
            return;
        const s = refImagesById.get(id);
        if (!s)
            return;
        const currentRotation = Number(s.rotation) || 0;
        s.rotation = currentRotation + delta;
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-set-rect', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        if (!id || !refImagesById.has(id))
            return;
        const r = p.rect && typeof p.rect === 'object' ? p.rect : null;
        if (!r)
            return;
        const s = refImagesById.get(id);
        if (!s)
            return;
        const vb = getVirtualBounds();
        const newX = Math.max(vb.x, Math.min(vb.x + vb.width - 24, Math.round(Number(r.x) || 0)));
        const newY = Math.max(vb.y, Math.min(vb.y + vb.height - 24, Math.round(Number(r.y) || 0)));
        const newW = Math.max(24, Math.round(Number(r.w) || 24));
        const newH = Math.max(24, Math.round(Number(r.h) || 24));
        s.rect = { x: newX, y: newY, w: newW, h: newH };
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-set-props', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        if (!id || !refImagesById.has(id))
            return;
        const s = refImagesById.get(id);
        if (!s)
            return;
        if (p.flipH !== undefined)
            s.flipH = !!p.flipH;
        if (p.flipV !== undefined)
            s.flipV = !!p.flipV;
        if (p.grayscale !== undefined)
            s.grayscale = !!p.grayscale;
        if (p.invert !== undefined)
            s.invert = !!p.invert;
        if (p.locked !== undefined)
            s.locked = !!p.locked;
        if (p.showGrid !== undefined)
            s.showGrid = !!p.showGrid;
        if (typeof p.brightness === 'number')
            s.brightness = Math.max(0, Math.min(3, p.brightness));
        if (typeof p.contrast === 'number')
            s.contrast = Math.max(0, Math.min(3, p.contrast));
        if (typeof p.opacity === 'number')
            s.opacity = Math.max(0.05, Math.min(1, p.opacity));
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-set-data-url', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        if (!id || !refImagesById.has(id))
            return;
        const s = refImagesById.get(id);
        if (!s)
            return;
        const dataUrl = normalizeDataUrl(p.dataUrl, MAX_REF_IMAGE_BYTES);
        if (!dataUrl)
            return;
        s.dataUrl = dataUrl;
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-reset-props', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const id = String(p.id || '').trim();
        if (!id || !refImagesById.has(id))
            return;
        const s = refImagesById.get(id);
        if (!s)
            return;
        s.flipH = false;
        s.flipV = false;
        s.grayscale = false;
        s.invert = false;
        s.locked = false;
        s.showGrid = false;
        s.brightness = 1;
        s.contrast = 1;
        s.opacity = 1;
        s.rotation = 0;
        if (s.originalDataUrl) {
            s.dataUrl = s.originalDataUrl;
        }
        refImageOrder = refImageOrder.filter(x => x !== id).concat([id]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('ref-image-duplicate', (_evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const srcId = String(p.srcId || '').trim();
        const newId = String(p.newId || '').trim();
        if (!srcId || !newId || !refImagesById.has(srcId))
            return;
        const src = refImagesById.get(srcId);
        if (!src)
            return;
        const offset = 24;
        const newRect = {
            x: (src.rect?.x || 0) + offset,
            y: (src.rect?.y || 0) + offset,
            w: src.rect?.w || 100,
            h: src.rect?.h || 100
        };
        refImagesById.set(newId, {
            id: newId,
            rect: newRect,
            dataUrl: src.dataUrl,
            originalDataUrl: src.originalDataUrl || src.dataUrl,
            rotation: src.rotation || 0,
            flipH: src.flipH || false,
            flipV: src.flipV || false,
            grayscale: src.grayscale || false,
            invert: src.invert || false,
            locked: false,
            showGrid: src.showGrid || false,
            brightness: src.brightness || 1,
            contrast: src.contrast || 1,
            opacity: src.opacity || 1,
            sentTo: new Set()
        });
        refImageOrder = refImageOrder.filter(x => x !== newId).concat([newId]);
        queueBroadcastPinnedShots();
    });
    ipcMain.on('overlay-focusable', (evt, focusable) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        try {
            bw.setFocusable(!!focusable);
        }
        catch { }
    });
    ipcMain.on('overlay-focus', (evt) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        try {
            bw.setFocusable(true);
        }
        catch { }
        try {
            bw.show();
        }
        catch { }
        try {
            bw.focus();
        }
        catch { }
    });
    ipcMain.on('screenshot-pick-mode', (evt, payload) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        const p = (payload && typeof payload === 'object') ? payload : {};
        const active = !!p.active;
        const displayId = p.displayId;
        if (active)
            beginToolHold();
        setScreenshotPickLock(bw, active, displayId);
        updateWindowVisibility();
        if (!active)
            endToolHold();
    });
    ipcMain.on('record-pick-mode', (evt, payload) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        const p = (payload && typeof payload === 'object') ? payload : {};
        const active = !!p.active;
        const displayId = p.displayId;
        const capture = !!p.capture;
        if (active)
            beginToolHold();
        setRecordPickLock(bw, active, displayId, capture);
        updateWindowVisibility();
        if (!active)
            endToolHold();
    });
    ipcMain.on('interactive-rects-updated', (evt, payload) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        const p = (payload && typeof payload === 'object') ? payload : {};
        const rects = normalizePinnedRects(p.rects);
        interactiveRectsByBwId.set(bw.id, rects);
        updateWindowVisibility();
    });
    ipcMain.on('close-settings', (evt) => {
        const bw = BrowserWindow.fromWebContents(evt.sender);
        if (!bw)
            return;
        if (!settingsWin || settingsWin.isDestroyed())
            return;
        if (bw.id !== settingsWin.id)
            return;
        try {
            settingsWin.close();
        }
        catch { }
    });
    ipcMain.handle('screenshot-save', async (evt, payload) => {
        try {
            console.log('[screenshot] save invoked');
        }
        catch { }
        const { pngBase64, suggestedName } = (payload && typeof payload === 'object') ? payload : {};
        const buf = parseBase64ToBuffer(pngBase64, MAX_PNG_BYTES);
        if (!buf)
            return { canceled: true, error: 'invalid_or_too_large' };
        const parent = BrowserWindow.fromWebContents(evt.sender);
        const now = new Date();
        const pad2 = (n) => String(n).padStart(2, '0');
        const ts = `${now.getFullYear()}${pad2(now.getMonth() + 1)}${pad2(now.getDate())}_${pad2(now.getHours())}${pad2(now.getMinutes())}${pad2(now.getSeconds())}`;
        const baseName = String(suggestedName || '').trim() || `TT_Screenshot_${ts}.png`;
        let picturesDir = null;
        try {
            picturesDir = app.getPath('pictures');
        }
        catch { }
        const defaultPath = picturesDir ? path.join(picturesDir, baseName) : baseName;
        const res = await dialog.showSaveDialog(parent || undefined, {
            title: '保存截图',
            defaultPath,
            filters: [{ name: 'PNG 图片', extensions: ['png'] }],
        });
        if (!res || res.canceled || !res.filePath)
            return { canceled: true };
        const filePath = String(res.filePath);
        try {
            await fs.promises.mkdir(path.dirname(filePath), { recursive: true });
        }
        catch { }
        try {
            await fs.promises.writeFile(filePath, buf);
        }
        catch { }
        return { canceled: false, filePath };
    });
    ipcMain.handle('screenshot-clipboard', async (_evt, payload) => {
        try {
            console.log('[screenshot] clipboard invoked');
        }
        catch { }
        const { pngBase64 } = (payload && typeof payload === 'object') ? payload : {};
        const buf = parseBase64ToBuffer(pngBase64, MAX_PNG_BYTES);
        if (!buf)
            return { ok: false, error: 'invalid_or_too_large' };
        try {
            const img = nativeImage.createFromBuffer(buf);
            if (!img || img.isEmpty())
                return { ok: false, error: 'nativeImage_failed' };
            clipboard.writeImage(img);
            return { ok: true };
        }
        catch {
            return { ok: false, error: 'clipboard_write_failed' };
        }
    });
    ipcMain.handle('copy-text', async (_evt, payload) => {
        const { text } = (payload && typeof payload === 'object') ? payload : {};
        const s = String(text || '');
        try {
            clipboard.writeText(s);
            return { ok: true };
        }
        catch {
            return { ok: false, error: 'clipboard_write_failed' };
        }
    });
    ipcMain.handle('desktop-source', async (_evt, payload) => {
        const { displayId } = (payload && typeof payload === 'object') ? payload : {};
        const idStr = String(displayId || '').trim();
        if (!desktopCapturer || typeof desktopCapturer.getSources !== 'function')
            return { sourceId: '', error: 'desktopCapturer_unavailable' };
        try {
            const sources = await desktopCapturer.getSources({ types: ['screen'], thumbnailSize: { width: 16, height: 16 } });
            if (!sources || !sources.length)
                return { sourceId: '', error: 'no_sources' };
            const src = sources.find(s => String(s && s.display_id ? s.display_id : '') === idStr) || sources[0];
            const sourceId = src && src.id ? String(src.id) : '';
            return { sourceId, error: sourceId ? '' : 'no_source_id' };
        }
        catch {
            return { sourceId: '', error: 'getSources_failed' };
        }
    });
    ipcMain.handle('recording-save', async (evt, payload) => {
        const p = (payload && typeof payload === 'object') ? payload : {};
        const ext = String(p.ext || '').trim().toLowerCase();
        const suggestedName = String(p.suggestedName || '').trim();
        let buf = null;
        const data = p.data;
        try {
            if (data && typeof data === 'object' && Buffer.isBuffer(data))
                buf = data;
            else if (data && typeof data === 'object' && data.type === 'Buffer' && Array.isArray(data.data))
                buf = Buffer.from(data.data);
            else if (data instanceof ArrayBuffer)
                buf = Buffer.from(new Uint8Array(data));
            else if (ArrayBuffer.isView(data))
                buf = Buffer.from(new Uint8Array(data.buffer, data.byteOffset, data.byteLength));
            else if (typeof data === 'string')
                buf = parseBase64ToBuffer(data, MAX_RECORD_BYTES);
        }
        catch {
            buf = null;
        }
        if (!buf || !buf.length || buf.length > MAX_RECORD_BYTES)
            return { canceled: true, error: 'invalid_or_too_large' };
        const parent = BrowserWindow.fromWebContents(evt.sender);
        const now = new Date();
        const pad2 = (n) => String(n).padStart(2, '0');
        const ts = `${now.getFullYear()}${pad2(now.getMonth() + 1)}${pad2(now.getDate())}_${pad2(now.getHours())}${pad2(now.getMinutes())}${pad2(now.getSeconds())}`;
        const safeExt = (ext === 'gif' || ext === 'mp4') ? ext : 'webm';
        const baseName = suggestedName || `TT_Record_${ts}.${safeExt}`;
        let defaultDir = null;
        try {
            defaultDir = app.getPath(safeExt === 'mp4' ? 'videos' : 'pictures');
        }
        catch { }
        const defaultPath = defaultDir ? path.join(defaultDir, baseName) : baseName;
        const res = await dialog.showSaveDialog(parent || undefined, {
            title: '保存录屏',
            defaultPath,
            filters: safeExt === 'mp4'
                ? [{ name: 'MP4 视频', extensions: ['mp4'] }]
                : safeExt === 'gif'
                    ? [{ name: 'GIF 动图', extensions: ['gif'] }]
                    : [{ name: 'WebM 视频', extensions: ['webm'] }],
        });
        if (!res || res.canceled || !res.filePath)
            return { canceled: true };
        const filePath = String(res.filePath);
        try {
            await fs.promises.mkdir(path.dirname(filePath), { recursive: true });
        }
        catch { }
        try {
            await fs.promises.writeFile(filePath, buf);
        }
        catch { }
        return { canceled: false, filePath };
    });
    ipcMain.handle('recording-export', async (evt, payload) => {
        const parseToBuffer = (data) => {
            let buf = null;
            try {
                if (data && typeof data === 'object' && Buffer.isBuffer(data))
                    buf = data;
                else if (data && typeof data === 'object' && data.type === 'Buffer' && Array.isArray(data.data))
                    buf = Buffer.from(data.data);
                else if (data instanceof ArrayBuffer)
                    buf = Buffer.from(new Uint8Array(data));
                else if (ArrayBuffer.isView(data))
                    buf = Buffer.from(new Uint8Array(data.buffer, data.byteOffset, data.byteLength));
                else if (typeof data === 'string')
                    buf = parseBase64ToBuffer(data, MAX_RECORD_BYTES);
            }
            catch {
                buf = null;
            }
            if (!buf || !buf.length || buf.length > MAX_RECORD_BYTES)
                return null;
            return buf;
        };
        const resolveFfmpegPath = () => {
            const cands = [];
            const isWin = process.platform === 'win32';
            const binName = isWin ? 'ffmpeg.exe' : 'ffmpeg';
            try {
                cands.push(path.join(__dirname, 'bin', binName));
            }
            catch { }
            try {
                const appPath = app.getAppPath();
                if (appPath)
                    cands.push(path.join(path.dirname(appPath), 'bin', binName));
            }
            catch { }
            try {
                if (process.resourcesPath)
                    cands.push(path.join(process.resourcesPath, 'bin', binName));
            }
            catch { }
            for (const p of cands) {
                try {
                    if (p && fs.existsSync(p))
                        return p;
                }
                catch { }
            }
            return '';
        };
        const runFfmpeg = (ffmpegPath, args) => new Promise((resolve) => {
            try {
                const cp = spawn(ffmpegPath, args, { windowsHide: true });
                let stderr = '';
                let stdout = '';
                try {
                    cp.stdout && cp.stdout.on('data', (d) => { try {
                        stdout += String(d || '');
                    }
                    catch { } });
                }
                catch { }
                try {
                    cp.stderr && cp.stderr.on('data', (d) => { try {
                        stderr += String(d || '');
                    }
                    catch { } });
                }
                catch { }
                cp.on('close', (code) => resolve({ code: typeof code === 'number' ? code : 1, stdout, stderr }));
                cp.on('error', () => resolve({ code: 1, stdout, stderr }));
            }
            catch {
                resolve({ code: 1, stdout: '', stderr: '' });
            }
        });
        const p = (payload && typeof payload === 'object') ? payload : {};
        const fmt = String(p.fmt || '').trim().toLowerCase();
        const suggestedName = String(p.suggestedName || '').trim();
        const safeFmt = (fmt === 'mp4' || fmt === 'gif') ? fmt : '';
        if (!safeFmt)
            return { canceled: true, error: 'invalid_format' };
        const buf = parseToBuffer(p.data);
        if (!buf)
            return { canceled: true, error: 'empty_data' };
        const crop0 = (p.crop && typeof p.crop === 'object') ? p.crop : {};
        const x = Math.max(0, Math.round(Number(crop0.x) || 0));
        const y = Math.max(0, Math.round(Number(crop0.y) || 0));
        const w = Math.max(2, Math.min(16384, Math.round(Number(crop0.w) || 0)));
        const h = Math.max(2, Math.min(16384, Math.round(Number(crop0.h) || 0)));
        const ffmpegPath = resolveFfmpegPath();
        if (!ffmpegPath)
            return { canceled: true, error: 'ffmpeg_missing' };
        const parent = BrowserWindow.fromWebContents(evt.sender);
        const now = new Date();
        const pad2 = (n) => String(n).padStart(2, '0');
        const ts = `${now.getFullYear()}${pad2(now.getMonth() + 1)}${pad2(now.getDate())}_${pad2(now.getHours())}${pad2(now.getMinutes())}${pad2(now.getSeconds())}`;
        const baseName = suggestedName || `TT_Record_${ts}.${safeFmt}`;
        let defaultDir = null;
        try {
            defaultDir = app.getPath(safeFmt === 'mp4' ? 'videos' : 'pictures');
        }
        catch { }
        const defaultPath = defaultDir ? path.join(defaultDir, baseName) : baseName;
        const res = await dialog.showSaveDialog(parent || undefined, {
            title: safeFmt === 'mp4' ? '导出 MP4' : '导出 GIF',
            defaultPath,
            filters: safeFmt === 'mp4'
                ? [{ name: 'MP4 视频', extensions: ['mp4'] }]
                : [{ name: 'GIF 动图', extensions: ['gif'] }],
        });
        if (!res || res.canceled || !res.filePath)
            return { canceled: true };
        const outPath = String(res.filePath);
        let tmpDir = null;
        try {
            tmpDir = app.getPath('temp');
        }
        catch { }
        if (!tmpDir)
            tmpDir = __dirname;
        const tag = `${Date.now()}_${Math.random().toString(16).slice(2)}`;
        const inPath = path.join(tmpDir, `tt_record_in_${tag}.webm`);
        const tmpOut = path.join(tmpDir, `tt_record_out_${tag}.${safeFmt}`);
        try {
            await fs.promises.writeFile(inPath, buf);
        }
        catch {
            return { canceled: true, error: 'write_input_failed' };
        }
        const baseArgs = ['-y', '-hide_banner', '-loglevel', 'error', '-i', inPath];
        let result = { code: 1, stdout: '', stderr: '' };
        if (safeFmt === 'mp4') {
            const vf = `crop=${w}:${h}:${x}:${y}`;
            result = await runFfmpeg(ffmpegPath, [...baseArgs, '-vf', vf, '-c:v', 'libx264', '-pix_fmt', 'yuv420p', tmpOut]);
            if (result.code !== 0)
                result = await runFfmpeg(ffmpegPath, [...baseArgs, '-vf', vf, '-c:v', 'mpeg4', tmpOut]);
        }
        else {
            const fps = Math.max(1, Math.min(60, Math.round(Number(p.fps) || 12)));
            const scale0 = (p.scale && typeof p.scale === 'object') ? p.scale : {};
            const sw = Math.max(2, Math.round(Number(scale0.w) || w));
            const sh = Math.max(2, Math.round(Number(scale0.h) || h));
            const filter = `crop=${w}:${h}:${x}:${y},fps=${fps},scale=${sw}:${sh}:flags=lanczos,split[s0][s1];[s0]palettegen[p];[s1][p]paletteuse`;
            result = await runFfmpeg(ffmpegPath, [...baseArgs, '-filter_complex', filter, '-loop', '0', tmpOut]);
        }
        try {
            await fs.promises.unlink(inPath);
        }
        catch { }
        try {
            const st = await fs.promises.stat(tmpOut);
            if (!st || !st.size || result.code !== 0)
                throw new Error('ffmpeg_failed');
            try {
                await fs.promises.mkdir(path.dirname(outPath), { recursive: true });
            }
            catch { }
            try {
                await fs.promises.copyFile(tmpOut, outPath);
            }
            catch {
                throw new Error('write_output_failed');
            }
            try {
                await fs.promises.unlink(tmpOut);
            }
            catch { }
            return { canceled: false, filePath: outPath };
        }
        catch {
            try {
                await fs.promises.unlink(tmpOut);
            }
            catch { }
            return { canceled: true, error: 'ffmpeg_failed', details: String(result && result.stderr ? result.stderr : '') };
        }
    });
    ipcMain.handle('screenshot-capture', async (_evt, payload) => {
        const { displayId } = (payload && typeof payload === 'object') ? payload : {};
        try {
            return await captureScreenInMain(displayId);
        }
        catch {
            return { pngBase64: '', error: 'getSources_failed' };
        }
    });
    ipcMain.handle('capture-all-displays', async () => {
        try {
            return await captureAllDisplays();
        }
        catch {
            return { displays: [], error: 'capture_failed' };
        }
    });
    uIOhook.on('keydown', (event) => {
        const routedBw = overlayVisible ? routeKeyDownEvent(event) : null;
        const pickBw = getPickLockWindow();
        if (pickBw && (!routedBw || pickBw.id !== routedBw.id))
            safeSendTo(pickBw, 'key-event', { type: 'down', data: event });
        if (debugKeysCount < 20) {
            debugKeysCount++;
            try {
                const { keycode, rawcode, keychar, mask } = event || {};
                console.log(`[key-debug] down kc=${keycode} rc=${rawcode} ch=${keychar} mask=${mask}`);
            }
            catch { }
        }
    });
    uIOhook.on('keyup', (event) => {
        const routedBw = overlayVisible ? routeKeyUpEvent(event) : null;
        const pickBw = getPickLockWindow();
        if (pickBw && (!routedBw || pickBw.id !== routedBw.id))
            safeSendTo(pickBw, 'key-event', { type: 'up', data: event });
        if (debugKeysCount < 20) {
            try {
                const { keycode, rawcode, keychar, mask } = event || {};
                console.log(`[key-debug]  up  kc=${keycode} rc=${rawcode} ch=${keychar} mask=${mask}`);
            }
            catch { }
        }
    });
    uIOhook.on('mousedown', (event) => {
        if (screenshotPickLock || recordPickLock) {
            sendMouseToPickLock('mouse-event', { type: 'down', data: event, ts: Date.now(), bubble: false });
            return;
        }
        if (mouseButtonVisible || mouseDotClickVisible)
            sendMouseToDisplay('mouse-event', { type: 'down', data: event, ts: Date.now(), bubble: !!mouseButtonVisible });
    });
    uIOhook.on('mouseup', (event) => {
        let changed = false;
        if (pinnedDrag) {
            pinnedDrag = null;
            changed = true;
        }
        if (refDrag) {
            refDrag = null;
            changed = true;
        }
        if (settingsDrag) {
            settingsDrag = null;
        }
        if (changed)
            queueBroadcastPinnedShots();
        if (screenshotPickLock || recordPickLock) {
            sendMouseToPickLock('mouse-event', { type: 'up', data: event, ts: Date.now(), bubble: false });
            return;
        }
        if (mouseButtonVisible || mouseDotClickVisible)
            sendMouseToDisplay('mouse-event', { type: 'up', data: event, ts: Date.now(), bubble: !!mouseButtonVisible });
    });
    uIOhook.on('mousemove', () => {
        if (settingsDrag && settingsWin && !settingsWin.isDestroyed()) {
            try {
                const pt = screen.getCursorScreenPoint();
                const nx = Math.round(pt.x - settingsDrag.offsetX);
                const ny = Math.round(pt.y - settingsDrag.offsetY);
                settingsWin.setBounds({ x: nx, y: ny, width: settingsDrag.width, height: settingsDrag.height });
            }
            catch { }
        }
        if (pinnedDrag && pinnedShotsById.has(pinnedDrag.id)) {
            try {
                const pt = screen.getCursorScreenPoint();
                const s = pinnedShotsById.get(pinnedDrag.id);
                const vb = getVirtualBounds();
                const nx = Math.max(vb.x, Math.min(vb.x + vb.width - s.rect.w, Math.round(pt.x - pinnedDrag.offsetX)));
                const ny = Math.max(vb.y, Math.min(vb.y + vb.height - s.rect.h, Math.round(pt.y - pinnedDrag.offsetY)));
                s.rect.x = nx;
                s.rect.y = ny;
                queueBroadcastPinnedShots();
            }
            catch { }
        }
        if (refDrag && refImagesById.has(refDrag.id)) {
            try {
                const pt = screen.getCursorScreenPoint();
                const s = refImagesById.get(refDrag.id);
                const vb = getVirtualBounds();
                const nx = Math.max(vb.x, Math.min(vb.x + vb.width - s.rect.w, Math.round(pt.x - refDrag.offsetX)));
                const ny = Math.max(vb.y, Math.min(vb.y + vb.height - s.rect.h, Math.round(pt.y - refDrag.offsetY)));
                s.rect.x = nx;
                s.rect.y = ny;
                queueBroadcastPinnedShots();
            }
            catch { }
        }
        if (screenshotPickLock) {
            scheduleMouseMoveDispatch(true, false);
            return;
        }
        if (recordPickLock)
            scheduleMouseMoveDispatch(true, false);
        try {
            const now = Date.now();
            if (now - lastHoverCheckAt < 16) {
                if (mouseDotMoveVisible)
                    scheduleMouseMoveDispatch(false, true);
                return;
            }
            lastHoverCheckAt = now;
            const d = getDisplayAtCursor();
            const bw = d ? wins.get(d.id) : null;
            if (bw && !bw.isDestroyed() && !rendererCaptureByBwId.get(bw.id)) {
                const pinnedRects = pinnedRectsByBwId.get(bw.id);
                const interactiveRects = interactiveRectsByBwId.get(bw.id);
                const pinnedForced = !!(pinnedRects && pinnedRects.length);
                const interactiveForced = !!(interactiveRects && interactiveRects.length);
                if (pinnedForced || interactiveForced) {
                    const pt = screen.getCursorScreenPoint();
                    const localPos = { x: pt.x - d.bounds.x, y: pt.y - d.bounds.y };
                    const insidePinned = pinnedForced ? pinnedRects.some(r => pointInRect(localPos, r)) : false;
                    const insideInteractive = interactiveForced ? interactiveRects.some(r => pointInRect(localPos, r)) : false;
                    const prevPinned = pinnedHoverByBwId.get(bw.id) || false;
                    const prevInteractive = interactiveHoverByBwId.get(bw.id) || false;
                    if (insidePinned !== prevPinned)
                        pinnedHoverByBwId.set(bw.id, insidePinned);
                    if (insideInteractive !== prevInteractive)
                        interactiveHoverByBwId.set(bw.id, insideInteractive);
                    const insideAny = insidePinned || insideInteractive;
                    const prevAny = prevPinned || prevInteractive;
                    if (insideAny !== prevAny) {
                        try {
                            if (insideAny)
                                bw.setIgnoreMouseEvents(false);
                            else
                                bw.setIgnoreMouseEvents(true, { forward: true });
                        }
                        catch { }
                    }
                }
                else {
                    pinnedHoverByBwId.set(bw.id, false);
                    interactiveHoverByBwId.set(bw.id, false);
                }
            }
        }
        catch { }
        if (mouseDotMoveVisible)
            scheduleMouseMoveDispatch(false, true);
    });
    uIOhook.start();
    screen.on('display-added', () => { createWindows(); broadcastPinnedShots(); });
    screen.on('display-removed', () => { createWindows(); broadcastPinnedShots(); });
    screen.on('display-metrics-changed', () => { createWindows(); broadcastPinnedShots(); });
});
app.on('will-quit', async () => {
    globalShortcut.unregisterAll();
    stopCleanupInterval();
    await flushSettingsWrite();
});
app.on('before-quit', () => { try {
    uIOhook.stop();
}
catch (err) {
    console.error('[main] Failed to stop uIOhook:', err);
} });
//# sourceMappingURL=main.js.map